(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 7.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    390939,       8778]
NotebookOptionsPosition[    363774,       7960]
NotebookOutlinePosition[    365241,       8007]
CellTagsIndexPosition[    364896,       7996]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["\<\

iLCGT Type-A VIS Rigid-Body Model
Double Pendulum Part (IM+IRM+TM+RM)\
\>", "Subtitle",
 CellChangeTimes->{{3.5109088633079133`*^9, 3.510908886523241*^9}, {
   3.5109089295327015`*^9, 3.5109089606324797`*^9}, {3.510998012935999*^9, 
   3.510998040953648*^9}, {3.511147199138874*^9, 3.511147201837679*^9}, 
   3.5111528655524273`*^9, {3.5112140422561007`*^9, 3.511214046727357*^9}, {
   3.511333214700381*^9, 3.5113332166191845`*^9}, {3.5113333850886836`*^9, 
   3.511333385931085*^9}, {3.512773766799003*^9, 3.51277377079261*^9}, {
   3.512775502577655*^9, 3.512775507866064*^9}}],

Cell[CellGroupData[{

Cell["Discription", "Section",
 CellChangeTimes->{{3.5054543012647257`*^9, 3.505454323450515*^9}, {
   3.510907714373198*^9, 3.5109077167963367`*^9}, 3.510908178125723*^9}],

Cell[CellGroupData[{

Cell["About This Model", "Subsection",
 CellChangeTimes->{{3.510394061790189*^9, 3.5103940822418246`*^9}, {
  3.510907707872826*^9, 3.5109077115670376`*^9}}],

Cell["\<\
A 3-D rigid-body model of the double pendulum part of type-A vibration \
isolation system for iLCGT.
* GAS filters work as ideal vertical springs (no saturation at high \
frequencies)
\
\>", "Text",
 CellChangeTimes->CompressedData["
1:eJxTTMoPSmViYGAQBmIQHaPdsU5e7pVjb7XsZhDtMutgTZ7SK0eOy9lNILpO
qOzYYSD9KWXPfRB93X/i0qNAesenf+tAdNLPt1liyq8cA3K5s0H0pwfZ0yrU
gfLrQ2aCaK74r51dQLpm+vkuEK3keufGbyDtFND+AESvK+X4AKIt5nhy/gHS
W/i/ZbVpvHJsts/NA9FT9i+NagfSf7btAtPrzpXozQfSFjP1DUC0BMuJtk1A
Ou1062QQfUbbaN4nIH2qZPcCEP1uq/cWEL0pdMp2EP3F9eQfac1Xjpc+l/4H
0ftbrm/7q/3K8e5Ezn0g2mTNqkMgWjr++UMQnXF6zyRRnVeOtt+OTAHRAMI2
n5w=
  "]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Version History", "Subsection",
 CellChangeTimes->{{3.5109077795259247`*^9, 3.510907781974065*^9}}],

Cell["\<\
04/26/10 - First release.
\
\>", "Text",
 CellChangeTimes->{{3.510907788475436*^9, 3.510907825669564*^9}, {
   3.5109981953783193`*^9, 3.5109981973127227`*^9}, 3.5111293590005293`*^9, {
   3.511301582319485*^9, 3.5113015951115074`*^9}, {3.5127252375215144`*^9, 
   3.512725238395116*^9}, {3.512725273198777*^9, 3.5127252867084007`*^9}}]
}, Closed]],

Cell[CellGroupData[{

Cell["Coordinate System", "Subsection",
 CellChangeTimes->{{3.5103935324149256`*^9, 3.510393535246088*^9}, {
  3.5103940971398506`*^9, 3.5103941199158907`*^9}}],

Cell["\<\
(VIRGO's Reference)
X-axis:  Transversal, Y-axis: Vertical, Z-axis: Longitudinal (Beam-axis)
Pitch: Rotation around x-axis, Yaw: Rotation around y-axis, Roll: Rotation \
around z-axis
\
\>", "Text",
 CellChangeTimes->{{3.5103935528940973`*^9, 3.510393696350746*^9}, {
   3.510394123098296*^9, 3.5103943553983045`*^9}, {3.510395802459265*^9, 
   3.510395835515723*^9}, {3.510624429500894*^9, 3.510624435179304*^9}, {
   3.5109079621453695`*^9, 3.510908048918333*^9}, {3.5109081088107586`*^9, 
   3.5109081217995014`*^9}, 3.512725284446397*^9}]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Preliminaries", "Section",
 CellChangeTimes->{{3.509358217720311*^9, 3.5093582194831142`*^9}, {
  3.509358283131226*^9, 3.5093582840048275`*^9}, {3.5109081602447004`*^9, 
  3.5109081665040584`*^9}}],

Cell[CellGroupData[{

Cell["Clear Values", "Subsection",
 CellChangeTimes->{{3.5106377661991253`*^9, 3.5106377726731367`*^9}, {
  3.510908300193705*^9, 3.510908303030867*^9}, {3.510908796062067*^9, 
  3.5109088095318375`*^9}}],

Cell["Clear values and definitions of all the symbols.", "Text",
 CellChangeTimes->{{3.5109082594663754`*^9, 3.510908268789909*^9}, {
  3.5109083050389824`*^9, 3.5109083170736704`*^9}, {3.510908356790942*^9, 
  3.5109083909098935`*^9}, {3.5109084458170342`*^9, 3.510908446235058*^9}, {
  3.5112987930589027`*^9, 3.5112987971351357`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{"Clear", "[", "\"\<Global`*\>\"", "]"}], " ", "*)"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5101923438572636`*^9, 3.510192346852469*^9}, {
   3.5105415211971025`*^9, 3.5105415220863047`*^9}, {3.5106246850761433`*^9, 
   3.5106246954033613`*^9}, {3.5106377478222933`*^9, 3.510637755653507*^9}, 
   3.5109583818557067`*^9, {3.511298860191743*^9, 3.511298870315322*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Alart Switch OFF", "Subsection",
 CellChangeTimes->{{3.5093582266123266`*^9, 3.509358235441942*^9}, {
  3.510908785327453*^9, 3.510908786812538*^9}, {3.512712672876814*^9, 
  3.51271269337525*^9}}],

Cell[CellGroupData[{

Cell["\<\
Disable spell-checking. There are so many variables with similar names that \
it' s useless.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5109082158268795`*^9, 3.5109082167409315`*^9}, 
   3.512712667838005*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"Off", "[", 
   RowBox[{
    RowBox[{"General", "::", "spell"}], ",", 
    RowBox[{"General", "::", "spell1"}]}], "]"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5106245299962707`*^9, 3.510624543927095*^9}, {
   3.5106247451362486`*^9, 3.5106247454950495`*^9}, {3.5106377976799808`*^9, 
   3.5106377985379825`*^9}, 3.510958382947708*^9, 3.512712667838005*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Disable the LinerSolve alart.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511418292811921*^9, 3.5114183319367895`*^9}, {
  3.5114185384811525`*^9, 3.5114185527395773`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"Off", "[", 
   RowBox[{"LinearSolve", "::", "luc"}], "]"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5110035674345737`*^9, 3.5110035736277847`*^9}, {
  3.5114183319367895`*^9, 3.511418334307994*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Set Kernel Directory", "Subsection",
 CellChangeTimes->{{3.5101923522344785`*^9, 3.510192358162489*^9}}],

Cell["\<\
Set the current working directory to the directory where this notebook exists.
Imported data and packages should be placed in this directory, or the parent \
directory.\
\>", "Text",
 CellChangeTimes->{{3.5109084715615067`*^9, 3.510908486584366*^9}, {
   3.510908577266553*^9, 3.510908645640464*^9}, {3.5112989682869253`*^9, 
   3.5112990157796416`*^9}, {3.5113016406323876`*^9, 3.5113016472467995`*^9}, 
   3.5114313091697197`*^9, 3.5114315039128633`*^9, {3.512725476732334*^9, 
   3.5127254897115574`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SetDirectory", "[", 
    RowBox[{"NotebookDirectory", "[", "]"}], "]"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{3.510192369066908*^9, 3.510958384320511*^9, 
  3.5114313091697197`*^9, 3.5114315039128633`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Import Packages", "Subsection",
 CellChangeTimes->{{3.509358549500699*^9, 3.5093585508579016`*^9}, {
  3.510908771531664*^9, 3.510908776462946*^9}, {3.511431144979431*^9, 
  3.5114311459622326`*^9}}],

Cell[CellGroupData[{

Cell["Standard package for use in plotting transfer functions.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109086828285904`*^9, 3.510908700501601*^9}, {
  3.5109087550517216`*^9, 3.5109087629341726`*^9}, {3.5114311527794447`*^9, 
  3.511431169050273*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"<<", "\"\<PlotLegends`\>\""}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{
  3.5093585572227125`*^9, 3.5109583868009152`*^9, 3.5114311527794447`*^9, {
   3.5114314680016003`*^9, 3.5114314701388035`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Manual package for modeling and simulating vibration isolation systems.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5114311609694595`*^9, 3.511431194384718*^9}, 
   3.511431248501213*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"<<", "\"\<../visUtil.m\>\""}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5114311961163206`*^9, 3.5114312118099484`*^9}, {
   3.5114312446636057`*^9, 3.511431248501213*^9}, {3.51143132645455*^9, 
   3.511431328217353*^9}, {3.5114314592187843`*^9, 3.51143146218279*^9}, {
   3.511432205581296*^9, 3.5114322437545633`*^9}, 3.5126997613310375`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Unit System", "Subsection",
 CellChangeTimes->{{3.507762673125991*^9, 3.5077626748989697`*^9}, {
  3.507887916125967*^9, 3.5078879248934684`*^9}, {3.510909011354381*^9, 
  3.5109090133324947`*^9}}],

Cell["\<\
MKS unit system is set as defalut. If you want to use another unit system or \
add units, please edit this subsection.\
\>", "Text",
 CellChangeTimes->{{3.510909029719432*^9, 3.5109090347067165`*^9}, {
  3.5112991343644247`*^9, 3.511299152603468*^9}, {3.5112992219044313`*^9, 
  3.5112992467198505`*^9}, {3.5112993190699887`*^9, 3.5112993260563884`*^9}, {
  3.5112996874180574`*^9, 3.5112997335146937`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"kg", "=", "1.0"}], ";", 
   RowBox[{"meter", "=", "1.0"}], ";", 
   RowBox[{"rad", "=", "1.0"}], ";", 
   RowBox[{"sec", "=", "1.0"}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Basical", " ", "Set"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"cm", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "2"}]], "meter"}]}], ";", 
   RowBox[{"mm", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "meter"}]}], ";", " ", 
   RowBox[{"km", "=", 
    RowBox[{
     SuperscriptBox["10", "3"], "meter"}]}], ";"}], 
  RowBox[{"(*", "Length", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"gram", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "kg"}]}], ";", " ", 
   RowBox[{"mg", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "6"}]], "kg"}]}], ";"}], 
  RowBox[{"(*", "Mass", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"deg", "=", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"N", "[", "Pi", "]"}], "/", "180"}], ")"}], "rad"}]}], ";", 
   " ", 
   RowBox[{"mrad", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "rad"}]}], ";"}], 
  RowBox[{"(*", "Angle", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"hertz", "=", 
    RowBox[{"1", "/", "sec"}]}], ";", 
   RowBox[{"mHz", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "hertz"}]}], ";", " ", 
   RowBox[{"kHz", "=", 
    RowBox[{
     SuperscriptBox["10", "3"], "hertz"}]}], ";"}], 
  RowBox[{"(*", "Frequency", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"newton", "=", 
    RowBox[{"kg", " ", 
     RowBox[{"meter", "/", 
      SuperscriptBox["sec", "2"]}]}]}], ";", 
   RowBox[{"pascal", "=", 
    RowBox[{"newton", "/", 
     SuperscriptBox["meter", "2"]}]}], ";"}], 
  RowBox[{"(*", "Force", "*)"}]}], "\n"}], "Input",
 CellChangeTimes->{{3.505459898591352*^9, 3.5054599656528482`*^9}, {
   3.505460214618761*^9, 3.505460246823139*^9}, {3.505460353839837*^9, 
   3.505460386329871*^9}, {3.5054609432258244`*^9, 3.505460943823134*^9}, {
   3.507717502489213*^9, 3.50771751561602*^9}, {3.507763954546257*^9, 
   3.50776398513273*^9}, {3.507887947930786*^9, 3.5078880664505653`*^9}, {
   3.507888099557459*^9, 3.50788814661615*^9}, {3.507888389647051*^9, 
   3.5078883993246045`*^9}, {3.507888436328721*^9, 3.507888448082393*^9}, 
   3.5109090416871157`*^9, 3.5109090778881865`*^9, {3.510909120506624*^9, 
   3.5109091273490157`*^9}, {3.5109092151620665`*^9, 3.510909215271267*^9}, {
   3.511300808916927*^9, 3.511300836123375*^9}, {3.511300869803834*^9, 
   3.5113008723622384`*^9}}]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Constant Values", "Section",
 CellChangeTimes->{{3.51142744973974*^9, 3.5114274556677504`*^9}, {
  3.511427538098295*^9, 3.5114275409375*^9}, {3.5126981805490613`*^9, 
  3.512698220251131*^9}, {3.5126995238362207`*^9, 3.5126995252714233`*^9}, {
  3.512709794472556*^9, 3.5127098002289667`*^9}}],

Cell[CellGroupData[{

Cell["Version", "Subsection",
 CellChangeTimes->{{3.5127256954603186`*^9, 3.5127256967083206`*^9}}],

Cell["\<\
Mass & MOI: Proposed by R. Desalvo in 04/26/11 (TM 10-kg version, but reduced \
IRM [60 kg] to achieve 300 kg payload)
Wire Specification: Proposed by E. Majorana in 04/18/11 (TM 10-kg version)
\
\>", "Text",
 CellChangeTimes->{{3.5127257083459415`*^9, 3.5127258858898535`*^9}, {
  3.512772484472748*^9, 3.5127726066521626`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Physical Constant", "Subsection",
 CellChangeTimes->{{3.5126995351930404`*^9, 3.512699537813845*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"g", "=", 
    RowBox[{"9.81", " ", 
     RowBox[{"meter", "/", 
      SuperscriptBox["sec", "2"]}]}]}], ";"}], " ", 
  RowBox[{"(*", 
   RowBox[{"local", " ", "gravity", " ", "constant"}], "*)"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.507762704553597*^9, 3.5077627272070923`*^9}, {
   3.507763249921364*^9, 3.507763251940638*^9}, 3.507765884859458*^9, {
   3.5078741427757235`*^9, 3.5078741430097237`*^9}, {3.5078882875472107`*^9, 
   3.5078883798324895`*^9}, 3.510910789651633*^9, {3.5127255934829397`*^9, 
   3.512725594387741*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Material Property", "Subsection",
 CellChangeTimes->{{3.5078885954778233`*^9, 3.5078886008591313`*^9}}],

Cell[CellGroupData[{

Cell["Usage note for symbols", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127058900508986`*^9, 3.5127058993329153`*^9}, {
  3.512706096907262*^9, 3.512706100963269*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "::", "usage"}], "=", 
   "\"\<rho['mat'] is volume density of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"young", "::", "usage"}], "=", 
   "\"\<young['mat'] is Young's modulus of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "::", "usage"}], "=", 
   "\"\<phi['mat'] is loss angle of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"poisson", "::", "usage"}], "=", 
   "\"\<poisson['mat'] is Poisson ratio of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127059638078284`*^9, 3.512706105643277*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Maraging Steel (MS)", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5103965360817537`*^9, 3.510396544037768*^9}, 
   3.5127059285829663`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "[", "\"\<MS\>\"", "]"}], "=", 
   RowBox[{"8.0", 
    RowBox[{"gram", "/", 
     SuperscriptBox["cm", "3"]}]}]}], ";", 
  RowBox[{
   RowBox[{"young", "[", "\"\<MS\>\"", "]"}], "=", 
   RowBox[{"195", " ", 
    SuperscriptBox["10", "9"], " ", "pascal"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "[", "\"\<MS\>\"", "]"}], "=", 
   RowBox[{"1.0", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "3"}]]}]}], ";", 
  RowBox[{
   RowBox[{"poisson", "[", "\"\<MS\>\"", "]"}], "=", "0.9"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5078886139238787`*^9, 3.507888680536689*^9}, {
   3.507888827152075*^9, 3.5078888913997498`*^9}, {3.5078889230655603`*^9, 
   3.507889061793495*^9}, {3.507889125559222*^9, 3.5078891537796717`*^9}, {
   3.507889406024915*^9, 3.507889441421377*^9}, {3.5078960325775614`*^9, 
   3.5078960964908733`*^9}, {3.5103965183445225`*^9, 3.510396588887847*^9}, {
   3.510909272975768*^9, 3.510909298450613*^9}, 3.510910792100837*^9, {
   3.510917538160895*^9, 3.5109175628401384`*^9}, {3.5110012657717195`*^9, 
   3.5110013428284593`*^9}, {3.5113005011767845`*^9, 
   3.5113005170108128`*^9}, {3.511300658220662*^9, 3.511300688422315*^9}, {
   3.512355711763343*^9, 3.512355728764315*^9}, {3.512699581587522*^9, 
   3.5126995827419243`*^9}, 3.5127059285829663`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Tungsten (W)", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5103965944882565`*^9, 3.5103965978578625`*^9}, 
   3.512705931312971*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "[", "\"\<W\>\"", "]"}], "=", 
   RowBox[{"19.25", 
    RowBox[{"gram", "/", 
     SuperscriptBox["cm", "3"]}]}]}], ";", " ", 
  RowBox[{
   RowBox[{"young", "[", "\"\<W\>\"", "]"}], "=", 
   RowBox[{"411", " ", 
    SuperscriptBox["10", "9"], " ", "pascal"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "[", "\"\<W\>\"", "]"}], "=", 
   RowBox[{"1.0", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "3"}]]}]}], ";", 
  RowBox[{
   RowBox[{"poisson", "[", "\"\<W\>\"", "]"}], "=", "0.3"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5078886139238787`*^9, 3.507888680536689*^9}, {
   3.507888827152075*^9, 3.5078888913997498`*^9}, {3.5078889230655603`*^9, 
   3.507889061793495*^9}, {3.507889125559222*^9, 3.5078891537796717`*^9}, {
   3.507889406024915*^9, 3.5078894378489714`*^9}, {3.507896109594897*^9, 
   3.507896113026903*^9}, {3.5103966082162805`*^9, 3.5103966252515106`*^9}, 
   3.5107455826730013`*^9, {3.5109092790285788`*^9, 3.5109092799021807`*^9}, 
   3.510910793146039*^9, {3.510917571108153*^9, 3.510917577987765*^9}, {
   3.5110013469000664`*^9, 3.5110013527812767`*^9}, {3.5110014676084857`*^9, 
   3.5110014906029263`*^9}, {3.5113002673217673`*^9, 
   3.5113002845129976`*^9}, {3.5113004092804227`*^9, 3.511300412774829*^9}, {
   3.511300542797658*^9, 3.5113005435620594`*^9}, {3.5113006258066044`*^9, 
   3.511300626805006*^9}, 3.51130069099632*^9, {3.512355660889433*^9, 
   3.5123556611114454`*^9}, {3.512699585487529*^9, 3.5126995863299303`*^9}, 
   3.512705931312971*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["C-70 Steel (C70)", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.5107455881642113`*^9, 3.5107455934682207`*^9}, 
   3.510910365408888*^9, 3.512705934167776*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "[", "\"\<C70\>\"", "]"}], "=", 
   RowBox[{"7.8", 
    RowBox[{"gram", "/", 
     SuperscriptBox["cm", "3"]}]}]}], ";", " ", 
  RowBox[{
   RowBox[{"young", "[", "\"\<C70\>\"", "]"}], "=", 
   RowBox[{"200", " ", 
    SuperscriptBox["10", "9"], " ", "pascal"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "[", "\"\<C70\>\"", "]"}], "=", 
   RowBox[{"1.0", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "3"}]]}]}], ";", 
  RowBox[{
   RowBox[{"poisson", "[", "\"\<C70\>\"", "]"}], "=", "0.3"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.5078886139238787`*^9, 3.507888680536689*^9}, {
   3.507888827152075*^9, 3.5078888913997498`*^9}, {3.5078889230655603`*^9, 
   3.507889061793495*^9}, {3.507889125559222*^9, 3.5078891537796717`*^9}, {
   3.507889406024915*^9, 3.5078894378489714`*^9}, {3.507896109594897*^9, 
   3.507896113026903*^9}, {3.5103966082162805`*^9, 3.5103966252515106`*^9}, {
   3.5107455826730013`*^9, 3.5107456317818875`*^9}, {3.5109092830533857`*^9, 
   3.5109092838801875`*^9}, 3.510910794362841*^9, {3.510917573572957*^9, 
   3.510917582168572*^9}, {3.5110014921005287`*^9, 3.5110015006025434`*^9}, {
   3.5113005549188795`*^9, 3.511300557196483*^9}, {3.5113006740078897`*^9, 
   3.5113006841791077`*^9}, {3.512355689126048*^9, 3.512355699808659*^9}, {
   3.512699588825935*^9, 3.512699589590336*^9}, 3.512705934167776*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Bodies", "Subsection",
 CellChangeTimes->{{3.5106251018152747`*^9, 3.5106251044516797`*^9}, {
  3.5109104340490084`*^9, 3.5109104353594103`*^9}}],

Cell[CellGroupData[{

Cell["Standard Bodies", "Subsubsection",
 CellChangeTimes->{{3.511311704314874*^9, 3.5113117105236855`*^9}, 
   3.5127091783182745`*^9}],

Cell[CellGroupData[{

Cell["Usage note for symbols.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113114603616457`*^9, 3.5113114747136707`*^9}, {
   3.511311505554925*^9, 3.511311549718603*^9}, 3.5126982410615673`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"mass", "::", "usage"}], "=", "\"\<mass of bodies\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"moix", "::", "usage"}], "=", 
   "\"\<moment of inertia around x-axis (pitch)\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"moiy", "::", "usage"}], "=", 
   "\"\<moment of inertia around y-axis (yaw)\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"moiz", "::", "usage"}], "=", 
   "\"\<moment of inertia around z-axis (roll)\>\""}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113096085915933`*^9, 3.5113098531844225`*^9}, {
   3.511309893479294*^9, 3.5113099178465366`*^9}, 3.511309950247793*^9, {
   3.511311488410495*^9, 3.511311526942563*^9}, 3.5113153674308243`*^9, {
   3.5126982410615673`*^9, 3.5126983525861635`*^9}, {3.512725606945763*^9, 
   3.5127256082405653`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Values", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511307444101391*^9, 3.511307473382642*^9}, {
  3.511308315877722*^9, 3.511308321212931*^9}, {3.511311718276899*^9, 
  3.5113117198681016`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", "      ", 
   RowBox[{"IM", ",", "  ", "IRM", ",", "  ", "TM", ",", "  ", "RM"}], "   ", 
   "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"mass", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"55.0", ",", "60.0", ",", "10.7", ",", "90.3"}], "}"}], 
      "kg"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"moix", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.80", ",", "3.00", ",", ".051", ",", "2.49"}], "}"}], "kg", 
      " ", 
      SuperscriptBox["meter", "2"]}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"moiy", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1.88", ",", "6.00", ",", ".051", ",", "2.49"}], "}"}], "kg", 
      " ", 
      SuperscriptBox["meter", "2"]}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"moiz", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1.22", ",", "3.00", ",", ".084", ",", "4.39"}], "}"}], "kg", 
      " ", 
      SuperscriptBox["meter", "2"]}]}], ";"}], 
   "\[IndentingNewLine]"}]}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511304710127389*^9, 3.511304728535421*^9}, {
   3.5113047674418893`*^9, 3.5113048860488977`*^9}, {3.5113050074015107`*^9, 
   3.51130514904976*^9}, {3.5113051832138195`*^9, 3.51130538269137*^9}, {
   3.511305417479431*^9, 3.511305551904867*^9}, {3.5113058796614428`*^9, 
   3.511305880269844*^9}, {3.5113061359454927`*^9, 3.5113061641503425`*^9}, {
   3.511306624819152*^9, 3.5113066315271635`*^9}, 3.511307473382642*^9, {
   3.511308315877722*^9, 3.511308321212931*^9}, {3.511386760794197*^9, 
   3.511386828105047*^9}, {3.5113869415443287`*^9, 3.511386973617985*^9}, {
   3.5114939721454124`*^9, 3.511493984048234*^9}, {3.5114942197842493`*^9, 
   3.511494225041459*^9}, {3.51149502317006*^9, 3.5114950728873477`*^9}, {
   3.512355081541296*^9, 3.5123550980172386`*^9}, {3.5123562660702295`*^9, 
   3.512356447654548*^9}, {3.5126972323489933`*^9, 3.512697253565031*^9}, {
   3.5126973286319647`*^9, 3.5126973856968646`*^9}, {3.5126974251337337`*^9, 
   3.512697450421378*^9}, {3.512697521946704*^9, 3.512697534910327*^9}, {
   3.5126975666875825`*^9, 3.5126976978682127`*^9}, {3.512697732281873*^9, 
   3.512697844960871*^9}, {3.5126983687633915`*^9, 3.5126983919918327`*^9}, {
   3.5127063777233553`*^9, 3.51270638635017*^9}, {3.512711677657466*^9, 
   3.5127117195591393`*^9}, {3.5127117556108027`*^9, 
   3.5127119120322776`*^9}, {3.5127256372098165`*^9, 3.5127256429818263`*^9}, 
   3.512730602065339*^9, {3.512772450558288*^9, 3.5127724661115155`*^9}, {
   3.512773848387147*^9, 3.5127738638467736`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Labels", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5127067034831276`*^9, 3.51270670477793*^9}, 
   3.512707796343047*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"b$IM", "=", "1"}], ";", 
   RowBox[{"b$IRM", "=", "2"}], ";", 
   RowBox[{"b$TM", "=", "3"}], ";", 
   RowBox[{"b$RM", "=", "4"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5127067095515385`*^9, 3.5127068375965633`*^9}, {
   3.512706877985034*^9, 3.512706889950255*^9}, 3.512707796343047*^9, {
   3.512725650142239*^9, 3.5127256522014427`*^9}, {3.5127738743299923`*^9, 
   3.5127738822392063`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Wires", "Subsection",
 CellChangeTimes->{{3.511306120572666*^9, 3.5113061311874847`*^9}}],

Cell[CellGroupData[{

Cell["Properties and Geometric Values", "Subsubsection",
 CellChangeTimes->{{3.511311536146579*^9, 3.5113115410137873`*^9}, {
  3.5113118928256054`*^9, 3.5113118955244102`*^9}}],

Cell[CellGroupData[{

Cell["Usage note for symbols.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113114603616457`*^9, 3.5113114747136707`*^9}, {
   3.511311505554925*^9, 3.511311549718603*^9}, 3.5126982410615673`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"nw", "::", "usage"}], "=", "\"\<number of wires\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"matw", "::", "usage"}], "=", "\"\<wire material\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"lNw", "::", "usage"}], "=", "\"\<natural length of wires\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dw", "::", "usage"}], "=", "\"\<wire diameters\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dxu", "::", "usage"}], "=", 
   "\"\<x-separation of wires & COM at upper suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dzu", "::", "usage"}], "=", 
   "\"\<z-separation of wires & COM at upper suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dxl", "::", "usage"}], "=", 
   "\"\<x-separation of wires & COM at lower suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dzl", "::", "usage"}], "=", 
   "\"\<z-separation of wires & COM at lower suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dyu", "::", "usage"}], "=", 
   "\"\<height of wire break-off below COM at upper body\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dyl", "::", "usage"}], "=", 
   "\"\<height of wire break-off above COM at lower body\>\""}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113096085915933`*^9, 3.5113098531844225`*^9}, {
   3.511309893479294*^9, 3.5113099178465366`*^9}, 3.511309950247793*^9, {
   3.511311488410495*^9, 3.511311526942563*^9}, 3.5113153674308243`*^9, 
   3.5126982410615673`*^9, 3.512706179431407*^9, {3.512711537023219*^9, 
   3.512711552982047*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Values", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113115557402134`*^9, 3.5113115654278307`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", "      ", 
   RowBox[{"IM", ",", "  ", "IRM", ",", "  ", "TM", ",", "  ", "RM"}], 
   "     ", "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"matw", "=", 
     RowBox[{"{", 
      RowBox[{
      "\"\<MS\>\"", ",", "\"\<C70\>\"", ",", "\"\<W\>\"", ",", 
       "\"\<C70\>\""}], "}"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"lNw", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "2.084", "}"}], ",", "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"2.084", ",", "2.084", ",", "2.084", ",", "2.084"}], "}"}], 
        ",", "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"2.084", ",", "2.084", ",", "2.084", ",", "2.084"}], "}"}], 
        ",", "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"2.084", ",", "2.084", ",", "2.084", ",", "2.084"}], "}"}]}],
        "\[IndentingNewLine]", "}"}], "meter"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dw", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "2.10", "}"}], ",", "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"0.50", ",", "0.50", ",", "0.50", ",", "0.50"}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"0.20", ",", "0.20", ",", "0.20", ",", "0.20"}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"0.50", ",", "0.50", ",", "0.50", ",", "0.50"}], "}"}]}], 
       "}"}], "mm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dyu", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "0.00", "}"}], ",", "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"0.00", ",", "0.00", ",", "0.00", ",", "0.00"}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"0.00", ",", "0.00", ",", "0.00", ",", "0.00"}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"0.00", ",", "0.00", ",", "0.00", ",", "0.00"}], "}"}]}], 
       "}"}], "mm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dyl", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"-", "5.0"}], "}"}], ",", "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"1.00", ",", "1.00", ",", "1.00", ",", "1.00"}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"1.00", ",", "1.00", ",", "1.00", ",", "1.00"}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"1.00", ",", "1.00", ",", "1.00", ",", "1.00"}], "}"}]}], 
       "}"}], "mm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dxu", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.00", ",", "35.0", ",", "12.5", ",", "27.0"}], "}"}], 
      "cm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dzu", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.00", ",", "35.0", ",", "1.50", ",", "3.00"}], "}"}], 
      "cm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dxl", "=", "dxu"}], ";", 
    RowBox[{"dzl", "=", "dzu"}], ";", 
    RowBox[{"(*", 
     RowBox[{"vertical", " ", "suspension"}], "*)"}], "\[IndentingNewLine]", 
    RowBox[{"nw", "=", 
     RowBox[{"{", 
      RowBox[{"1", ",", "4", ",", "4", ",", "4"}], "}"}]}], ";"}], 
   "\[IndentingNewLine]"}]}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5126979552998652`*^9, 3.5126979725066953`*^9}, {
   3.512698487807201*^9, 3.512698593512986*^9}, 3.512699268042571*^9, {
   3.512706425599839*^9, 3.5127064636951065`*^9}, {3.512707714364903*^9, 
   3.512707749839365*^9}, {3.5127079340132885`*^9, 3.512707950642918*^9}, {
   3.5127115595340586`*^9, 3.5127115689564753`*^9}, {3.5127119288491073`*^9, 
   3.5127120018884354`*^9}, {3.5127120337436914`*^9, 3.512712065645747*^9}, {
   3.5127121919123693`*^9, 3.5127122302416363`*^9}, {3.5127259305527315`*^9, 
   3.5127259383683453`*^9}, 3.512730513737984*^9, {3.512773898042034*^9, 
   3.5127739340936975`*^9}, {3.5127744525294075`*^9, 3.5127744565542145`*^9}, 
   3.5127745474399743`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Effective Bending Point Compensation", "Subsubsection",
 CellChangeTimes->{{3.511310011805502*^9, 3.511310013724305*^9}, {
  3.5113101034712625`*^9, 3.511310109976474*^9}, {3.511312045081873*^9, 
  3.5113120493874807`*^9}}],

Cell[CellGroupData[{

Cell["Discription", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511310024020323*^9, 3.5113100245975237`*^9}, {
   3.5113101204128923`*^9, 3.511310123595298*^9}, 3.511311097317808*^9}],

Cell["\<\
Bending point of wire is apart from the suspension point because of the wire \
bending stiffness.
In this subsubsection, the separation between effective bending point & \
suspension point is estimated (1/lambda),
and then wire length and wire break-off are compensated. If you want to \
disable this compensation, 
please make the following symbol \"False\".\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511310128790107*^9, 3.511310207960246*^9}, {
   3.511310336208071*^9, 3.511310459167487*^9}, {3.511310496030352*^9, 
   3.511310656320634*^9}, {3.51131089921306*^9, 3.5113110646045504`*^9}, 
   3.511311097317808*^9, 3.5113135489584303`*^9, {3.512725999270852*^9, 
   3.51272600367006*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"wirecompensation", "=", "True"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511311074011367*^9, 3.511311097317808*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Load weight", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511308651590311*^9, 3.511308660326327*^9}, {
  3.51131116687833*^9, 3.51131117236954*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"loadw", "=", 
      RowBox[{"{", "}"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"AppendTo", "[", 
      RowBox[{"loadw", ",", 
       RowBox[{
        RowBox[{"mass", "[", 
         RowBox[{"[", "b$IM", "]"}], "]"}], "+", 
        RowBox[{"mass", "[", 
         RowBox[{"[", "b$TM", "]"}], "]"}], "+", 
        RowBox[{"mass", "[", 
         RowBox[{"[", "b$RM", "]"}], "]"}]}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"AppendTo", "[", 
      RowBox[{"loadw", ",", 
       RowBox[{"mass", "[", 
        RowBox[{"[", "b$IRM", "]"}], "]"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"AppendTo", "[", 
      RowBox[{"loadw", ",", 
       RowBox[{"mass", "[", 
        RowBox[{"[", "b$TM", "]"}], "]"}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"AppendTo", "[", 
      RowBox[{"loadw", ",", 
       RowBox[{"mass", "[", 
        RowBox[{"[", "b$RM", "]"}], "]"}]}], "]"}], ";"}]}], 
   "\[IndentingNewLine]", "]"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113086367702856`*^9, 3.5113087745029273`*^9}, {
   3.5113088097589893`*^9, 3.5113088188382053`*^9}, {3.5113088861367235`*^9, 
   3.511308890411131*^9}, {3.51131110959503*^9, 3.5113111384394803`*^9}, 
   3.51131117236954*^9, {3.512706614687772*^9, 3.512706633719805*^9}, {
   3.512706853539791*^9, 3.5127068600294027`*^9}, {3.5127069021182766`*^9, 
   3.512706931742728*^9}, 3.5127739589133406`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Second moment of area", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113089605956545`*^9, 3.511309018003755*^9}, 
   3.5113093541395464`*^9, {3.511309392125613*^9, 3.511309434089687*^9}, {
   3.511311157487114*^9, 3.5113111858791637`*^9}, 3.511311299977764*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"iw", "=", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"N", "[", "Pi", "]"}], 
         RowBox[{
          SuperscriptBox[
           RowBox[{"dw", "[", 
            RowBox[{"[", 
             RowBox[{"n", ",", "m"}], "]"}], "]"}], "4"], "/", "64"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "dw", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"nw", "[", 
           RowBox[{"[", "n", "]"}], "]"}]}], "}"}]}], "]"}]}], ";"}]}], 
   "\[IndentingNewLine]", "]"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113091838496475`*^9, 3.5113092569513755`*^9}, {
   3.5113093129398737`*^9, 3.5113093236882925`*^9}, 3.5113093541395464`*^9, {
   3.511311150248701*^9, 3.5113111520583043`*^9}, {3.5113111858791637`*^9, 
   3.51131121202481*^9}, 3.511311299977764*^9, {3.5113151870237074`*^9, 
   3.5113152264761767`*^9}, {3.511315400799283*^9, 3.511315403076887*^9}, {
   3.512358626588375*^9, 3.512358627446377*^9}, 3.512706501493973*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Lambda", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511311218998022*^9, 3.5113112216032267`*^9}, 
   3.511311306186575*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"lambdaw", "=", 
      RowBox[{"Table", "[", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Sqrt", "[", 
         RowBox[{
          RowBox[{"loadw", "[", 
           RowBox[{"[", "n", "]"}], "]"}], 
          RowBox[{
           RowBox[{
            RowBox[{"g", "/", 
             RowBox[{"nw", "[", 
              RowBox[{"[", "n", "]"}], "]"}]}], "/", 
            RowBox[{"young", "[", 
             RowBox[{"matw", "[", 
              RowBox[{"[", "n", "]"}], "]"}], "]"}]}], "/", 
           RowBox[{"iw", "[", 
            RowBox[{"[", 
             RowBox[{"n", ",", "m"}], "]"}], "]"}]}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "dw", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"nw", "[", 
           RowBox[{"[", "n", "]"}], "]"}]}], "}"}]}], "\[IndentingNewLine]", 
       "]"}]}], ";"}]}], "\[IndentingNewLine]", "]"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511309025819369*^9, 3.5113091071423116`*^9}, {
   3.511309263518987*^9, 3.5113093858388023`*^9}, 3.5113111858791637`*^9, {
   3.5113112437708654`*^9, 3.5113112539732833`*^9}, 3.511311306186575*^9, {
   3.5113134454678483`*^9, 3.5113134500074563`*^9}, {3.5113152729798584`*^9, 
   3.5113152861306815`*^9}, {3.511315413669306*^9, 3.511315416804911*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Compensation", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.511309454182522*^9, 3.511309456428926*^9}, {
   3.5113113155153913`*^9, 3.511311321724202*^9}, 3.5113115879074697`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"lNw", "=", 
      RowBox[{"lNw", "-", 
       RowBox[{"2", "/", "lambdaw"}]}]}], ";", 
     RowBox[{"(*", 
      RowBox[{"wire", " ", "length"}], "*)"}], "\[IndentingNewLine]", 
     RowBox[{"dyu", "=", 
      RowBox[{"dyu", "+", 
       RowBox[{"1", "/", "lambdaw"}]}]}], ";", 
     RowBox[{"(*", 
      RowBox[{
       RowBox[{"wire", " ", "break"}], "-", 
       RowBox[{"off", " ", "at", " ", "upper", " ", "body"}]}], "*)"}], 
     "\[IndentingNewLine]", 
     RowBox[{"dyl", "=", 
      RowBox[{"dyl", "+", 
       RowBox[{"1", "/", "lambdaw"}]}]}], ";"}]}], 
   RowBox[{"(*", 
    RowBox[{
     RowBox[{"wire", " ", "break"}], "-", 
     RowBox[{"off", " ", "at", " ", "lower", " ", "body"}]}], "*)"}], 
   "\[IndentingNewLine]", "]"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.5113113332994223`*^9, 3.5113114085071545`*^9}, {
  3.5113115879074697`*^9, 3.5113116037102976`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Suspension Points", "Subsubsection",
 CellChangeTimes->{{3.511313811132491*^9, 3.5113138177781024`*^9}, 
   3.5113138856070213`*^9}],

Cell[CellGroupData[{

Cell["Description", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511327852568959*^9, 3.511327855345764*^9}, 
   3.51132791554627*^9}],

Cell["\<\
The clamp positions of wires in local coordinate of the attached mass.
If you want to add geometric asymmetry to the system, please arrange the \
following definition.
\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113160716784616`*^9, 3.511316135513774*^9}, {
   3.511316199583086*^9, 3.511316246585969*^9}, 3.51132791554627*^9, 
   3.5126992435505285`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Body corrdinates for all objects originate at COM", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.510968232951127*^9, 3.510968236383133*^9}, 
   3.511327918369875*^9, {3.5113279682587624`*^9, 3.511327969631565*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"posCOM", "=", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5109682272727165`*^9, 3.5109682511563587`*^9}, 
   3.511327918369875*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Suspension Points at upper bodies", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113278633641787`*^9, 3.511327866530984*^9}, 
   3.5113279214898806`*^9, {3.51269895991083*^9, 3.5126989766652594`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"spu", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Key", " ", "Stone", " ", "4"}], "*)"}], 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "0"}], "}"}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Filter", " ", "4"}], "*)"}], "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "2"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "3"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "4"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}]}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Intermediate", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "1"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "2"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "3"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "4"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}]}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Intermediate", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "1"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "2"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "3"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dyu", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "4"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzu", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}]}], "}"}]}], 
     "\[IndentingNewLine]", "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.510541271815065*^9, 3.5105413275383625`*^9}, {
   3.510541364338827*^9, 3.5105414659262056`*^9}, {3.5105415729735937`*^9, 
   3.5105416770257764`*^9}, {3.510541718537449*^9, 3.510541745213496*^9}, {
   3.5105418057416024`*^9, 3.5105418656925077`*^9}, {3.5105419541758633`*^9, 
   3.510541964097481*^9}, {3.510562740817388*^9, 3.51056276442023*^9}, {
   3.5106273753944683`*^9, 3.5106273856904864`*^9}, {3.5106361554478965`*^9, 
   3.5106361572418995`*^9}, {3.510636263328886*^9, 3.5106362654972897`*^9}, {
   3.510636399979926*^9, 3.510636408606741*^9}, {3.510655831710415*^9, 
   3.510655836608824*^9}, {3.510910690123458*^9, 3.5109107397159452`*^9}, {
   3.5109112296192055`*^9, 3.5109112741416836`*^9}, {3.510999339027935*^9, 
   3.5109993921616287`*^9}, {3.511154589829262*^9, 3.51115459458727*^9}, {
   3.511219031273158*^9, 3.5112190316787586`*^9}, {3.5113138771362066`*^9, 
   3.511313877448207*^9}, {3.511313922329486*^9, 3.511314076176956*^9}, {
   3.5113144895308824`*^9, 3.511314559107004*^9}, {3.511315592929221*^9, 
   3.5113156070628457`*^9}, {3.5113156392145023`*^9, 
   3.5113160247691793`*^9}, {3.5113268353067727`*^9, 3.5113268435123873`*^9}, 
   3.5113279214898806`*^9, {3.5126986628863087`*^9, 3.512698934170785*^9}, 
   3.512699128297526*^9, {3.512699214409677*^9, 3.5126992208680887`*^9}, {
   3.512707995851797*^9, 3.5127080201878405`*^9}, {3.512712331173814*^9, 
   3.512712357038659*^9}, 3.512773990581396*^9, {3.5127745616203995`*^9, 
   3.5127745896224484`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Suspension points at lower bodies", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.5126989944024906`*^9, 3.512699019736935*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"spl", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Intermediate", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"0", ",", 
         RowBox[{"+", 
          RowBox[{"dyl", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ",", "0"}], "}"}], "}"}],
       ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Intermediate", " ", "Recoil", " ", "Mass"}], "*)"}], 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "2"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "3"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", "4"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}]}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Test", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "1"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "2"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "3"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"3", ",", "4"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "3", "]"}], "]"}]}]}], "}"}]}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"(*", 
        RowBox[{"Recoil", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "1"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "2"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "3"}], "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"+", 
           RowBox[{"dxl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}], ",", 
          RowBox[{"+", 
           RowBox[{"dyl", "[", 
            RowBox[{"[", 
             RowBox[{"4", ",", "4"}], "]"}], "]"}]}], ",", 
          RowBox[{"-", 
           RowBox[{"dzl", "[", 
            RowBox[{"[", "4", "]"}], "]"}]}]}], "}"}]}], "}"}]}], 
     "\[IndentingNewLine]", "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.510541271815065*^9, 3.5105413275383625`*^9}, {
   3.510541364338827*^9, 3.5105414659262056`*^9}, {3.5105415729735937`*^9, 
   3.5105416770257764`*^9}, {3.510541718537449*^9, 3.510541745213496*^9}, {
   3.5105418057416024`*^9, 3.5105418656925077`*^9}, {3.5105419541758633`*^9, 
   3.510541964097481*^9}, {3.510562740817388*^9, 3.51056276442023*^9}, {
   3.5106273753944683`*^9, 3.5106273856904864`*^9}, {3.5106361554478965`*^9, 
   3.5106361572418995`*^9}, {3.510636263328886*^9, 3.5106362654972897`*^9}, {
   3.510636399979926*^9, 3.510636408606741*^9}, {3.510655831710415*^9, 
   3.510655836608824*^9}, {3.510910690123458*^9, 3.5109107397159452`*^9}, {
   3.5109112296192055`*^9, 3.5109112741416836`*^9}, {3.510999339027935*^9, 
   3.5109993921616287`*^9}, {3.511154589829262*^9, 3.51115459458727*^9}, {
   3.511219031273158*^9, 3.5112190316787586`*^9}, {3.5113138771362066`*^9, 
   3.511313877448207*^9}, {3.511313922329486*^9, 3.511314076176956*^9}, {
   3.5113144895308824`*^9, 3.511314559107004*^9}, {3.511315592929221*^9, 
   3.5113156070628457`*^9}, {3.5113156392145023`*^9, 
   3.5113160247691793`*^9}, {3.5113268353067727`*^9, 3.5113268435123873`*^9}, 
   3.5113279214898806`*^9, {3.5126986628863087`*^9, 3.512698934170785*^9}, {
   3.512699019736935*^9, 3.512699078798639*^9}, {3.51269913639394*^9, 
   3.5126992265620985`*^9}, {3.5127080281594543`*^9, 3.512708062214314*^9}, {
   3.5127123629042697`*^9, 3.512712393121523*^9}, 3.5127740004250135`*^9, {
   3.5127745972664623`*^9, 3.51277462470691*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["GAS Filters", "Subsection",
 CellChangeTimes->{{3.5110753524379687`*^9, 3.51107535310877*^9}, {
  3.5113185577100286`*^9, 3.5113185602528334`*^9}}],

Cell[CellGroupData[{

Cell["Stiffness", "Subsubsection",
 CellChangeTimes->{{3.5113166496750765`*^9, 3.511316666757107*^9}, {
  3.5113167132919884`*^9, 3.5113167240872073`*^9}, {3.5113177945098886`*^9, 
  3.51131779518069*^9}}],

Cell[CellGroupData[{

Cell["Resonant frequency and loss angle", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113168213533783`*^9, 3.5113168316493964`*^9}, {
  3.5113185852440777`*^9, 3.5113185893156843`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", "       ", "F4", "       ", "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"f0GAS", "=", 
     RowBox[{
      RowBox[{"{", "330", "}"}], "mHz"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"phiGAS", "=", 
     RowBox[{"{", "0.01", "}"}]}], ";"}], "\[IndentingNewLine]"}]}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511316742120839*^9, 3.5113168316493964`*^9}, {
  3.511318591889689*^9, 3.5113186119669247`*^9}, {3.511329385379252*^9, 
  3.511329385472852*^9}, {3.511494276037948*^9, 3.511494281232757*^9}, {
  3.5127740310946674`*^9, 3.512774036336277*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Load weight", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113168396054106`*^9, 3.5113168431622167`*^9}, 
   3.5113177835742693`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"loadGAS", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"AppendTo", "[", 
   RowBox[{"loadGAS", ",", 
    RowBox[{
     RowBox[{"mass", "[", 
      RowBox[{"[", "b$IM", "]"}], "]"}], "+", 
     RowBox[{"mass", "[", 
      RowBox[{"[", "b$TM", "]"}], "]"}], "+", 
     RowBox[{"mass", "[", 
      RowBox[{"[", "b$RM", "]"}], "]"}]}]}], "]"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113168459234214`*^9, 3.511316894018306*^9}, {
   3.5113177342469826`*^9, 3.5113177835742693`*^9}, {3.512707115339451*^9, 
   3.512707137163889*^9}, 3.51277404382429*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Vertical Stiffness", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113177997670975`*^9, 3.511317803713905*^9}, 
   3.511318346860059*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"kGAS", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"loadGAS", "[", 
       RowBox[{"[", "n", "]"}], "]"}], 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"2", 
         RowBox[{"N", "[", "Pi", "]"}], 
         RowBox[{"f0GAS", "[", 
          RowBox[{"[", "n", "]"}], "]"}]}], ")"}], "2"]}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", 
       RowBox[{"Length", "[", "f0GAS", "]"}]}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"kGASc", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"loadGAS", "[", 
       RowBox[{"[", "n", "]"}], "]"}], 
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"I", " ", 
         RowBox[{"phiGAS", "[", 
          RowBox[{"[", "n", "]"}], "]"}]}]}], ")"}], 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"2", 
         RowBox[{"N", "[", "Pi", "]"}], 
         RowBox[{"f0GAS", "[", 
          RowBox[{"[", "n", "]"}], "]"}]}], ")"}], "2"]}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", 
       RowBox[{"Length", "[", "f0GAS", "]"}]}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511318257549902*^9, 3.511318346860059*^9}, {
  3.5113186244781466`*^9, 3.5113186501869917`*^9}, {3.5127326436425276`*^9, 
  3.5127326447813296`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Working Point & Direction", "Subsubsection",
 CellChangeTimes->{{3.511318358560079*^9, 3.5113183725221033`*^9}, {
  3.5113184710830765`*^9, 3.511318474639883*^9}}],

Cell[CellGroupData[{

Cell["Working position in local coordinate system of Filters", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113184065613637`*^9, 3.5113184581974545`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"wpGAS", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"-", 
        RowBox[{"dyu", "[", 
         RowBox[{"[", 
          RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ",", "0"}], "}"}], 
     "\[IndentingNewLine]", "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511075372624404*^9, 3.5110755086098433`*^9}, {
   3.511078738029937*^9, 3.511078805141255*^9}, {3.511130324732766*^9, 
   3.5111303256808205`*^9}, {3.51113120990919*^9, 3.511131263807285*^9}, {
   3.511131301715351*^9, 3.5111313116993685`*^9}, {3.5111315230641403`*^9, 
   3.511131537010565*^9}, {3.511134239146511*^9, 3.511134263607354*^9}, {
   3.511134354009513*^9, 3.5111343542903132`*^9}, {3.5111345428634443`*^9, 
   3.5111345430038447`*^9}, {3.511139200314025*^9, 3.511139200891226*^9}, {
   3.511146068745289*^9, 3.51114606933809*^9}, {3.511147884884879*^9, 
   3.5111479220441446`*^9}, {3.5111494918593016`*^9, 3.511149492046502*^9}, 
   3.511150491431057*^9, {3.5111506726877756`*^9, 3.5111506756205807`*^9}, 
   3.511152181304025*^9, {3.5111522247657013`*^9, 3.5111522629389687`*^9}, 
   3.511153090021221*^9, {3.511154622589319*^9, 3.51115462296372*^9}, {
   3.5111548408181024`*^9, 3.51115484506131*^9}, {3.5112182732458262`*^9, 
   3.5112182916694584`*^9}, {3.511316690640749*^9, 3.5113167012955675`*^9}, {
   3.511318380041317*^9, 3.5113183970765467`*^9}, {3.511318445967033*^9, 
   3.5113184581974545`*^9}, {3.5113188332689133`*^9, 3.51131884275373*^9}, {
   3.5113234944167*^9, 3.5113235005007105`*^9}, {3.5126993813299704`*^9, 
   3.512699384559176*^9}, {3.5127107390698156`*^9, 3.5127107454970264`*^9}, {
   3.5127740633243237`*^9, 3.512774073105541*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Working direction in local coordinate system of Filters", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113184793822913`*^9, 3.511318487291505*^9}, 
   3.5113187131955023`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"wdGAS", "=", 
    RowBox[{"{", 
     RowBox[{"(*", "vertical", "*)"}], "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}], "\[IndentingNewLine]", 
     "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511075372624404*^9, 3.5110755086098433`*^9}, {
   3.511078738029937*^9, 3.511078805141255*^9}, {3.511130324732766*^9, 
   3.5111303256808205`*^9}, {3.51113120990919*^9, 3.511131263807285*^9}, {
   3.511131301715351*^9, 3.5111313116993685`*^9}, {3.5111315230641403`*^9, 
   3.511131537010565*^9}, {3.511134239146511*^9, 3.511134263607354*^9}, {
   3.511134354009513*^9, 3.5111343542903132`*^9}, {3.5111345428634443`*^9, 
   3.5111345430038447`*^9}, {3.511139200314025*^9, 3.511139200891226*^9}, {
   3.511146068745289*^9, 3.51114606933809*^9}, {3.511147884884879*^9, 
   3.5111479220441446`*^9}, {3.5111494918593016`*^9, 3.511149492046502*^9}, 
   3.511150491431057*^9, {3.5111506726877756`*^9, 3.5111506756205807`*^9}, 
   3.511152181304025*^9, {3.5111522247657013`*^9, 3.5111522629389687`*^9}, 
   3.511153090021221*^9, {3.511154622589319*^9, 3.51115462296372*^9}, {
   3.5111548408181024`*^9, 3.51115484506131*^9}, {3.5112182732458262`*^9, 
   3.5112182916694584`*^9}, {3.511316690640749*^9, 3.5113167012955675`*^9}, {
   3.511318380041317*^9, 3.5113183970765467`*^9}, 3.511318445967033*^9, 
   3.5113185773192635`*^9, {3.511318679702243*^9, 3.5113187131955023`*^9}, {
   3.511318810352473*^9, 3.51131882010249*^9}, {3.511322861866789*^9, 
   3.5113228718040066`*^9}, {3.512707318529808*^9, 3.5127073231162157`*^9}, {
   3.5127740686127334`*^9, 3.5127740702975364`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness Matrix & Storing Force Vector", "Subsubsection",
 CellChangeTimes->{{3.5113187371571445`*^9, 3.5113187452223587`*^9}, {
  3.511318874530986*^9, 3.511318883922202*^9}}],

Cell[CellGroupData[{

Cell["Stiffness matrix", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511318890068613*^9, 3.511318893001418*^9}, 
   3.511319080342147*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matkGAS", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{"Table", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"DiagonalMatrix", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"kGAS", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", "0", ",", "0", ",", "0", ",", 
          "0"}], "}"}], "]"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"n", ",", 
         RowBox[{"Length", "[", "wpGAS", "]"}]}], "}"}]}], 
      "\[IndentingNewLine]", "]"}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5111311698795195`*^9, 3.5111311941375628`*^9}, {
   3.511131337860615*^9, 3.511131457653225*^9}, {3.5111317695913734`*^9, 
   3.511131777110586*^9}, {3.511131811649047*^9, 3.5111319010060043`*^9}, {
   3.511131942798477*^9, 3.5111319912833624`*^9}, {3.5111346002715454`*^9, 
   3.511134738175788*^9}, 3.511139306721812*^9, {3.5111437243419714`*^9, 
   3.5111437272279763`*^9}, {3.5111460759837017`*^9, 3.511146076576503*^9}, 
   3.511318867557773*^9, {3.5113189150130568`*^9, 3.5113189754007626`*^9}, {
   3.511319080342147*^9, 3.5113191008873835`*^9}, {3.512707360119481*^9, 
   3.512707373909905*^9}, {3.512733964933648*^9, 3.5127339789580727`*^9}, {
   3.512735039026735*^9, 3.512735043488343*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness matrix (with imaginary part)", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511318890068613*^9, 3.511318893001418*^9}, 
   3.511319080342147*^9, 3.5127261836631765`*^9, {3.512726360193086*^9, 
   3.512726364389494*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matkGASc", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{"Table", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"DiagonalMatrix", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"kGASc", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", "0", ",", "0", ",", "0", ",", 
          "0"}], "}"}], "]"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"n", ",", 
         RowBox[{"Length", "[", "wpGAS", "]"}]}], "}"}]}], 
      "\[IndentingNewLine]", "]"}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5111311698795195`*^9, 3.5111311941375628`*^9}, {
   3.511131337860615*^9, 3.511131457653225*^9}, {3.5111317695913734`*^9, 
   3.511131777110586*^9}, {3.511131811649047*^9, 3.5111319010060043`*^9}, {
   3.511131942798477*^9, 3.5111319912833624`*^9}, {3.5111346002715454`*^9, 
   3.511134738175788*^9}, 3.511139306721812*^9, {3.5111437243419714`*^9, 
   3.5111437272279763`*^9}, {3.5111460759837017`*^9, 3.511146076576503*^9}, 
   3.511318867557773*^9, {3.5113189150130568`*^9, 3.5113189754007626`*^9}, {
   3.511319080342147*^9, 3.5113191008873835`*^9}, {3.512707360119481*^9, 
   3.512707373909905*^9}, 3.5127261836631765`*^9, {3.512726349912668*^9, 
   3.512726353141874*^9}, {3.5127333199505157`*^9, 3.512733324786524*^9}, {
   3.512733987413288*^9, 3.5127339935908985`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Storing force vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113189787547684`*^9, 3.511318984573579*^9}, 
   3.5113190853029556`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"vecfGAS", "=", 
    RowBox[{"Table", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"g", " ", 
         RowBox[{"loadGAS", "[", 
          RowBox[{"[", "n", "]"}], "]"}], 
         RowBox[{"wdGAS", "[", 
          RowBox[{"[", "n", "]"}], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"n", ",", 
        RowBox[{"Length", "[", "wdGAS", "]"}]}], "}"}]}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5111311698795195`*^9, 3.5111311941375628`*^9}, {
   3.511131337860615*^9, 3.511131457653225*^9}, {3.5111317695913734`*^9, 
   3.511131777110586*^9}, {3.511131811649047*^9, 3.5111319010060043`*^9}, {
   3.511131942798477*^9, 3.5111319912833624`*^9}, {3.5111346002715454`*^9, 
   3.511134738175788*^9}, 3.511139306721812*^9, {3.5111437243419714`*^9, 
   3.5111437272279763`*^9}, {3.5111460759837017`*^9, 3.511146076576503*^9}, 
   3.511318867557773*^9, {3.5113189150130568`*^9, 3.5113189884891853`*^9}, {
   3.511319022450445*^9, 3.5113191264246283`*^9}, {3.5113228881840353`*^9, 
   3.511322937589322*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Actuators", "Subsection",
 CellChangeTimes->{{3.5113201245923815`*^9, 3.511320127103986*^9}}],

Cell[CellGroupData[{

Cell["Positions", "Subsubsection",
 CellChangeTimes->{{3.5113204481369495`*^9, 3.511320449650152*^9}}],

Cell[CellGroupData[{

Cell["Values", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113202091757298`*^9, 3.511320210205332*^9}, 
   3.511320410462883*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"dxTMa", "=", 
    RowBox[{"12", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"x", "-", 
    RowBox[{
    "separation", " ", "of", " ", "actuators", " ", "at", " ", "TM"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dyTMa", "=", 
    RowBox[{"12", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"y", "-", 
    RowBox[{
    "separation", " ", "of", " ", "actators", " ", "at", " ", "TM"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dzTMa", "=", 
    RowBox[{"5", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{
    RowBox[{
     RowBox[{"z", "-", 
      RowBox[{
      "separation", " ", "between", " ", "COM", " ", "of", " ", "TM"}]}], " ",
      "&"}], " ", "actuators"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dzRMa", "=", 
    RowBox[{"6", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{
    RowBox[{
     RowBox[{"z", "-", 
      RowBox[{
      "separation", " ", "between", " ", "COM", " ", "of", " ", "RM"}]}], " ",
      "&"}], " ", "actuators"}], "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113202124517355`*^9, 3.5113202217649517`*^9}, {
  3.51132026614703*^9, 3.511320410462883*^9}, {3.5113204870590177`*^9, 
  3.5113205365111046`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Position of actuators in local coordinate", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113204715057907`*^9, 3.511320480023405*^9}, 
   3.511320711652612*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"posTMa", "=", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"-", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzTMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"+", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzTMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"+", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"+", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzTMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"+", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"-", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzTMa"}]}], "}"}]}], "\[IndentingNewLine]", "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"posRMa", "=", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"-", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzRMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"+", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzRMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"+", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"+", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzRMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"+", "dxTMa"}], "/", "2"}], ",", 
       RowBox[{
        RowBox[{"-", "dyTMa"}], "/", "2"}], ",", 
       RowBox[{"-", "dzRMa"}]}], "}"}]}], "\[IndentingNewLine]", "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511320549069127*^9, 3.5113207534450855`*^9}, {
  3.5113912026046133`*^9, 3.5113912502938967`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Working Direction", "Subsubsection",
 CellChangeTimes->{{3.511320420556101*^9, 3.5113204221161036`*^9}, {
  3.5113204602893705`*^9, 3.5113204617401733`*^9}}],

Cell[CellGroupData[{

Cell["Working direction in local coordinate system at recoil body.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511320810213585*^9, 3.511320827560816*^9}, 
   3.511323594038475*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"dirTMa", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "\[IndentingNewLine]", 
     "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511320767765911*^9, 3.5113207953779593`*^9}, {
   3.5113208372172327`*^9, 3.5113208425680423`*^9}, 3.511323594038475*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Body Shape", "Subsection",
 CellChangeTimes->{{3.511303371161437*^9, 3.5113033797726517`*^9}}],

Cell[CellGroupData[{

Cell["Discription", "Subsubsection",
 CellChangeTimes->{{3.5113191638334937`*^9, 3.5113191686227016`*^9}}],

Cell["\<\
This subsection contains the definition of the body appearance for eigen-mode \
3D graphics.
Containing nothing important for eigen-mode calculation or transfer function \
plots.
The 3D graphics shows not only standard bodies, but also mass-less bodies; \
the ground and key stones.
If you do not want to show them, please make the following symbols \"False\".\
\>", "Text",
 CellChangeTimes->{{3.51131927562329*^9, 3.511319695373227*^9}, {
  3.5113201748088694`*^9, 3.5113201768836727`*^9}, {3.51277273048518*^9, 
  3.51277274766081*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"showground", "=", "True"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"showkeystones", "=", "True"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.511319700568036*^9, 3.511319716386464*^9}, {
  3.5113208734716964`*^9, 3.511320878869306*^9}, {3.5113305238848543`*^9, 
  3.5113305240252547`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Shape Definition", "Subsubsection",
 CellChangeTimes->{{3.5113192607252636`*^9, 3.5113192614740653`*^9}, {
  3.511319726401682*^9, 3.5113197279772844`*^9}, {3.512708700551835*^9, 
  3.5127087296770864`*^9}}],

Cell[CellGroupData[{

Cell["Ground", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113059380991454`*^9, 3.5113059389259467`*^9}, 
   3.5113197356836977`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{"showground", ",", "\[IndentingNewLine]", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"shapeg", "=", "\"\<Doughnut\>\""}], ";", "\[IndentingNewLine]", 
      RowBox[{"rstage", "=", 
       RowBox[{"40", "cm"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"rgi", "=", 
       RowBox[{"rstage", "-", 
        RowBox[{"20", "cm"}]}]}], ";", 
      RowBox[{"(*", 
       RowBox[{"inner", " ", "radius"}], "*)"}], "\[IndentingNewLine]", 
      RowBox[{"rgo", "=", 
       RowBox[{"rstage", "+", 
        RowBox[{"20", "cm"}]}]}], ";", 
      RowBox[{"(*", 
       RowBox[{"outer", " ", "radius"}], "*)"}], "\[IndentingNewLine]", 
      RowBox[{"dimg", "=", 
       RowBox[{"{", 
        RowBox[{"rgi", ",", "rgo"}], "}"}]}], ";"}]}], "\[IndentingNewLine]", 
    "]"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113034020962915`*^9, 3.5113035097988806`*^9}, 
   3.5113035528549557`*^9, {3.511303899549965*^9, 3.5113039129503884`*^9}, {
   3.5113197356836977`*^9, 3.511319750269724*^9}, 3.511321098112091*^9, {
   3.512708410016925*^9, 3.5127084313577623`*^9}, {3.512774886069769*^9, 
   3.5127748932145815`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Key Stones", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.5113199123228083`*^9, 3.511319917002816*^9}, 
   3.511319954770483*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{"showkeystones", ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"shapeKS", "=", "\"\<yCylinder\>\""}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"rKS", "=", 
       RowBox[{"2", "cm"}]}], ";", 
      RowBox[{"(*", "radius", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"hKS", "=", 
       RowBox[{"1", "cm"}]}], ";", 
      RowBox[{"(*", "height", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"dimKS", "=", 
       RowBox[{"{", 
        RowBox[{"rKS", ",", "hKS"}], "}"}]}], ";"}]}], "\[IndentingNewLine]", 
    "]"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.511319920107222*^9, 3.5113199778741236`*^9}, 
   3.5113211007016954`*^9, {3.5127084353981695`*^9, 3.512708443666184*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Standard Filters", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511303652367531*^9, 3.5113036562051373`*^9}, 
   3.51131980525982*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"shapeGAS", "=", "\"\<TruncatedCone\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"ruGAS", "=", 
    RowBox[{"28", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"upper", "-", 
    RowBox[{"side", " ", "radius"}]}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"rlGAS", "=", 
    RowBox[{"37", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"lower", "-", 
    RowBox[{"side", " ", "radius"}]}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"hGAS", "=", 
    RowBox[{"20", "cm"}]}], ";"}], 
  RowBox[{"(*", "height", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dimGAS", "=", 
   RowBox[{"{", 
    RowBox[{"ruGAS", ",", "rlGAS", ",", "hGAS"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.5113036594499435`*^9, 3.511303700181615*^9}, {
   3.5113037338464737`*^9, 3.511303748338899*^9}, {3.511303780506156*^9, 
   3.511303810895009*^9}, {3.51130401070016*^9, 3.5113040321189976`*^9}, 
   3.51131980525982*^9, {3.512708527625531*^9, 3.5127085451443624`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Intermediate Mass", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.511303978642104*^9, 3.5113039899521236`*^9}, 
   3.511320006078973*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"shapeIM", "=", "\"\<Cuboid\>\""}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"sxIM", "=", 
    RowBox[{"60", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"x", "-", "size"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"syIM", "=", 
    RowBox[{"10", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"y", "-", "size"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"szIM", "=", 
    RowBox[{"60", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"z", "-", "size"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dimIM", "=", 
   RowBox[{"{", 
    RowBox[{"sxIM", ",", "syIM", ",", "szIM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.511303993103329*^9, 3.5113040028377466`*^9}, {
   3.5113040458002214`*^9, 3.5113040574690423`*^9}, {3.51130410174192*^9, 
   3.511304154002012*^9}, 3.511320006078973*^9, {3.5123558719185033`*^9, 
   3.512355874900674*^9}, {3.5127085520551744`*^9, 3.512708564379196*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Intermediate Recoil Mass", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10005.},
 CellChangeTimes->{{3.5113041616928253`*^9, 3.511304167308835*^9}, 
   3.51132001576659*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"shapeIRM", "=", "\"\<OpenCuboid\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"sxIRM", "=", 
    RowBox[{"70", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"x", "-", "size"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"syIRM", "=", 
    RowBox[{"15", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"y", "-", "size"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"szIRM", "=", 
    RowBox[{"70", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"z", "-", "size"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dimIRM", "=", 
   RowBox[{"{", 
    RowBox[{"sxIRM", ",", "syIRM", ",", "szIRM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10005.},
 CellChangeTimes->{{3.5113041691184387`*^9, 3.511304250488181*^9}, 
   3.51132001576659*^9, {3.5123558811200294`*^9, 3.512355883265152*^9}, {
   3.5127085691684046`*^9, 3.512708574753214*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Test Mass", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10006.},
 CellChangeTimes->{{3.5113042574613934`*^9, 3.5113042620946016`*^9}, {
  3.5113200209769993`*^9, 3.51132002713901*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"shapeTM", "=", "\"\<zCylinder\>\""}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"rTM", "=", 
    RowBox[{"12.5", "cm"}]}], ";"}], 
  RowBox[{"(*", "radius", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"lTM", "=", 
    RowBox[{"10", "cm"}]}], ";"}], 
  RowBox[{"(*", "length", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dimTM", "=", 
   RowBox[{"{", 
    RowBox[{"rTM", ",", "lTM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10006.},
 CellChangeTimes->{{3.5113042642630053`*^9, 3.511304330968723*^9}, 
   3.51132002713901*^9, {3.512708577545619*^9, 3.512708587061636*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Recoil Mass", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10007.},
 CellChangeTimes->{{3.5113043179427*^9, 3.5113043198459034`*^9}, 
   3.511320033597421*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"shapeRM", "=", "\"\<zCylinder\>\""}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"rRM", "=", 
    RowBox[{"27", "cm"}]}], ";"}], 
  RowBox[{"(*", "radius", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"lRM", "=", 
    RowBox[{"20", "cm"}]}], ";"}], 
  RowBox[{"(*", "length", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dimRM", "=", 
   RowBox[{"{", 
    RowBox[{"rRM", ",", "lRM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10007.},
 CellChangeTimes->{{3.511304322747508*^9, 3.511304366006384*^9}, 
   3.511320033597421*^9, 3.511321122916135*^9, {3.5123558431168556`*^9, 
   3.5123558445139356`*^9}, {3.5127085899632406`*^9, 
   3.5127086038784657`*^9}, {3.5127122763553176`*^9, 3.512712276870118*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Standard body shape & dimension", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10008.},
 CellChangeTimes->{{3.5127087372430997`*^9, 3.512708758178336*^9}, 
   3.5127091283357863`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"shape", "=", 
   RowBox[{"{", 
    RowBox[{"shapeIM", ",", "shapeIRM", ",", "shapeTM", ",", "shapeRM"}], 
    "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dim", "=", 
   RowBox[{"{", 
    RowBox[{"dimIM", ",", "dimIRM", ",", "dimTM", ",", "dimRM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10008.},
 CellChangeTimes->{{3.5127087466967163`*^9, 3.5127088545553055`*^9}, {
   3.5127089173610163`*^9, 3.512708972429113*^9}, 3.5127091283357863`*^9, {
   3.512774149093275*^9, 3.5127741538824835`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Variables", "Section",
 CellChangeTimes->{{3.5106252414511204`*^9, 3.5106252530107403`*^9}, {
   3.510625426015044*^9, 3.5106254276374474`*^9}, {3.5106259504567657`*^9, 
   3.510625951330367*^9}, 3.510911145254257*^9}],

Cell[CellGroupData[{

Cell["Variable List", "Subsection",
 CellChangeTimes->{{3.510626076005786*^9, 3.510626107814242*^9}, {
  3.510911153662672*^9, 3.5109111566890774`*^9}, {3.510911385167079*^9, 
  3.5109113901902876`*^9}}],

Cell[CellGroupData[{

Cell["Variables Specifying Each Body", "Subsubsection",
 CellChangeTimes->{{3.5109114063519163`*^9, 3.5109114182391367`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"varg", "=", 
    RowBox[{"{", 
     RowBox[{
     "xg", ",", "yg", ",", "zg", ",", "pitchg", ",", "yawg", ",", "rollg"}], 
     "}"}]}], ";"}], 
  RowBox[{"(*", "Ground", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"varIM", "=", 
    RowBox[{"{", 
     RowBox[{
     "xIM", ",", "yIM", ",", "zIM", ",", "pitchIM", ",", "yawIM", ",", 
      "rollIM"}], "}"}]}], ";", 
   RowBox[{"(*", 
    RowBox[{"Intermediate", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
   RowBox[{"varIRM", "=", 
    RowBox[{"{", 
     RowBox[{
     "xIRM", ",", "yIRM", ",", "zIRM", ",", "pitchIRM", ",", "yawIRM", ",", 
      "rollIRM"}], "}"}]}], ";", 
   RowBox[{"(*", 
    RowBox[{"Intermediate", " ", "Recoil", " ", "Mass"}], "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"varTM", "=", 
    RowBox[{"{", 
     RowBox[{
     "xTM", ",", "yTM", ",", "zTM", ",", "pitchTM", ",", "yawTM", ",", 
      "rollTM"}], "}"}]}], ";", 
   RowBox[{"(*", 
    RowBox[{"Test", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
   RowBox[{"varRM", "=", 
    RowBox[{"{", 
     RowBox[{
     "xRM", ",", "yRM", ",", "zRM", ",", "pitchRM", ",", "yawRM", ",", 
      "rollRM"}], "}"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Recoil", " ", "Mass"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"varGAS", "=", 
   RowBox[{"{", "qGAS4", "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"Key", " ", "Stones"}], "*)"}], "\[IndentingNewLine]"}]}], "Input",
 CellChangeTimes->CompressedData["
1:eJxTTMoPSmViYGAQAWIQfe6w/os/6q8cNbIdXoHoNt3tEh0arxyvHfghBaKP
VfFrgmgDUUUDEC00wyEURNtlbQDTJo8+ZW0G0t49ufkgWvFN/TYQ/ez8ETC9
R5I96gaQvmV6Jx5ECwneygPRW+5E5oPoB8kLC28C6XeF24tAtNuM4goQvYcl
rQlEM2zx3wCi/0ZqbwTRQV7VRz8B6W8fP4HpTv7aqyD6n8ZOMC0j/2jBHyBd
NOcFmPZT5bn3F0j/10t9AKLrVsV6/gPSbdFyXiC683RyN4hmm/gJTK9qepkr
p/nKUazTqwBEc/3UCwgA0lGceRkguqEo+YiozivHWd1xR0E0AALNoxg=
  "]]
}, Open  ]],

Cell[CellGroupData[{

Cell["All the Variables & Parameters", "Subsubsection",
 CellChangeTimes->{{3.510911424416748*^9, 3.5109114350559664`*^9}, {
  3.5109114803272457`*^9, 3.510911485444055*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"allparams", "=", 
    RowBox[{"Join", "[", "varg", "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"allvars", "=", 
   RowBox[{"Join", "[", 
    RowBox[{"varIM", ",", "varIRM", ",", "varTM", ",", "varRM"}], "]"}]}], 
  ";", 
  RowBox[{"(*", "Variables", "*)"}], "\[IndentingNewLine]", 
  RowBox[{"allfloats", "=", 
   RowBox[{"Join", "[", "varGAS", "]"}]}], ";", 
  RowBox[{"(*", "Floats", "*)"}], "\[IndentingNewLine]"}]}], "Input",
 CellChangeTimes->{{3.5105436285940065`*^9, 3.5105436779524927`*^9}, {
   3.5105505824294252`*^9, 3.510550583427827*^9}, {3.5105609245649977`*^9, 
   3.5105609275602026`*^9}, {3.5106256594538546`*^9, 
   3.5106256981731224`*^9}, {3.51062573105798*^9, 3.5106257692624474`*^9}, {
   3.510626150183916*^9, 3.5106261590603323`*^9}, {3.5106556411248803`*^9, 
   3.510655642092082*^9}, {3.5109113733266582`*^9, 3.510911378552667*^9}, {
   3.5109114473799877`*^9, 3.51091146552282*^9}, {3.5109116508113503`*^9, 
   3.5109116689697824`*^9}, {3.5109119407846594`*^9, 
   3.5109119478358717`*^9}, {3.510999937866187*^9, 3.510999943482197*^9}, {
   3.511075588996784*^9, 3.5110755901199865`*^9}, {3.511075685716954*^9, 
   3.511075686060155*^9}, {3.5110777999846897`*^9, 3.511077809531906*^9}, {
   3.5110778570807896`*^9, 3.511077864069602*^9}, {3.511077912663687*^9, 
   3.51107791422369*^9}, {3.5110782844591403`*^9, 3.5110782845839405`*^9}, {
   3.5111296569725723`*^9, 3.5111296786588125`*^9}, {3.511129758521381*^9, 
   3.5111297593664293`*^9}, {3.5111498442951202`*^9, 
   3.5111498693955646`*^9}, {3.5111523947904005`*^9, 
   3.5111523988152075`*^9}, {3.511153235678677*^9, 3.5111532386738825`*^9}, {
   3.511153792901862*^9, 3.5111537970826693`*^9}, {3.5112190816924467`*^9, 
   3.5112190848436522`*^9}, {3.5113214435902977`*^9, 
   3.5113214588627243`*^9}, {3.511321620947009*^9, 3.5113216299794254`*^9}, 
   3.5113217686324687`*^9, {3.5127741928045516`*^9, 3.512774194177354*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Velocities", "Subsubsection",
 CellChangeTimes->{{3.5109115271741285`*^9, 3.5109115296233325`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"velparams", "=", 
    RowBox[{"makevel", "[", "allparams", "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"velvars", "=", 
    RowBox[{"makevel", "[", "allvars", "]"}]}], ";"}], 
  RowBox[{"(*", "Variables", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"velfloats", "=", 
    RowBox[{"makevel", "[", "allfloats", "]"}]}], ";"}], 
  RowBox[{"(*", "Floats", "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5109116827914066`*^9, 3.510911693009424*^9}, {
  3.5109118265768595`*^9, 3.510911843393689*^9}, {3.5109119242330303`*^9, 
  3.5109119576326895`*^9}, {3.511129681916999*^9, 3.511129693044636*^9}, {
  3.511129762019581*^9, 3.5111297628596287`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Initial Values", "Subsection",
 CellChangeTimes->{{3.510560357878402*^9, 3.5105603606396074`*^9}, {
  3.510625979738017*^9, 3.5106259827956223`*^9}, {3.510911978505526*^9, 
  3.510911979644328*^9}}],

Cell[CellGroupData[{

Cell["Initial Position & Orientation of Each Body", "Subsubsection",
 CellChangeTimes->{{3.51091199199955*^9, 3.510912024151206*^9}, {
  3.5113222648381405`*^9, 3.5113222658053417`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"lw", "=", 
    RowBox[{"2.1", "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"typical", " ", "length", " ", "of", " ", "wires"}], 
   "*)"}]}]], "Input",
 CellChangeTimes->{{3.5113219241803417`*^9, 3.511321944241977*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"initg", "=", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}]}],
    ";"}], 
  RowBox[{"(*", "Ground", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"initIM", "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{
      RowBox[{"-", "1"}], "lw"}], ",", "0", ",", "0", ",", "0", ",", "0"}], 
    "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"Intermediate", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
  RowBox[{"initIRM", "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{
      RowBox[{"-", "1"}], "lw"}], ",", "0", ",", "0", ",", "0", ",", "0"}], 
    "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"Intermediate", " ", "Recoil", " ", "Mass"}], "*)"}], 
  "\[IndentingNewLine]", 
  RowBox[{"initTM", "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{
      RowBox[{"-", "2"}], "lw"}], ",", "0", ",", "0", ",", "0", ",", "0"}], 
    "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"Test", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
  RowBox[{"initRM", "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{
      RowBox[{"-", "2"}], "lw"}], ",", "0", ",", "0", ",", "0", ",", "0"}], 
    "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"Recoil", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
  RowBox[{"initGAS", "=", 
   RowBox[{"{", "0", "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"Key", " ", "Stone"}], "*)"}], "\[IndentingNewLine]"}]}], "Input",
 CellChangeTimes->{{3.510542002925949*^9, 3.5105420494920306`*^9}, {
   3.5105436066915674`*^9, 3.510543608516771*^9}, {3.510543698622529*^9, 
   3.5105437015397344`*^9}, {3.51055031780616*^9, 3.510550319256963*^9}, {
   3.5106259640599895`*^9, 3.510625996414446*^9}, {3.510912034182024*^9, 
   3.510912037130429*^9}, {3.510912170073862*^9, 3.5109122070927277`*^9}, {
   3.5109999557906184`*^9, 3.510999980454262*^9}, {3.511075660164109*^9, 
   3.511075679258543*^9}, {3.5110757102869973`*^9, 3.511075712408601*^9}, {
   3.5110779207601013`*^9, 3.5110779389341335`*^9}, 3.511078289373149*^9, {
   3.5111296990169773`*^9, 3.511129702435173*^9}, {3.5111297677139063`*^9, 
   3.5111297678259125`*^9}, {3.5111524153668365`*^9, 3.511152441465682*^9}, {
   3.5111532486735*^9, 3.5111532518247056`*^9}, {3.5111537772238345`*^9, 
   3.5111537806558404`*^9}, {3.511219101894482*^9, 3.511219113048502*^9}, {
   3.5112290480209055`*^9, 3.5112290640733337`*^9}, {3.511231809771756*^9, 
   3.5112318099277563`*^9}, {3.511321800238124*^9, 3.511321902168703*^9}, {
   3.5113219541479945`*^9, 3.511322052240967*^9}, {3.512774222678604*^9, 
   3.512774234550225*^9}, {3.5127748105656366`*^9, 3.5127748144188433`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Default Values of Parameters & Initial Values of Variables and Floats\
\>", "Subsubsection",
 CellChangeTimes->{{3.5109123598637958`*^9, 3.5109123642786036`*^9}, {
  3.510912508001656*^9, 3.510912540184512*^9}, {3.5113222758205595`*^9, 
  3.511322278378964*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"defparams", "=", 
    RowBox[{"Join", "[", "initg", "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"initvars", "=", 
   RowBox[{"Join", "[", 
    RowBox[{"initIM", ",", "initIRM", ",", "initTM", ",", "initRM"}], "]"}]}],
   ";", 
  RowBox[{"(*", "Variables", "*)"}], "\[IndentingNewLine]", 
  RowBox[{"initfloats", "=", 
   RowBox[{"Join", "[", "initGAS", "]"}]}], ";", 
  RowBox[{"(*", "Floats", "*)"}], "\[IndentingNewLine]"}]}], "Input",
 CellChangeTimes->{{3.510912583599389*^9, 3.510912631117072*^9}, {
  3.5109188304827647`*^9, 3.5109188378615775`*^9}, {3.5109999852746706`*^9, 
  3.5109999912026806`*^9}, {3.5110757227670193`*^9, 3.511075724280222*^9}, {
  3.511077968542986*^9, 3.511077968683386*^9}, {3.5111297121007257`*^9, 
  3.5111297534580913`*^9}, {3.511149882561988*^9, 3.5111498881779976`*^9}, {
  3.5111532547107105`*^9, 3.5111532656775303`*^9}, {3.51115380350988*^9, 
  3.5111538083926888`*^9}, {3.511219133266137*^9, 3.511219137290944*^9}, {
  3.511322074003005*^9, 3.5113220940958405`*^9}, {3.5127742447214427`*^9, 
  3.512774245361044*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Substitution", "Subsubsection",
 CellChangeTimes->{{3.510913689415735*^9, 3.510913714516179*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"subdefparams", "=", 
    RowBox[{"makesub", "[", 
     RowBox[{"allparams", ",", "defparams"}], "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"subinitvars", "=", 
    RowBox[{"makesub", "[", 
     RowBox[{"allvars", ",", "initvars"}], "]"}]}], ";"}], 
  RowBox[{"(*", "Variables", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"subinitfloats", "=", 
    RowBox[{"makesub", "[", 
     RowBox[{"allvars", ",", "initvars"}], "]"}]}], ";"}], 
  RowBox[{"(*", "Floats", "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5109137313954086`*^9, 3.510913798990327*^9}, {
  3.5111524709497337`*^9, 3.511152475504942*^9}, {3.5111532804195557`*^9, 
  3.511153306814802*^9}, {3.5113221093838673`*^9, 3.5113221168250804`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Position of Bodies", "Subsection",
 CellChangeTimes->{{3.5111299640521364`*^9, 3.5111299739127007`*^9}, {
   3.511215303957964*^9, 3.5112153043635645`*^9}, 3.511219171330204*^9, {
   3.511322411587598*^9, 3.511322413272401*^9}}],

Cell[CellGroupData[{

Cell["Standard Bodies", "Subsubsection",
 CellChangeTimes->{{3.5113240301528406`*^9, 3.5113240329452457`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"posg", "=", "varg"}], ";", 
   RowBox[{"posIM", "=", "varIM"}], ";", 
   RowBox[{"posIRM", "=", "varIRM"}], ";", 
   RowBox[{"posTM", "=", "varTM"}], ";", 
   RowBox[{"posRM", "=", "varRM"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5110785212363563`*^9, 3.5110785817020626`*^9}, {
   3.511078660653801*^9, 3.5110786610750017`*^9}, {3.5110786937726593`*^9, 
   3.511078720760707*^9}, {3.511078819072079*^9, 3.5110788518009367`*^9}, 
   3.511129970563509*^9, {3.5111345040973763`*^9, 3.5111345247986126`*^9}, {
   3.511139023940115*^9, 3.5111390369505377`*^9}, {3.5111411593030653`*^9, 
   3.5111411653558764`*^9}, {3.5111478027663345`*^9, 
   3.5111478096615467`*^9}, {3.5111479049621143`*^9, 
   3.5111479063817167`*^9}, {3.5111506315505033`*^9, 3.511150631644103*^9}, {
   3.5111539126944723`*^9, 3.5111539127880726`*^9}, {3.5112153117735777`*^9, 
   3.51121536986808*^9}, {3.51121718335091*^9, 3.511217211243759*^9}, {
   3.5112191753238115`*^9, 3.5112192073038673`*^9}, {3.511219240344725*^9, 
   3.5112193137740545`*^9}, {3.5112199675092163`*^9, 3.511220015697701*^9}, {
   3.5112200468197556`*^9, 3.5112200990954475`*^9}, {3.511220139171918*^9, 
   3.5112201564879484`*^9}, {3.511223344655672*^9, 3.5112233853405437`*^9}, {
   3.5112234306586227`*^9, 3.511223499641944*^9}, 3.511223584864894*^9, {
   3.511223625081764*^9, 3.5112238683953915`*^9}, {3.511223913994272*^9, 
   3.511223934586308*^9}, {3.511224317972582*^9, 3.5112243645854635`*^9}, {
   3.511224438295593*^9, 3.5112245600694065`*^9}, {3.511224590973061*^9, 
   3.511224591238261*^9}, 3.511225501827461*^9, {3.5112257523951006`*^9, 
   3.5112257752959414`*^9}, {3.511225857430085*^9, 3.5112258621412935`*^9}, {
   3.5112259063985715`*^9, 3.511225914682186*^9}, {3.511228887559024*^9, 
   3.5112288890878263`*^9}, {3.511228928821096*^9, 3.511228934031505*^9}, {
   3.511229587438653*^9, 3.5112295923682613`*^9}, {3.5112299221528406`*^9, 
   3.5112299244304447`*^9}, {3.5112302659306445`*^9, 3.511230266133445*^9}, {
   3.511230339812374*^9, 3.5112303404519753`*^9}, {3.511322430869232*^9, 
   3.511322506700965*^9}, {3.511322597196724*^9, 3.51132271461813*^9}, {
   3.51132295932016*^9, 3.5113230738555613`*^9}, {3.5113231260064526`*^9, 
   3.5113231777829437`*^9}, {3.511323242959858*^9, 3.51132324969907*^9}, {
   3.5113233113035784`*^9, 3.5113233186355915`*^9}, 3.511323521420347*^9, {
   3.511324086125739*^9, 3.5113241183397956`*^9}, {3.5113875058597193`*^9, 
   3.5113876400667553`*^9}, {3.5113876748080163`*^9, 
   3.5113877253989053`*^9}, {3.5113891555846176`*^9, 3.511389165412635*^9}, {
   3.5114067382148457`*^9, 3.5114067398216486`*^9}, {3.5114068871327076`*^9, 
   3.511406911905551*^9}, 3.512726595160699*^9, {3.5127742616630726`*^9, 
   3.512774262583474*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Key Stones", "Subsubsection",
 CellChangeTimes->{{3.5113240803693295`*^9, 3.511324082022932*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"posKS4", "=", 
    RowBox[{"Join", "[", 
     RowBox[{
      RowBox[{"b2s", "[", 
       RowBox[{"posg", ",", 
        RowBox[{
         RowBox[{"wpGAS", "[", 
          RowBox[{"[", "1", "]"}], "]"}], "+", 
         RowBox[{"qGAS4", " ", 
          RowBox[{"wdGAS", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}]}], "]"}], ",", 
      RowBox[{"posg", "[", 
       RowBox[{"[", 
        RowBox[{"4", ";;", "6"}], "]"}], "]"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5113240932393513`*^9, 3.511324108652179*^9}, {
  3.511332905806238*^9, 3.5113329127170506`*^9}, {3.512774269915487*^9, 
  3.512774274314695*^9}, {3.5127747569171424`*^9, 3.5127747605519485`*^9}}]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Energy Calculation", "Section",
 CellChangeTimes->{{3.5106262518492947`*^9, 3.5106262674337225`*^9}, {
  3.5109170152635765`*^9, 3.5109170473996325`*^9}}],

Cell[CellGroupData[{

Cell["Body Definition", "Subsection",
 CellChangeTimes->{{3.5109140124455023`*^9, 3.51091404542396*^9}, {
  3.5109169162814026`*^9, 3.5109169184810066`*^9}, {3.5109662739308863`*^9, 
  3.51096627628649*^9}}],

Cell[CellGroupData[{

Cell["About bodylist", "Subsubsection",
 CellChangeTimes->{{3.5127269557397323`*^9, 3.51272696006094*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "bodylist"}]], "Input",
 CellChangeTimes->{{3.5127269619485435`*^9, 3.5127269651465487`*^9}}],

Cell[BoxData[
 StyleBox["\<\"bodylist is a list of structures defining bodies. Each \
structure containing the following items:\\n\tcoordinate list defining the \
body,\\n\tmass,\\n\tmoment of inertia tensor,\\n\tshape,\\n\tdimension\"\>", 
  "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.512774910156212*^9},
 CellTags->"Info3512807310-4917596"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Standard Bodies", "Subsubsection",
 CellChangeTimes->{{3.5113252072217083`*^9, 3.511325209982913*^9}}],

Cell[CellGroupData[{

Cell["Position List", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.512709373490217*^9, 3.5127093777334247`*^9}, 
   3.512709433129122*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"posb", "=", 
    RowBox[{"{", 
     RowBox[{"posIM", ",", "posIRM", ",", "posTM", ",", "posRM"}], "}"}]}], 
   ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127093809626303`*^9, 3.512709433129122*^9}, {
  3.512774305405549*^9, 3.5127743064663515`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Make bodylist", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113245884622216`*^9, 3.511324596262235*^9}, {
  3.511324730375671*^9, 3.5113247350088787`*^9}, {3.512709437309929*^9, 
  3.512709445343943*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"bodylist", "=", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"posb", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", 
        RowBox[{"mass", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"moix", "[", 
             RowBox[{"[", "n", "]"}], "]"}], ",", "0", ",", "0"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", 
            RowBox[{"moiy", "[", 
             RowBox[{"[", "n", "]"}], "]"}], ",", "0"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "0", ",", 
            RowBox[{"moiz", "[", 
             RowBox[{"[", "n", "]"}], "]"}]}], "}"}]}], "}"}], ",", 
        RowBox[{"(*", 
         RowBox[{"moment", " ", "of", " ", "inertia"}], "*)"}], 
        "\[IndentingNewLine]", 
        RowBox[{"shape", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", 
        RowBox[{"(*", 
         RowBox[{"type", " ", "of", " ", "shape"}], "*)"}], 
        RowBox[{"dim", "[", 
         RowBox[{"[", "n", "]"}], "]"}]}], 
       RowBox[{"(*", 
        RowBox[{"size", " ", "information"}], "*)"}], "\[IndentingNewLine]", 
       "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", 
        RowBox[{"Length", "[", "mass", "]"}]}], "}"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5109140609147873`*^9, 3.5109140779968176`*^9}, {
   3.5109141081360703`*^9, 3.5109142185842648`*^9}, {3.5109143213572445`*^9, 
   3.5109145997085333`*^9}, 3.510917211730322*^9, {3.5109176320262594`*^9, 
   3.5109176547555*^9}, 3.510917743488456*^9, {3.5109662907321157`*^9, 
   3.510966396843502*^9}, {3.510967091044721*^9, 3.510967168249257*^9}, {
   3.5109955112540817`*^9, 3.5109955697864294`*^9}, {3.510997044967098*^9, 
   3.5109970462151003`*^9}, {3.5110002784771852`*^9, 
   3.5110003227344627`*^9}, {3.511000359144927*^9, 3.5110003704549465`*^9}, {
   3.511078476823078*^9, 3.511078512422341*^9}, 3.5111541017668047`*^9, {
   3.5112152637410936`*^9, 3.511215297234352*^9}, {3.511217240369011*^9, 
   3.511217254159435*^9}, {3.5112246532795706`*^9, 3.5112248007154293`*^9}, {
   3.51122485224232*^9, 3.5112249190416374`*^9}, {3.5112249915817647`*^9, 
   3.511225023593021*^9}, {3.5112253387915745`*^9, 3.5112253422859807`*^9}, {
   3.5112260225719757`*^9, 3.5112260227123756`*^9}, {3.511226094581702*^9, 
   3.5112260955489035`*^9}, {3.5113242138899636`*^9, 3.511324447937175*^9}, 
   3.511324596262235*^9, {3.511324730375671*^9, 3.5113247350088787`*^9}, 
   3.5113247888913736`*^9, {3.5127093008876896`*^9, 3.5127093655186033`*^9}, {
   3.5127094492907505`*^9, 3.5127095235312805`*^9}, {3.512710355012741*^9, 
   3.512710367024762*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Others", "Subsubsection",
 CellChangeTimes->{{3.5113252238201375`*^9, 3.5113252269869432`*^9}}],

Cell[CellGroupData[{

Cell["Ground", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511325241245368*^9, 3.511325249388582*^9}, 
   3.5113253530819645`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{"showground", ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"AppendTo", "[", 
       RowBox[{"bodylist", ",", 
        RowBox[{"{", "\[IndentingNewLine]", 
         RowBox[{"posg", ",", 
          RowBox[{"(*", "coordinate", "*)"}], "0", ",", 
          RowBox[{"(*", "mass", "*)"}], "\[IndentingNewLine]", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "}"}], ",", 
          RowBox[{"(*", 
           RowBox[{"moment", " ", "of", " ", "inertia"}], "*)"}], 
          "\[IndentingNewLine]", "shapeg", ",", 
          RowBox[{"(*", 
           RowBox[{"type", " ", "of", " ", "shape"}], "*)"}], "dimg"}], 
         RowBox[{"(*", 
          RowBox[{"size", " ", "information"}], "*)"}], "}"}]}], "]"}], 
      ";"}]}], "\[IndentingNewLine]", "]"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113252534913893`*^9, 3.5113253530819645`*^9}, {
  3.512709563763751*^9, 3.5127095645437527`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Key Stones", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511325241245368*^9, 3.511325249388582*^9}, {
   3.5113253530819645`*^9, 3.5113253677459903`*^9}, 3.5113254448569255`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{"showkeystones", ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"posKS", "=", 
       RowBox[{"{", "posKS4", "}"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"bodylist", "=", 
       RowBox[{"Join", "[", 
        RowBox[{"bodylist", ",", "\[IndentingNewLine]", 
         RowBox[{"Table", "[", "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{"posKS", "[", 
              RowBox[{"[", "n", "]"}], "]"}], ",", 
             RowBox[{"(*", "coordinate", "*)"}], "0", ",", 
             RowBox[{"(*", "mass", "*)"}], "\[IndentingNewLine]", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"{", 
                RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
               RowBox[{"{", 
                RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
               RowBox[{"{", 
                RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "}"}], ",", 
             RowBox[{"(*", 
              RowBox[{"moment", " ", "of", " ", "inertia"}], "*)"}], 
             "\[IndentingNewLine]", "shapeKS", ",", 
             RowBox[{"(*", 
              RowBox[{"type", " ", "of", " ", "shape"}], "*)"}], "dimKS"}], 
            RowBox[{"(*", 
             RowBox[{"size", " ", "information"}], "*)"}], "}"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"{", 
            RowBox[{"n", ",", 
             RowBox[{"Length", "[", "posKS", "]"}]}], "}"}]}], 
          "\[IndentingNewLine]", "]"}]}], "\[IndentingNewLine]", "]"}]}], 
      ";"}]}], "\[IndentingNewLine]", "]"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113252534913893`*^9, 3.5113254448569255`*^9}, {
   3.511325483295393*^9, 3.5113254856821976`*^9}, {3.5127096271154623`*^9, 
   3.512709710482009*^9}, {3.512726680758049*^9, 3.51272668116365*^9}, 
   3.512774329164391*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Wire Definition", "Subsection",
 CellChangeTimes->{{3.5109140124455023`*^9, 3.51091404542396*^9}, {
  3.5109169162814026`*^9, 3.5109169184810066`*^9}}],

Cell[CellGroupData[{

Cell["About wirelist", "Subsubsection",
 CellChangeTimes->{{3.512726995441802*^9, 3.51272700010621*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "wirelist"}]], "Input",
 CellChangeTimes->{{3.5127270107922287`*^9, 3.512727011993431*^9}}],

Cell[BoxData[
 StyleBox["\<\"wirelist is a list of structures defining wires. Each \
structure containing the following items:\\n\tcoordinate list defining first \
mass,\\n\tattachment point for first mass (local coordinates),\\n\tcoordinate \
list defining second mass,\\n\tattachment point for second mass (local \
coordinates),\\n\tunstretched length,\\n\tlongitudinal elasticity,\\n\tloss \
angle of the material,\\n\tshear modulus of the material,\\n\tsecond moment \
of area\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.5127749102498116`*^9},
 CellTags->"Info3512807310-2061953"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making wirelist", "Subsubsection",
 CellChangeTimes->{{3.5127270034446163`*^9, 3.51272700581582*^9}}],

Cell[CellGroupData[{

Cell["Coordinate list of wire-attaching bodies", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113271026756425`*^9, 3.5113271134708614`*^9}, {
   3.511327177774174*^9, 3.5113271910185976`*^9}, 3.511327439480234*^9, 
   3.51271000708533*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"posub", "=", 
   RowBox[{"{", 
    RowBox[{"posKS4", ",", "posg", ",", "posIM", ",", "posIM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"poslb", "=", 
   RowBox[{"{", 
    RowBox[{"posIM", ",", "posIRM", ",", "posTM", ",", "posRM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511327117714069*^9, 3.5113272142158384`*^9}, 
   3.511327439480234*^9, {3.512710003122923*^9, 3.51271000708533*^9}, {
   3.5127743561992383`*^9, 3.512774359724845*^9}, {3.5127746732229953`*^9, 
   3.512774673488196*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making wirelist", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113264636323147`*^9, 3.511326468936324*^9}, 
   3.511326909453703*^9, {3.5113272309858675`*^9, 3.51132723226507*^9}, {
   3.5113274460322456`*^9, 3.5113274471086473`*^9}, {3.5113277295939436`*^9, 
   3.5113277327919493`*^9}, {3.5127270146298356`*^9, 3.512727014754636*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"wirelist", "=", "\[IndentingNewLine]", 
    RowBox[{"Flatten", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"Table", "[", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"posub", "[", 
           RowBox[{"[", "m", "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "coordinate", " ", "list", " ", "defining", " ", "upper", " ", 
            "body"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{"spu", "[", 
           RowBox[{"[", 
            RowBox[{"m", ",", "n"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "attachment", " ", "point", " ", "at", " ", "upper", " ", "body"}],
            "*)"}], "\[IndentingNewLine]", 
          RowBox[{"poslb", "[", 
           RowBox[{"[", "m", "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "coordinate", " ", "list", " ", "defining", " ", "lower", " ", 
            "body"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{"spl", "[", 
           RowBox[{"[", 
            RowBox[{"m", ",", "n"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "attachment", " ", "point", " ", "at", " ", "lower", " ", "body"}],
            "*)"}], "\[IndentingNewLine]", 
          RowBox[{"lNw", "[", 
           RowBox[{"[", 
            RowBox[{"m", ",", "n"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{"Natural", " ", "Length"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"N", "[", "Pi", "]"}], 
           SuperscriptBox[
            RowBox[{"dw", "[", 
             RowBox[{"[", 
              RowBox[{"m", ",", "n"}], "]"}], "]"}], "2"], 
           RowBox[{
            RowBox[{
             RowBox[{"young", "[", 
              RowBox[{"matw", "[", 
               RowBox[{"[", "m", "]"}], "]"}], "]"}], "/", 
             RowBox[{"lNw", "[", 
              RowBox[{"[", 
               RowBox[{"m", ",", "n"}], "]"}], "]"}]}], "/", "4"}]}], ",", 
          RowBox[{"(*", "Stiffness", "*)"}], "\[IndentingNewLine]", 
          RowBox[{"phi", "[", 
           RowBox[{"matw", "[", 
            RowBox[{"[", "m", "]"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{"loss", " ", "angle"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{
            RowBox[{"young", "[", 
             RowBox[{"matw", "[", 
              RowBox[{"[", "m", "]"}], "]"}], "]"}], "/", "2"}], "/", 
           RowBox[{"(", 
            RowBox[{"1", "+", 
             RowBox[{"poisson", "[", 
              RowBox[{"matw", "[", 
               RowBox[{"[", "m", "]"}], "]"}], "]"}]}], ")"}]}], ",", 
          RowBox[{"(*", 
           RowBox[{"shear", " ", "modulus"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"N", "[", "Pi", "]"}], 
           RowBox[{
            SuperscriptBox[
             RowBox[{"dw", "[", 
              RowBox[{"[", 
               RowBox[{"m", ",", "n"}], "]"}], "]"}], "4"], "/", "32"}]}]}], 
         "}"}], ",", 
        RowBox[{"(*", 
         RowBox[{"second", " ", "moment", " ", "of", " ", "area"}], "*)"}], 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "nw", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"nw", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "}"}]}], "\[IndentingNewLine]", 
       "]"}], ",", "1"}], "\[IndentingNewLine]", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109140609147873`*^9, 3.5109140779968176`*^9}, {
   3.5109141081360703`*^9, 3.5109142185842648`*^9}, {3.5109143213572445`*^9, 
   3.5109145997085333`*^9}, 3.510917211730322*^9, {3.5109176320262594`*^9, 
   3.5109176547555*^9}, 3.510917743488456*^9, {3.511000024399539*^9, 
   3.5110000885468516`*^9}, {3.5110001243021145`*^9, 3.51100016160178*^9}, {
   3.5110001988390455`*^9, 3.5110002427687225`*^9}, {3.511001607649932*^9, 
   3.511001780966236*^9}, {3.5111538584687767`*^9, 3.511153869965997*^9}, {
   3.511155069015303*^9, 3.511155069842105*^9}, {3.5112246193027105`*^9, 
   3.511224643903954*^9}, {3.5112258457456646`*^9, 3.5112258461824656`*^9}, {
   3.5112259909039197`*^9, 3.511226005833146*^9}, {3.5113264112630234`*^9, 
   3.5113264144922285`*^9}, {3.5113264759095364`*^9, 
   3.5113266364338183`*^9}, {3.5113267140789595`*^9, 3.511326734109395*^9}, {
   3.511326898596084*^9, 3.511326924086529*^9}, {3.511327235587876*^9, 
   3.5113272773335495`*^9}, {3.511327366737306*^9, 3.511327419044198*^9}, {
   3.5113275248903837`*^9, 3.5113275334391985`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Spring Definition", "Subsection",
 CellChangeTimes->{{3.5109140124455023`*^9, 3.51091404542396*^9}, {
  3.5109169162814026`*^9, 3.5109169184810066`*^9}, {3.5109662739308863`*^9, 
  3.51096627628649*^9}, {3.5110788859025965`*^9, 3.5110788867761984`*^9}}],

Cell[CellGroupData[{

Cell["About springlist", "Subsubsection",
 CellChangeTimes->{{3.5127270198558445`*^9, 3.512727025378255*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "springlist"}]], "Input",
 CellChangeTimes->{{3.5127270268134575`*^9, 3.5127270288258605`*^9}}],

Cell[BoxData[
 StyleBox["\<\"springlist is a list of structures defining GAS springs. Each \
structure containing the following items:\\n\tcoordinate list defining first \
mass,\\n\tworking point at first mass (local coordinate)\\n\tcoordinate list \
defining second mass,\\n\tattached point at second mass (local coordinate)\\n\
\tspring constant tensor,\\n\tstoring force element vector,\\n\tspring \
constant tensor with imaginary part\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.5127749103434114`*^9},
 CellTags->"Info3512807310-4632052"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making springlist", "Subsubsection",
 CellChangeTimes->{{3.512727031540265*^9, 3.51272703420787*^9}}],

Cell[CellGroupData[{

Cell["Coordinate list of spring-attaching bodies", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113271026756425`*^9, 3.5113271134708614`*^9}, {
   3.511327177774174*^9, 3.5113271910185976`*^9}, 3.511327439480234*^9, {
   3.511327661250223*^9, 3.511327663059827*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"poss1", "=", 
   RowBox[{"{", "posg", "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"poss2", "=", 
   RowBox[{"{", "posKS4", "}"}]}], ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511327117714069*^9, 3.5113272142158384`*^9}, 
   3.511327439480234*^9, {3.5113276571162157`*^9, 3.5113277233539324`*^9}, 
   3.5113277842408395`*^9, {3.512774376869275*^9, 3.512774378460478*^9}, {
   3.5127746955934343`*^9, 3.5127746958430347`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making springlist", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511327747565175*^9, 3.5113277509503813`*^9}, 
   3.5113280529669113`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"springlist", "=", "\[IndentingNewLine]", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"(*", "GAS", "*)"}], "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"poss1", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", 
        RowBox[{"(*", 
         RowBox[{
         "coordinate", " ", "list", " ", "defining", " ", "first", " ", 
          "body"}], "*)"}], "\[IndentingNewLine]", 
        RowBox[{"wpGAS", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", 
        RowBox[{"(*", 
         RowBox[{"working", " ", "point", " ", 
          RowBox[{"(", 
           RowBox[{"first", " ", "body", " ", "coordinate"}], ")"}]}], "*)"}],
         "\[IndentingNewLine]", 
        RowBox[{"poss2", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", 
        RowBox[{"(*", 
         RowBox[{
         "coordinate", " ", "list", " ", "defining", " ", "second", " ", 
          "body"}], "*)"}], "\[IndentingNewLine]", "posCOM", ",", 
        RowBox[{"(*", 
         RowBox[{"attachment", " ", "point", " ", 
          RowBox[{"(", 
           RowBox[{"second", " ", "body", " ", "coordinate"}], ")"}]}], 
         "*)"}], "\[IndentingNewLine]", 
        RowBox[{"matkGAS", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", 
        RowBox[{"(*", 
         RowBox[{"spring", " ", "constant", " ", "tensor"}], "*)"}], 
        "\[IndentingNewLine]", 
        RowBox[{"vecfGAS", "[", 
         RowBox[{"[", "n", "]"}], "]"}], ",", 
        RowBox[{"(*", 
         RowBox[{"storing", " ", "force", " ", "vector"}], "*)"}], 
        "\[IndentingNewLine]", 
        RowBox[{"matkGASc", "[", 
         RowBox[{"[", "n", "]"}], "]"}]}], 
       RowBox[{"(*", 
        RowBox[{
        "spring", " ", "constant", " ", "tensor", " ", "with", " ", 
         "imaginary", " ", "part"}], "*)"}], "\[IndentingNewLine]", "}"}], 
      ",", 
      RowBox[{"{", 
       RowBox[{"n", ",", 
        RowBox[{"Length", "[", "poss1", "]"}]}], "}"}]}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5109140609147873`*^9, 3.5109140779968176`*^9}, {
   3.5109141081360703`*^9, 3.5109142185842648`*^9}, {3.5109143213572445`*^9, 
   3.5109145997085333`*^9}, 3.510917211730322*^9, {3.5109176320262594`*^9, 
   3.5109176547555*^9}, 3.510917743488456*^9, {3.5109662907321157`*^9, 
   3.510966396843502*^9}, {3.510967091044721*^9, 3.510967168249257*^9}, {
   3.5109955112540817`*^9, 3.5109955697864294`*^9}, {3.510997044967098*^9, 
   3.5109970462151003`*^9}, {3.5110002784771852`*^9, 
   3.5110003227344627`*^9}, {3.511000359144927*^9, 3.5110003704549465`*^9}, {
   3.511078476823078*^9, 3.511078512422341*^9}, {3.511078893874211*^9, 
   3.5110789163538504`*^9}, {3.5111301717180142`*^9, 
   3.5111302177266455`*^9}, {3.511130264499321*^9, 3.5111303154552355`*^9}, {
   3.5111303524563518`*^9, 3.511130387641364*^9}, {3.5111305329186735`*^9, 
   3.5111305565030227`*^9}, {3.5111306659992857`*^9, 
   3.5111307500230913`*^9}, {3.5111310998197966`*^9, 3.51113113022425*^9}, 
   3.511132179263693*^9, {3.511134281157385*^9, 3.511134313043841*^9}, {
   3.511134822821536*^9, 3.511134849169983*^9}, 3.51113914042552*^9, 
   3.5111393361746635`*^9, {3.51113936804552*^9, 3.5111393805567417`*^9}, {
   3.5111525121650066`*^9, 3.5111525123522067`*^9}, {3.511154131282056*^9, 
   3.51115413343486*^9}, {3.5112291050234056`*^9, 3.511229108252611*^9}, {
   3.511327629769368*^9, 3.5113276327957735`*^9}, {3.5113277552715883`*^9, 
   3.511327803865674*^9}, {3.51132798381199*^9, 3.511327991175203*^9}, {
   3.511328024902462*^9, 3.5113280529669113`*^9}, {3.512726923057675*^9, 
   3.5127269360212975`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Damper Definition", "Subsection",
 CellChangeTimes->{{3.511388575091998*^9, 3.51138859902244*^9}}],

Cell[CellGroupData[{

Cell["About damperlist", "Subsubsection",
 CellChangeTimes->{{3.512727052319502*^9, 3.5127270548623066`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "damperlist"}]], "Input",
 CellChangeTimes->{{3.5127270626779203`*^9, 3.5127270641287227`*^9}}],

Cell[BoxData[
 StyleBox["\<\"damperlist is is a list of structures defining dampers. Each \
structure containing the following items:\\n\tcoordinate list defining first \
mass,\\n\tcoordinate list defining second mass,\\n\tdamping coefficient \
matrix\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.512774910421412*^9},
 CellTags->"Info3512807310-2021627"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making damperlist", "Subsubsection",
 CellChangeTimes->{{3.5127270573115106`*^9, 3.5127270610087175`*^9}}],

Cell[CellGroupData[{

Cell["Making damperlist", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113885915032268`*^9, 3.5113885957776346`*^9}, 
   3.511388709876235*^9, {3.5127270675607285`*^9, 3.5127270704467335`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"damperlist", "=", 
    RowBox[{"{", "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511388612048463*^9, 3.511388709876235*^9}, {
  3.511389274160426*^9, 3.511389300805273*^9}, {3.5127104680973396`*^9, 
  3.5127104738693495`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Potential Energy Calculation", "Subsection",
 CellChangeTimes->{{3.510560448249361*^9, 3.510560448670562*^9}, {
  3.5106264014535575`*^9, 3.510626411702776*^9}, {3.5109166230008874`*^9, 
  3.510916633390506*^9}, {3.510917225723546*^9, 3.510917232306758*^9}}],

Cell[CellGroupData[{

Cell["Wire Stretching Potential", "Subsubsection",
 CellChangeTimes->{{3.511219454212905*^9, 3.511219459080114*^9}, {
  3.5113282965753393`*^9, 3.511328305264555*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"wirepot", "=", 
    RowBox[{"makewirepot", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"only", " ", "real", " ", "part"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"wirepotc", "=", 
    RowBox[{"makewirepotc", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"with", " ", "imaginary", " ", "part"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.510542091144104*^9, 3.510542093936509*^9}, {
   3.510542888481906*^9, 3.5105428899327087`*^9}, {3.510542932146383*^9, 
   3.5105429338311863`*^9}, {3.5105434500672927`*^9, 3.510543476493739*^9}, {
   3.510543521063017*^9, 3.5105435385974483`*^9}, {3.5105435839311275`*^9, 
   3.5105435894847374`*^9}, {3.510545463687629*^9, 3.5105454866664696`*^9}, {
   3.5105498942966166`*^9, 3.510550005088011*^9}, {3.5105505425245547`*^9, 
   3.5105505701990037`*^9}, {3.5105506566075554`*^9, 3.510550672535183*^9}, {
   3.5105509738653126`*^9, 3.5105509743645134`*^9}, {3.510551048137043*^9, 
   3.5105511440304117`*^9}, 3.510557050919793*^9, 3.5105572108200736`*^9, 
   3.5105577896122904`*^9, 3.510557847488392*^9, {3.5105583630068974`*^9, 
   3.510558405704172*^9}, {3.5105608134148026`*^9, 3.510560853085672*^9}, 
   3.5105608836617255`*^9, {3.5106264247287984`*^9, 3.5106264690796766`*^9}, {
   3.5109166369473124`*^9, 3.510916652516139*^9}, {3.510916956872674*^9, 
   3.510916976934309*^9}, {3.5109172133839245`*^9, 3.510917261400809*^9}, {
   3.51091783730702*^9, 3.5109178806750965`*^9}, 3.511002137801263*^9, {
   3.5111411385862293`*^9, 3.5111411749810934`*^9}, 3.511141210159155*^9, {
   3.5111541843533497`*^9, 3.5111541968957715`*^9}, {3.5112194905765696`*^9, 
   3.5112195044761934`*^9}, {3.5113282322408266`*^9, 3.511328247341653*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Wire Torsional Potential", "Subsubsection",
 CellChangeTimes->{{3.511328281521313*^9, 3.511328293689335*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"torsionpot", "=", 
    RowBox[{"maketorsionpot", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"only", " ", "real", " ", "part"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"torsionpotc", "=", 
    RowBox[{"maketorsionpotc", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"with", " ", "imaginary", " ", "part"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511002097319192*^9, 3.511002109783613*^9}, 
   3.5110021423564706`*^9, {3.5110022020421753`*^9, 3.5110022106377907`*^9}, {
   3.5110034113562994`*^9, 3.5110034150379057`*^9}, {3.511233156981127*^9, 
   3.5112331628935375`*^9}, {3.511328253441264*^9, 3.511328271802496*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Spring Potential", "Subsubsection",
 CellChangeTimes->{{3.5112195186878185`*^9, 3.511219521293023*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"springpot", "=", 
    RowBox[{"makespringpot", "[", "springlist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"only", " ", "real", " ", "part"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"springpotc", "=", 
    RowBox[{"makespringpotc", "[", "springlist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"with", " ", "imaginary", " ", "part"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.510542091144104*^9, 3.510542093936509*^9}, {
   3.510542888481906*^9, 3.5105428899327087`*^9}, {3.510542932146383*^9, 
   3.5105429338311863`*^9}, {3.5105434500672927`*^9, 3.510543476493739*^9}, {
   3.510543521063017*^9, 3.5105435385974483`*^9}, {3.5105435839311275`*^9, 
   3.5105435894847374`*^9}, {3.510545463687629*^9, 3.5105454866664696`*^9}, {
   3.5105498942966166`*^9, 3.510550005088011*^9}, {3.5105505425245547`*^9, 
   3.5105505701990037`*^9}, {3.5105506566075554`*^9, 3.510550672535183*^9}, {
   3.5105509738653126`*^9, 3.5105509743645134`*^9}, {3.510551048137043*^9, 
   3.5105511440304117`*^9}, 3.510557050919793*^9, 3.5105572108200736`*^9, 
   3.5105577896122904`*^9, 3.510557847488392*^9, {3.5105583630068974`*^9, 
   3.510558405704172*^9}, {3.5105608134148026`*^9, 3.510560853085672*^9}, 
   3.5105608836617255`*^9, {3.5106264247287984`*^9, 3.5106264690796766`*^9}, {
   3.5109166369473124`*^9, 3.510916652516139*^9}, {3.510916956872674*^9, 
   3.510916976934309*^9}, {3.5109172133839245`*^9, 3.510917261400809*^9}, {
   3.51091783730702*^9, 3.5109178806750965`*^9}, 3.511002137801263*^9, {
   3.5111301323777637`*^9, 3.511130142680353*^9}, {3.5111389035391035`*^9, 
   3.511138929201149*^9}, {3.511139340261871*^9, 3.511139364129913*^9}, 
   3.511139422817216*^9, {3.51114012523965*^9, 3.511140134584066*^9}, {
   3.5111409838339577`*^9, 3.5111409926791735`*^9}, {3.5111542019033804`*^9, 
   3.5111542101869946`*^9}, {3.5113283226741858`*^9, 3.5113283456218257`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Gravity Potential", "Subsubsection",
 CellChangeTimes->{{3.511219530216239*^9, 3.511219534256646*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"gravpot", "=", 
    RowBox[{"makegravpot", "[", "bodylist", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5109172647548146`*^9, 3.5109172699028234`*^9}, {
   3.510966585884634*^9, 3.5109665906738424`*^9}, {3.5109669988485594`*^9, 
   3.5109670018593645`*^9}, 3.5110021586428995`*^9, {3.511227259181363*^9, 
   3.5112272796485996`*^9}, 3.5113283557930436`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Total", "Subsubsection",
 CellChangeTimes->{{3.511219549981474*^9, 3.511219550652275*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"potential", "=", 
   RowBox[{"wirepot", "+", "gravpot", "+", "torsionpot", "+", "springpot"}]}],
   ";"}], "\n", 
 RowBox[{
  RowBox[{"potentialc", "=", 
   RowBox[{
   "wirepotc", "+", "gravpot", "+", "torsionpot", "+", "springpotc"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109172740680313`*^9, 3.5109172945196667`*^9}, 
   3.5109179087707458`*^9, 3.5110004500306864`*^9, {3.5110004867843513`*^9, 
   3.5110005204336104`*^9}, 3.5110021501720843`*^9, {3.5110022152553988`*^9, 
   3.5110022192646055`*^9}, {3.5111433491457124`*^9, 3.511143363653738*^9}, {
   3.511152543224661*^9, 3.511152549573872*^9}, {3.511226445838719*^9, 
   3.511226472530366*^9}, {3.5113289728677273`*^9, 3.511328987375753*^9}, {
   3.5127744110957346`*^9, 3.5127744184121475`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Kinetic Energy Calculation", "Subsection",
 CellChangeTimes->{{3.51091730216368*^9, 3.510917310494095*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"kinetic", "=", 
    RowBox[{"makekinetic", "[", "bodylist", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.510917352099368*^9, 3.510917508349242*^9}, {
   3.5109668339562693`*^9, 3.51096683429947*^9}, {3.5109669425480604`*^9, 
   3.5109669488504715`*^9}, 3.510967004105768*^9, 3.510967051093051*^9, {
   3.511000390828583*^9, 3.511000401436601*^9}, {3.51114029066234*^9, 
   3.511140299008355*^9}, {3.5111504745986276`*^9, 3.5111504779058332`*^9}, {
   3.511227322049474*^9, 3.511227329007086*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Damping Energy Calculation", "Subsection",
 CellChangeTimes->{{3.5113890192871785`*^9, 3.5113890229843845`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"damping", "=", 
    RowBox[{"makedamping", "[", "damperlist", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5113890305347977`*^9, 3.511389047632428*^9}, {
  3.511389361348979*^9, 3.5113893730333996`*^9}}]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Find Equilibrium Point", "Section",
 CellChangeTimes->{{3.5106265435854073`*^9, 3.5106265504962196`*^9}, 
   3.5106554796021967`*^9}],

Cell[CellGroupData[{

Cell["Potential with Default Values of Parameters", "Subsection",
 CellChangeTimes->{{3.5106268136062813`*^9, 3.5106268456331377`*^9}, {
  3.510627087792363*^9, 3.5106271045155926`*^9}, {3.510917973136459*^9, 
  3.510917992106092*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"potential0", "=", 
    RowBox[{"potential", "/.", "subdefparams"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5105539888460145`*^9, 3.510554070636958*^9}, {
   3.510554121430648*^9, 3.5105542109124045`*^9}, 3.5105546485935736`*^9, {
   3.5105546974216595`*^9, 3.510554704051671*^9}, {3.5105571731928077`*^9, 
   3.5105571828960247`*^9}, 3.51055786907883*^9, 3.510557938810952*^9, {
   3.5105584149237885`*^9, 3.5105584273414106`*^9}, {3.510559192210754*^9, 
   3.5105592145655932`*^9}, {3.510560255901023*^9, 3.5105602574922256`*^9}, {
   3.5105606405976987`*^9, 3.51056068084577*^9}, 3.5105607235898447`*^9, 
   3.5105611168353353`*^9, {3.5106265647702446`*^9, 3.5106265760490646`*^9}, {
   3.510627120958021*^9, 3.51062714858567*^9}, {3.5106273594512405`*^9, 
   3.5106273673448544`*^9}, {3.5106274131777344`*^9, 3.510627420806148*^9}, {
   3.510636250108863*^9, 3.5106362563176737`*^9}, {3.5106557863611355`*^9, 
   3.51065579467595*^9}, {3.5108253325205708`*^9, 3.5108253359993763`*^9}, {
   3.5109179503760185`*^9, 3.5109179592680345`*^9}, {3.5109180139149303`*^9, 
   3.510918040434977*^9}, 3.5109183532935266`*^9, 3.510918580757526*^9, 
   3.510918631816416*^9, 3.51091887893645*^9, {3.511150554049567*^9, 
   3.5111505637215843`*^9}, {3.511150600678049*^9, 3.5111506134388714`*^9}, 
   3.511154234382637*^9, 3.5111613806292715`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Find Equilibrium Point", "Subsection",
 CellChangeTimes->{{3.5106274478565955`*^9, 3.510627536183951*^9}, 
   3.510655478166994*^9}],

Cell["Find optimal values that make the potential energy minimum.", "Text",
 CellChangeTimes->{{3.510627542611162*^9, 3.5106275966652565`*^9}, {
   3.5109180810106487`*^9, 3.510918084910655*^9}, {3.5109182507701464`*^9, 
   3.510918255060154*^9}, 3.510975865186532*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"subminvars", "=", 
  RowBox[{
   RowBox[{"FindMinimum", "[", "\[IndentingNewLine]", 
    RowBox[{"potential0", ",", "\[IndentingNewLine]", 
     RowBox[{"Join", "[", 
      RowBox[{
       RowBox[{"comb", "[", 
        RowBox[{"allvars", ",", "initvars"}], "]"}], ",", 
       RowBox[{"comb", "[", 
        RowBox[{"allfloats", ",", "initfloats"}], "]"}]}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"Method", "\[Rule]", "\"\<Newton\>\""}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"MaxIterations", "\[Rule]", "100"}]}], "\[IndentingNewLine]", 
    "]"}], "[", 
   RowBox[{"[", "2", "]"}], "]"}]}]], "Input",
 CellChangeTimes->{{3.51055781213873*^9, 3.510557830281562*^9}, {
   3.510557969761407*^9, 3.5105579803850255`*^9}, {3.510558511285158*^9, 
   3.510558539942408*^9}, {3.5105590865985684`*^9, 3.5105591272054396`*^9}, {
   3.510559279305707*^9, 3.510559289632925*^9}, 3.5105605175134826`*^9, 
   3.51056060141043*^9, {3.510560695213395*^9, 3.5105606957281957`*^9}, 
   3.5105609712558794`*^9, 3.510561120158141*^9, {3.510627334818797*^9, 
   3.510627350855625*^9}, {3.5106274406493826`*^9, 3.510627443504188*^9}, 
   3.510627615010889*^9, {3.5106295313562555`*^9, 3.5106295448814793`*^9}, 
   3.5106361885199547`*^9, 3.51063623703604*^9, {3.510636431164381*^9, 
   3.5106364370455914`*^9}, {3.510655802725564*^9, 3.5106558188091927`*^9}, 
   3.510655856608059*^9, {3.510655900756136*^9, 3.510655922518174*^9}, 
   3.510918005334915*^9, {3.510918228306107*^9, 3.510918233828517*^9}, {
   3.5109182690221786`*^9, 3.5109182811278*^9}, {3.511004376030794*^9, 
   3.5110044189620695`*^9}, {3.5110044623613453`*^9, 3.511004473421765*^9}, {
   3.5111403358556194`*^9, 3.5111403660104723`*^9}, {3.5111412252287817`*^9, 
   3.511141227818386*^9}, 3.5111413265977592`*^9, {3.5111413751606445`*^9, 
   3.5111414888988447`*^9}, {3.5111549058546166`*^9, 3.5111549297538586`*^9}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"xIM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yIM", "\[Rule]", 
    RowBox[{"-", "2.0836720435665974`"}]}], ",", 
   RowBox[{"zIM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"pitchIM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yawIM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"rollIM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"xIRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yIRM", "\[Rule]", 
    RowBox[{"-", "2.092793744574689`"}]}], ",", 
   RowBox[{"zIRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"pitchIRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yawIRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"rollIRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"xTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yTM", "\[Rule]", 
    RowBox[{"-", "4.172902978124215`"}]}], ",", 
   RowBox[{"zTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"pitchTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yawTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"rollTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"xRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yRM", "\[Rule]", 
    RowBox[{"-", "4.180405886818138`"}]}], ",", 
   RowBox[{"zRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"pitchRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yawRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"rollRM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"qGAS4", "\[Rule]", "4.7868715545286215`*^-17"}]}], 
  "}"}]], "Output",
 CellChangeTimes->{
  3.510918281798601*^9, {3.5109185842987328`*^9, 3.510918588183139*^9}, 
   3.5109188021843147`*^9, 3.5109188818380547`*^9, 3.510956282795619*^9, 
   3.510958553471608*^9, 3.5109760020455723`*^9, 3.5109762224427595`*^9, 
   3.510981584761199*^9, 3.510997021052256*^9, 3.5109970696775417`*^9, 
   3.51100053174363*^9, 3.5110022334138308`*^9, 3.5110032212855654`*^9, 
   3.511003477625216*^9, 3.5110044026288404`*^9, {3.5110044639837484`*^9, 
   3.511004474498167*^9}, 3.511004507944626*^9, 3.5110674021996913`*^9, 
   3.5111403773048925`*^9, 3.511141010588004*^9, 3.5111412287075877`*^9, 
   3.5111413274869614`*^9, {3.5111413840838604`*^9, 3.5111414391659575`*^9}, {
   3.511141482034833*^9, 3.5111414896320467`*^9}, 3.511143384729375*^9, 
   3.5111469195239835`*^9, 3.5111472172193065`*^9, 3.511147933229364*^9, {
   3.511149501266118*^9, 3.5111495096589327`*^9}, 3.511150531741528*^9, 
   3.511150641955722*^9, {3.511150687227001*^9, 3.5111507000658236`*^9}, 
   3.5111525605562916`*^9, 3.511152660521267*^9, 3.5111541628097115`*^9, 
   3.5111546461141605`*^9, 3.511154858804934*^9, 3.511154922016245*^9, 
   3.5111550870333347`*^9, 3.511227338569903*^9, 3.5112285791620817`*^9, 
   3.5112287411059666`*^9, 3.5112289115986657`*^9, 3.511228943719122*^9, 
   3.5112291311846514`*^9, 3.5112291859407473`*^9, 3.511229342253022*^9, 
   3.511229667950394*^9, 3.51122993873567*^9, 3.51123036609842*^9, 
   3.511230514938282*^9, 3.5112317294784155`*^9, {3.511231796870534*^9, 
   3.5112318180085707`*^9}, 3.511232314782448*^9, 3.5112324235614386`*^9, 
   3.511232456883097*^9, 3.511232491608758*^9, 3.5112325360376368`*^9, 
   3.511232579889313*^9, 3.511232655409046*^9, 3.511232843857377*^9, 
   3.5112329061326866`*^9, 3.5112332275400505`*^9, 3.5113291732188797`*^9, 
   3.511329411712098*^9, 3.511330414918663*^9, 3.511330611135808*^9, 
   3.5113309898420725`*^9, 3.511331145155946*^9, 3.511332932014284*^9, 
   3.511343444537177*^9, 3.5113893904430304`*^9, 3.5114066658619184`*^9, 
   3.5114067613964863`*^9, 3.5114069281139793`*^9, 3.5114315309789104`*^9, 
   3.5114900390388813`*^9, 3.511494313353214*^9, 3.5114947992160673`*^9, 
   3.5114950951797867`*^9, 3.5114976128240194`*^9, 3.5123557673055196`*^9, 
   3.5123559844585347`*^9, 3.512357720304784*^9, 3.5123578229841638`*^9, 
   3.5123586976933002`*^9, 3.512358797330675*^9, 3.5123589226144953`*^9, 
   3.5124237581731677`*^9, 3.5124275277636547`*^9, 3.5127107827030926`*^9, 
   3.5127123057457695`*^9, 3.5127124974389057`*^9, 3.5127125912262707`*^9, 
   3.5127130589306917`*^9, 3.5127131447932425`*^9, 3.51273061964657*^9, 
   3.5127326937654157`*^9, 3.512733364067393*^9, 3.5127350758739996`*^9, 
   3.512774776027176*^9, 3.5127748237320595`*^9}]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Equations of Motion", "Section",
 CellChangeTimes->{{3.510627640017733*^9, 3.510627674649794*^9}, 
   3.5109189734102154`*^9}],

Cell[CellGroupData[{

Cell["Mass Matrix", "Subsection",
 CellChangeTimes->{{3.510561334346517*^9, 3.510561337388523*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matMvv", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"kinetic", ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellChangeTimes->{{3.510561339822127*^9, 3.5105613828470025`*^9}, {
   3.5105614843563805`*^9, 3.51056150657082*^9}, {3.510561663541896*^9, 
   3.5105617389524283`*^9}, {3.5105619183683434`*^9, 
   3.5105619553404083`*^9}, {3.5106285477833276`*^9, 
   3.5106286902115774`*^9}, {3.510629347182332*^9, 3.510629358086751*^9}, {
   3.5106295553334975`*^9, 3.5106295558638983`*^9}, {3.5106573586878967`*^9, 
   3.5106574988697433`*^9}, 3.5106575498350325`*^9, {3.5106575855122952`*^9, 
   3.510657586838298*^9}, 3.510657701592099*^9, {3.510918992301849*^9, 
   3.510919033813522*^9}, {3.510919072236389*^9, 3.5109190853248124`*^9}, {
   3.511002629373726*^9, 3.511002636112938*^9}, {3.511142697495368*^9, 
   3.511142697651368*^9}, {3.511148104096464*^9, 3.5111481386037245`*^9}, {
   3.5112308639264946`*^9, 3.5112308672337008`*^9}, 3.5127273081131516`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matMpv", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"kinetic", ",", 
           RowBox[{"velvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"velparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "velparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.510629277247409*^9, 3.5106293107406683`*^9}, {
   3.510629367384368*^9, 3.510629370301573*^9}, {3.5106295599043055`*^9, 
   3.5106295603567066`*^9}, {3.510657365037108*^9, 3.5106574369532347`*^9}, 
   3.510657709126912*^9, {3.5106577452409763`*^9, 3.510657747034979*^9}, {
   3.5109190411611347`*^9, 3.510919114247263*^9}, 3.51091916922176*^9, {
   3.5109193418204627`*^9, 3.51091934586087*^9}, 3.5109555482566385`*^9, {
   3.5110026383125415`*^9, 3.5110026412609468`*^9}, {3.5111427006777735`*^9, 
   3.511142701020974*^9}, {3.5111428283327975`*^9, 3.5111428289723988`*^9}, {
   3.51114814166133*^9, 3.5111481490869427`*^9}, {3.511230607696045*^9, 
   3.511230615230858*^9}, {3.5112308557832804`*^9, 3.5112308586848855`*^9}, 
   3.5127273090023527`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness Matrix", "Subsection",
 CellChangeTimes->{{3.510628701318797*^9, 3.510628705218804*^9}, {
  3.510628763250906*^9, 3.5106287646393085`*^9}}],

Cell[CellGroupData[{

Cell["Only Real", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109555066872606`*^9, 3.510955522479164*^9}, {
  3.5111459241018353`*^9, 3.511145929171844*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matKvv", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"potential", ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5105619627036214`*^9, 3.510561989847669*^9}, 
   3.510562426586036*^9, {3.510562624565984*^9, 3.5105626282787905`*^9}, 
   3.510562823575534*^9, {3.5106287756061277`*^9, 3.5106287991777687`*^9}, 
   3.510629372735177*^9, {3.5106295648963146`*^9, 3.5106295735387297`*^9}, 
   3.510634023423352*^9, 3.5106340995670853`*^9, 3.510634705986951*^9, 
   3.5106357421095705`*^9, {3.5106577195477304`*^9, 3.5106577496869836`*^9}, {
   3.510919174588169*^9, 3.5109192067710257`*^9}, {3.51091935724889*^9, 
   3.5109193604780955`*^9}, 3.510955522479164*^9, {3.511000578777713*^9, 
   3.5110005872953277`*^9}, 3.511002643460551*^9, {3.51100273993112*^9, 
   3.511002769945573*^9}, {3.5110030522904687`*^9, 3.511003069060498*^9}, {
   3.511141726877263*^9, 3.511141728328066*^9}, 3.511141847169074*^9, {
   3.51114310883689*^9, 3.511143116262503*^9}, {3.511143405602212*^9, 
   3.5111434129186244`*^9}, {3.5111459241018353`*^9, 3.511145929171844*^9}, {
   3.5112308780601196`*^9, 3.5112308853609324`*^9}, 3.512727244075039*^9, 
   3.512727310281555*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matKpv", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"potential", ",", 
           RowBox[{"allvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"allparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "allparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.510629298089046*^9, 3.510629328524699*^9}, {
   3.5106295791391397`*^9, 3.510629587048353*^9}, 3.5106341033890924`*^9, 
   3.5106343797903776`*^9, 3.510634742647015*^9, 3.510635745759977*^9, {
   3.510657728892147*^9, 3.510657756722596*^9}, 3.5109191710313625`*^9, {
   3.5109192125430355`*^9, 3.5109192243990564`*^9}, {3.510919348949675*^9, 
   3.5109193550336857`*^9}, {3.510955522479164*^9, 3.5109555416132584`*^9}, {
   3.5110027536747446`*^9, 3.511002764969164*^9}, {3.5111417317132716`*^9, 
   3.5111417374696817`*^9}, 3.511141852644684*^9, {3.5111428317960033`*^9, 
   3.5111428323108044`*^9}, {3.5111459241018353`*^9, 3.511145929171844*^9}, {
   3.511230887888137*^9, 3.5112309006801596`*^9}, 3.5127272467426434`*^9, 
   3.5127273111395564`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Length", "[", "allfloats", "]"}], "\[GreaterEqual]", "1"}], 
     ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"matKfv", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potential", ",", 
              RowBox[{"allvars", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKff", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potential", ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKfp", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potential", ",", 
              RowBox[{"allparams", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allparams", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKvv", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfv", ".", 
          RowBox[{"Inverse", "[", "matKff", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfv", "]"}]}], "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"matKpv", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfv", ".", 
          RowBox[{"Inverse", "[", "matKff", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfp", "]"}]}], "]"}]}]}], ";"}]}], 
    "\[IndentingNewLine]", "]"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5111416497975273`*^9, 3.5111417209960527`*^9}, {
  3.5111417526641083`*^9, 3.5111418619267*^9}, {3.511142246202175*^9, 
  3.5111422962938633`*^9}, {3.5111425664863377`*^9, 3.5111425873747745`*^9}, {
  3.5111428355400105`*^9, 3.5111428761468816`*^9}, {3.511142909967741*^9, 
  3.511142930606577*^9}, {3.511142981868267*^9, 3.5111430139887238`*^9}, {
  3.5111430674656177`*^9, 3.511143067824418*^9}, {3.511143466379918*^9, 
  3.51114346733152*^9}, {3.5111435471100597`*^9, 3.5111435507604666`*^9}, {
  3.5111436040657597`*^9, 3.5111436094009695`*^9}, {3.511145929171844*^9, 
  3.5111459456454725`*^9}, {3.511149406308751*^9, 3.5111494064335513`*^9}, {
  3.5112309057189684`*^9, 3.5112309636574697`*^9}, {3.5112312731776133`*^9, 
  3.5112313016164637`*^9}, {3.5127272495194483`*^9, 3.5127272553538585`*^9}, {
  3.512727288207516*^9, 3.5127273062099476`*^9}, {3.512735314694819*^9, 
  3.512735322167232*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["With Imaginary Part", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.51095551455171*^9, 3.510955530382616*^9}, {
  3.5111481601941624`*^9, 3.511148165466972*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matKvvc", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"potentialc", ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5105619627036214`*^9, 3.510561989847669*^9}, 
   3.510562426586036*^9, {3.510562624565984*^9, 3.5105626282787905`*^9}, 
   3.510562823575534*^9, {3.5106287756061277`*^9, 3.5106287991777687`*^9}, 
   3.510629372735177*^9, {3.5106295648963146`*^9, 3.5106295735387297`*^9}, 
   3.510634023423352*^9, 3.5106340995670853`*^9, 3.510634705986951*^9, 
   3.5106357421095705`*^9, {3.5106577195477304`*^9, 3.5106577496869836`*^9}, {
   3.510919174588169*^9, 3.5109192067710257`*^9}, {3.51091935724889*^9, 
   3.5109193604780955`*^9}, {3.510922241814151*^9, 3.5109222467374325`*^9}, 
   3.510955530382616*^9, {3.5110005587316775`*^9, 3.5110005689340954`*^9}, {
   3.511002971747527*^9, 3.511002992214763*^9}, {3.5111437697380514`*^9, 
   3.5111437766956635`*^9}, 3.511143821826543*^9, {3.5111481601941624`*^9, 
   3.511148165466972*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matKpvc", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"potentialc", ",", 
           RowBox[{"allvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"allparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "allparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.510629298089046*^9, 3.510629328524699*^9}, {
   3.5106295791391397`*^9, 3.510629587048353*^9}, 3.5106341033890924`*^9, 
   3.5106343797903776`*^9, 3.510634742647015*^9, 3.510635745759977*^9, {
   3.510657728892147*^9, 3.510657756722596*^9}, 3.5109191710313625`*^9, {
   3.5109192125430355`*^9, 3.5109192243990564`*^9}, {3.510919348949675*^9, 
   3.5109193550336857`*^9}, {3.5109222445573077`*^9, 
   3.5109222483205233`*^9}, {3.510955530382616*^9, 3.510955540563198*^9}, {
   3.511002993930766*^9, 3.5110029968011713`*^9}, {3.511143772499256*^9, 
   3.511143780985671*^9}, 3.5111438228093443`*^9, {3.5111481601941624`*^9, 
   3.511148165466972*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Length", "[", "allfloats", "]"}], "\[GreaterEqual]", "1"}], 
     ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"matKfvc", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potentialc", ",", 
              RowBox[{"allvars", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKffc", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potentialc", ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKfpc", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potentialc", ",", 
              RowBox[{"allparams", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allparams", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKvvc", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfvc", ".", 
          RowBox[{"Inverse", "[", "matKffc", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfvc", "]"}]}], "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"matKpvc", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfvc", ".", 
          RowBox[{"Inverse", "[", "matKffc", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfpc", "]"}]}], "]"}]}]}], ";"}]}], 
    "\[IndentingNewLine]", "]"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5111416497975273`*^9, 3.5111417209960527`*^9}, {
   3.5111417526641083`*^9, 3.5111418619267*^9}, {3.511142246202175*^9, 
   3.5111422962938633`*^9}, {3.5111425664863377`*^9, 
   3.5111425873747745`*^9}, {3.5111428355400105`*^9, 
   3.5111428761468816`*^9}, {3.511142909967741*^9, 3.511142930606577*^9}, {
   3.511142981868267*^9, 3.5111430139887238`*^9}, {3.5111430674656177`*^9, 
   3.511143067824418*^9}, {3.511143466379918*^9, 3.51114346733152*^9}, {
   3.5111435471100597`*^9, 3.5111435507604666`*^9}, {3.5111436040657597`*^9, 
   3.5111436094009695`*^9}, {3.5111438011565065`*^9, 3.511143815914132*^9}, 
   3.511148165466972*^9, {3.5111494314871955`*^9, 3.511149431689996*^9}, {
   3.512727323993979*^9, 3.5127273242591796`*^9}, {3.5127352324358745`*^9, 
   3.512735240625889*^9}, {3.5127353013099957`*^9, 3.5127353103736115`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Damping Matrix", "Subsection",
 CellChangeTimes->{{3.510561334346517*^9, 3.510561337388523*^9}, {
  3.5113900124941225`*^9, 3.511390015442528*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matGvv", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"damping", ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellChangeTimes->{{3.510561339822127*^9, 3.5105613828470025`*^9}, {
   3.5105614843563805`*^9, 3.51056150657082*^9}, {3.510561663541896*^9, 
   3.5105617389524283`*^9}, {3.5105619183683434`*^9, 
   3.5105619553404083`*^9}, {3.5106285477833276`*^9, 
   3.5106286902115774`*^9}, {3.510629347182332*^9, 3.510629358086751*^9}, {
   3.5106295553334975`*^9, 3.5106295558638983`*^9}, {3.5106573586878967`*^9, 
   3.5106574988697433`*^9}, 3.5106575498350325`*^9, {3.5106575855122952`*^9, 
   3.510657586838298*^9}, 3.510657701592099*^9, {3.510918992301849*^9, 
   3.510919033813522*^9}, {3.510919072236389*^9, 3.5109190853248124`*^9}, {
   3.511002629373726*^9, 3.511002636112938*^9}, {3.511142697495368*^9, 
   3.511142697651368*^9}, {3.511148104096464*^9, 3.5111481386037245`*^9}, {
   3.5112308639264946`*^9, 3.5112308672337008`*^9}, {3.511390018359733*^9, 
   3.511390025722946*^9}, {3.5113900655654163`*^9, 3.511390079979841*^9}, 
   3.512727336255601*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matGpv", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"damping", ",", 
           RowBox[{"velvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"velparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "velparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.510629277247409*^9, 3.5106293107406683`*^9}, {
   3.510629367384368*^9, 3.510629370301573*^9}, {3.5106295599043055`*^9, 
   3.5106295603567066`*^9}, {3.510657365037108*^9, 3.5106574369532347`*^9}, 
   3.510657709126912*^9, {3.5106577452409763`*^9, 3.510657747034979*^9}, {
   3.5109190411611347`*^9, 3.510919114247263*^9}, 3.51091916922176*^9, {
   3.5109193418204627`*^9, 3.51091934586087*^9}, 3.5109555482566385`*^9, {
   3.5110026383125415`*^9, 3.5110026412609468`*^9}, {3.5111427006777735`*^9, 
   3.511142701020974*^9}, {3.5111428283327975`*^9, 3.5111428289723988`*^9}, {
   3.51114814166133*^9, 3.5111481490869427`*^9}, {3.511230607696045*^9, 
   3.511230615230858*^9}, {3.5112308557832804`*^9, 3.5112308586848855`*^9}, {
   3.511390021198938*^9, 3.5113900346461616`*^9}, {3.5113900824290457`*^9, 
   3.511390094378667*^9}, 3.5127273373944025`*^9}]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Eigen Mode", "Section",
 CellChangeTimes->{{3.510639974173004*^9, 3.51063997762061*^9}}],

Cell[CellGroupData[{

Cell["Eigen Frequencies & Vectors", "Subsection",
 CellChangeTimes->{{3.5106399863410254`*^9, 3.510639995685442*^9}, {
  3.5109193175936203`*^9, 3.510919324769633*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eigen", "=", 
   RowBox[{"Eigensystem", "[", 
    RowBox[{
     RowBox[{"Inverse", "[", "matMvv", "]"}], ".", 
     RowBox[{"Re", "[", "matKvv", "]"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"eigenf", "=", 
   RowBox[{"Reverse", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", 
      RowBox[{"eigen", "[", 
       RowBox[{"[", "1", "]"}], "]"}], "]"}], "/", 
     RowBox[{"(", 
      RowBox[{"2", 
       RowBox[{"N", "[", "Pi", "]"}]}], ")"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"eigenv", "=", 
   RowBox[{"Reverse", "[", 
    RowBox[{"eigen", "[", 
     RowBox[{"[", "2", "]"}], "]"}], "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5106404812986946`*^9, 3.5106405612644353`*^9}, {
  3.510642026727812*^9, 3.510642034652625*^9}, {3.510642839520439*^9, 
  3.510642845308049*^9}, {3.5106429258041906`*^9, 3.510642929423397*^9}, {
  3.5106434421806974`*^9, 3.510643450791913*^9}, {3.5109192830239596`*^9, 
  3.510919329933242*^9}, {3.511143879000643*^9, 3.5111438815434475`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Eigen Mode Listup", "Subsection",
 CellChangeTimes->{{3.5106417564485364`*^9, 3.510641767571356*^9}, {
  3.5106435074824123`*^9, 3.5106435076384125`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eigenmodetable", "[", 
  RowBox[{"eigenf", ",", "eigenv", ",", "0.3"}], "]"}]], "Input",
 CellChangeTimes->{{3.5106432816876154`*^9, 3.510643291593633*^9}, {
   3.510921715107025*^9, 3.510921737338297*^9}, {3.510922100897091*^9, 
   3.5109221122847424`*^9}, 3.5109585862628655`*^9, {3.511407145469161*^9, 
   3.5114071462179623`*^9}, {3.5114900165592413`*^9, 3.511490016886842*^9}, {
   3.512727357924039*^9, 3.51272735881324*^9}, {3.5127749540546885`*^9, 
   3.5127749543510895`*^9}}],

Cell[BoxData[
 InterpretationBox[GridBox[{
    {"\<\"N\"\>", "\<\"Freq\"\>", "\<\"Type\"\>", "\<\"\"\>", "\<\"\"\>", "\<\
\"\"\>", "\<\"\"\>"},
    {"1", 
     "0.016496022745954898`", "\<\"yawIM(0.577053)\"\>", \
"\<\"yawTM(0.57745)\"\>", "\<\"yawRM(0.577548)\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"2", 
     "0.241794444626887`", "\<\"rollIM(-0.575412)\"\>", \
"\<\"rollTM(-0.575378)\"\>", "\<\"rollRM(-0.576231)\"\>", "\<\"\"\>", "\<\"\"\
\>"},
    {"3", 
     "0.2563816857952316`", "\<\"pitchIM(0.360749)\"\>", \
"\<\"zTM(-0.516022)\"\>", "\<\"pitchTM(0.360414)\"\>", \
"\<\"zRM(-0.519044)\"\>", "\<\"pitchRM(0.388999)\"\>"},
    {"4", 
     "0.25973509295839653`", "\<\"rollIM(-0.51936)\"\>", \
"\<\"rollTM(-0.519317)\"\>", "\<\"rollRM(-0.520207)\"\>", "\<\"\"\>", "\<\"\"\
\>"},
    {"5", 
     "0.31008229994655206`", "\<\"pitchIM(0.555846)\"\>", \
"\<\"pitchTM(0.552832)\"\>", "\<\"pitchRM(0.620423)\"\>", "\<\"\"\>", "\<\"\"\
\>"},
    {"6", 
     "0.3291523412354136`", "\<\"yIM(0.576007)\"\>", "\<\"yTM(0.577071)\"\>", \
"\<\"yRM(0.578969)\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"7", 
     "0.3450003366401993`", "\<\"xIRM(0.961903)\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>"},
    {"8", 
     "0.3450003366402034`", "\<\"xIRM(0.30928)\"\>", \
"\<\"zIRM(0.950971)\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"9", 
     "0.3450822947923005`", "\<\"zTM(0.991894)\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>"},
    {"10", 
     "0.34508540777607155`", "\<\"xTM(-0.993063)\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"11", 
     "0.5400145679177233`", "\<\"yawIRM(1.)\"\>", "\<\"\"\>", "\<\"\"\>", "\<\
\"\"\>", "\<\"\"\>"},
    {"12", 
     "0.6275245499862456`", "\<\"yawTM(0.99967)\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>"},
    {"13", 
     "0.7821291562779641`", "\<\"zIM(0.945684)\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>"},
    {"14", 
     "0.7821652979509809`", "\<\"xIM(0.946082)\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>"},
    {"15", 
     "0.8673575348911251`", "\<\"yawIM(0.600579)\"\>", \
"\<\"yawTM(-0.668019)\"\>", "\<\"yawRM(-0.439381)\"\>", "\<\"\"\>", \
"\<\"\"\>"},
    {"16", 
     "1.560963195028493`", "\<\"pitchTM(-0.985308)\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"17", 
     "1.9331504374755073`", "\<\"pitchTM(-0.953477)\"\>", "\<\"\"\>", "\<\"\"\
\>", "\<\"\"\>", "\<\"\"\>"},
    {"18", 
     "5.64652884533787`", "\<\"yIRM(1.)\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\
\>", "\<\"\"\>"},
    {"19", 
     "6.477056272763566`", "\<\"yTM(0.925911)\"\>", "\<\"\"\>", "\<\"\"\>", "\
\<\"\"\>", "\<\"\"\>"},
    {"20", 
     "8.83907211787191`", "\<\"pitchIRM(-0.999953)\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"21", 
     "8.839072117871915`", "\<\"rollIRM(-0.999988)\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"22", 
     "9.184321861975958`", "\<\"yIM(0.396291)\"\>", "\<\"yTM(-0.908464)\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"23", 
     "10.171649129692439`", "\<\"rollTM(0.992117)\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>"},
    {"24", 
     "12.951870635016405`", "\<\"rollIM(-0.396778)\"\>", \
"\<\"rollTM(0.913226)\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[2.0999999999999996`]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  TableForm[{{"N", "Freq", "Type"}, {
    1, 0.016496022745954898`, "yawIM(0.577053)", "yawTM(0.57745)", 
     "yawRM(0.577548)"}, {
    2, 0.241794444626887, "rollIM(-0.575412)", "rollTM(-0.575378)", 
     "rollRM(-0.576231)"}, {
    3, 0.2563816857952316, "pitchIM(0.360749)", "zTM(-0.516022)", 
     "pitchTM(0.360414)", "zRM(-0.519044)", "pitchRM(0.388999)"}, {
    4, 0.25973509295839653`, "rollIM(-0.51936)", "rollTM(-0.519317)", 
     "rollRM(-0.520207)"}, {
    5, 0.31008229994655206`, "pitchIM(0.555846)", "pitchTM(0.552832)", 
     "pitchRM(0.620423)"}, {
    6, 0.3291523412354136, "yIM(0.576007)", "yTM(0.577071)", 
     "yRM(0.578969)"}, {7, 0.3450003366401993, "xIRM(0.961903)"}, {
    8, 0.3450003366402034, "xIRM(0.30928)", "zIRM(0.950971)"}, {
    9, 0.3450822947923005, "zTM(0.991894)"}, {
    10, 0.34508540777607155`, "xTM(-0.993063)"}, {
    11, 0.5400145679177233, "yawIRM(1.)"}, {
    12, 0.6275245499862456, "yawTM(0.99967)"}, {
    13, 0.7821291562779641, "zIM(0.945684)"}, {
    14, 0.7821652979509809, "xIM(0.946082)"}, {
    15, 0.8673575348911251, "yawIM(0.600579)", "yawTM(-0.668019)", 
     "yawRM(-0.439381)"}, {16, 1.560963195028493, "pitchTM(-0.985308)"}, {
    17, 1.9331504374755073`, "pitchTM(-0.953477)"}, {
    18, 5.64652884533787, "yIRM(1.)"}, {
    19, 6.477056272763566, "yTM(0.925911)"}, {
    20, 8.83907211787191, "pitchIRM(-0.999953)"}, {
    21, 8.839072117871915, "rollIRM(-0.999988)"}, {
    22, 9.184321861975958, "yIM(0.396291)", "yTM(-0.908464)"}, {
    23, 10.171649129692439`, "rollTM(0.992117)"}, {
    24, 12.951870635016405`, "rollIM(-0.396778)", 
     "rollTM(0.913226)"}}]]], "Output",
 CellChangeTimes->{
  3.510643292919636*^9, 3.5106433400473185`*^9, 3.5106434238662653`*^9, 
   3.5106434762355576`*^9, 3.510643723168391*^9, 3.5106438344589863`*^9, 
   3.510643892756289*^9, 3.5106577700762196`*^9, 3.5108253730494413`*^9, 
   3.51082974384672*^9, 3.510829776169977*^9, 3.5108302338369813`*^9, 
   3.5108303752356296`*^9, {3.5108305553691463`*^9, 3.51083057477558*^9}, 
   3.5108307570151005`*^9, 3.510878378699349*^9, 3.5109217415145354`*^9, 
   3.510921781893845*^9, {3.5109221017821417`*^9, 3.5109221126697645`*^9}, 
   3.510956284636422*^9, 3.510958555156411*^9, 3.5109585879476686`*^9, 
   3.510976004120376*^9, 3.5109762241431627`*^9, 3.510981586648802*^9, 
   3.510997623852915*^9, 3.510997670574997*^9, 3.5109978286968746`*^9, 
   3.5110005994633493`*^9, 3.5110012262880497`*^9, 3.511002249466259*^9, 
   3.5110027958260183`*^9, 3.5110032275255766`*^9, 3.511003483288026*^9, 
   3.5110045135606356`*^9, 3.5110674087517023`*^9, 3.511144001897659*^9, 
   3.5111469200231843`*^9, 3.511147217780907*^9, 3.5111479339157653`*^9, 
   3.511149448413225*^9, {3.5111495020461197`*^9, 3.5111495104389343`*^9}, 
   3.511150737209489*^9, 3.51115124445998*^9, 3.5111525863899364`*^9, 
   3.511152661114068*^9, 3.5111551241458*^9, 3.5112282013762183`*^9, 
   3.511228596821313*^9, 3.5112287471743765`*^9, 3.5112289126906676`*^9, 
   3.5112289448423243`*^9, 3.5112292057371826`*^9, 3.5112293438286247`*^9, 
   3.511229669666397*^9, {3.511229940545273*^9, 3.511229945724482*^9}, 
   3.511230367908024*^9, 3.5112305163578844`*^9, 3.5112318289909897`*^9, 
   3.5112323159368496`*^9, 3.511232424778241*^9, 3.5112324581154995`*^9, 
   3.51123249266956*^9, 3.511232537082838*^9, 3.5112325809033155`*^9, 
   3.5112326565478477`*^9, 3.5112328450429792`*^9, 3.511232907536689*^9, 
   3.511233228663253*^9, 3.511329491833839*^9, 3.5113304379287033`*^9, 
   3.5113306313690434`*^9, 3.511331015972119*^9, 3.511331177635203*^9, 
   3.5113329869263806`*^9, 3.511406697171174*^9, 3.5114067818637223`*^9, 
   3.5114069491428165`*^9, 3.5114071480587654`*^9, 3.5114315657357717`*^9, 
   3.511490058913316*^9, 3.511494334787651*^9, 3.511496090224537*^9, 
   3.512355787464673*^9, 3.5123560041613693`*^9, 3.5127108132167463`*^9, 
   3.5127123258698044`*^9, 3.5127125177033415`*^9, 3.5127126119275064`*^9, 
   3.512730655448633*^9, 3.5127329310730324`*^9, 3.512774841625291*^9, 
   3.5127749548034897`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Eigen Mode 3D-Graphics", "Subsection",
 CellChangeTimes->{{3.5108295124827137`*^9, 3.510829513075515*^9}, {
  3.510956048202407*^9, 3.510956066017638*^9}}],

Cell[CellGroupData[{

Cell["Plot Range Definition", "Subsubsection",
 CellChangeTimes->{{3.51082764299103*^9, 3.5108276467818365`*^9}, {
   3.5109210878847294`*^9, 3.51092116774127*^9}, 3.5109212261165724`*^9, 
   3.511003172847481*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"maxz", "=", 
    RowBox[{"1.5", "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"z", "-", 
    RowBox[{"width", " ", "of", " ", "the", " ", "system"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"maxx", "=", 
    RowBox[{"1.5", "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"x", "-", 
    RowBox[{"width", " ", "of", " ", "the", " ", "system"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"ytop", "=", 
    RowBox[{"0.3", "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Top", " ", "height", " ", "of", " ", "the", " ", "system"}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"ybottom", "=", 
   RowBox[{
    RowBox[{"-", "5.0"}], "meter"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"Bottom", " ", "height", " ", "of", " ", "the", " ", "system"}], 
   "*)"}], "\[IndentingNewLine]"}]}], "Input",
 CellChangeTimes->{{3.5109211968977213`*^9, 3.5109212408741984`*^9}, {
   3.5109220308510847`*^9, 3.5109220309910927`*^9}, {3.510958542473589*^9, 
   3.51095854336279*^9}, {3.510968368593365*^9, 3.510968418575853*^9}, {
   3.5109954409480605`*^9, 3.510995447197418*^9}, {3.5109987157704387`*^9, 
   3.510998749638098*^9}, {3.511001189939986*^9, 3.511001205306013*^9}, 
   3.511003172847481*^9, {3.511129432507734*^9, 3.5111294518908424`*^9}, {
   3.511152966983805*^9, 3.5111529708682117`*^9}, {3.511228562750853*^9, 
   3.5112285692248645`*^9}, {3.511229309087364*^9, 3.5112293130029707`*^9}, {
   3.511329984950708*^9, 3.5113299977115307`*^9}, {3.5127101502779818`*^9, 
   3.512710157438394*^9}, {3.512774924648637*^9, 3.5127749351006556`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Eigen Mode 3D-Graphics", "Subsubsection",
 CellChangeTimes->{{3.512709012365183*^9, 3.5127090238936033`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eigenplot", "[", 
  RowBox[{"eigenv", ",", "0.4", ",", "11"}], "]"}]], "Input",
 CellChangeTimes->{{3.5108281708803577`*^9, 3.5108281797879734`*^9}, {
   3.510829089456771*^9, 3.510829089612771*^9}, {3.5108291378012557`*^9, 
   3.510829143760466*^9}, {3.510829207346178*^9, 3.5108292251926093`*^9}, {
   3.510829318496373*^9, 3.5108293323959975`*^9}, {3.510829380802882*^9, 
   3.510829395279708*^9}, {3.5108294539514112`*^9, 3.510829454965413*^9}, {
   3.5108783919749727`*^9, 3.5108784826891317`*^9}, {3.5108858833120837`*^9, 
   3.510885917039343*^9}, {3.510922053102357*^9, 3.5109220655000668`*^9}, {
   3.510922145501642*^9, 3.510922154304146*^9}, {3.5109586034072957`*^9, 
   3.5109586510965796`*^9}, 3.510965620835739*^9, {3.510980713272068*^9, 
   3.510980713412468*^9}, {3.510980751289335*^9, 3.5109807514297347`*^9}, {
   3.5109808130498433`*^9, 3.5109808621743293`*^9}, {3.5109855664036036`*^9, 
   3.5109855667624035`*^9}, {3.5109976790926123`*^9, 
   3.5109976792174125`*^9}, {3.511001058291355*^9, 3.511001058431755*^9}, {
   3.5110023185587797`*^9, 3.5110024376649895`*^9}, {3.5110673866620636`*^9, 
   3.511067386771264*^9}, {3.5110674313873425`*^9, 3.5110674526969795`*^9}, {
   3.511144014627281*^9, 3.511144014752082*^9}, {3.5111443615094905`*^9, 
   3.5111444172171884`*^9}, {3.5111444872457113`*^9, 
   3.5111444874173117`*^9}, {3.5111456221165047`*^9, 3.511145622912106*^9}, {
   3.5111469295080013`*^9, 3.5111469378540154`*^9}, {3.511151653352298*^9, 
   3.5111516692799263`*^9}, {3.511151703802787*^9, 3.511151709512397*^9}, {
   3.511151764393293*^9, 3.5111517990409536`*^9}, {3.5111527228745766`*^9, 
   3.5111527274921846`*^9}, {3.5111617369272847`*^9, 
   3.5111617373016853`*^9}, {3.5112282465694976`*^9, 
   3.5112282467566977`*^9}, {3.51122895378114*^9, 3.5112289539215403`*^9}, {
   3.511229358477051*^9, 3.511229397243119*^9}, {3.511229844729905*^9, 
   3.5112298448859053`*^9}, {3.5112299955977697`*^9, 3.511230027671426*^9}, {
   3.511230389014861*^9, 3.5112304126021023`*^9}, {3.511230529056307*^9, 
   3.5112305886172113`*^9}, {3.5112323789921603`*^9, 3.511232379179361*^9}, {
   3.511329531691909*^9, 3.5113295318791094`*^9}, {3.5113300860076857`*^9, 
   3.5113301632590213`*^9}, {3.5113302681380053`*^9, 3.511330268933607*^9}, {
   3.511330485617987*^9, 3.5113304862263885`*^9}, {3.5114070321661625`*^9, 
   3.5114070887786617`*^9}, {3.511407177605218*^9, 3.511407177886018*^9}, {
   3.511431574518587*^9, 3.511431579869396*^9}, {3.512710907830912*^9, 
   3.5127109337113576`*^9}, {3.512730678006273*^9, 3.5127307128567343`*^9}, 
   3.5127748485049033`*^9, {3.512774963820306*^9, 3.5127750423196435`*^9}}],

Cell[BoxData[
 Graphics3DBox[{Point3DBox[{0.75, 0.3, 0.75}], Point3DBox[{-0.75, 0.3, 0.75}],
    Point3DBox[{0.75, 0.3, -0.75}], Point3DBox[{-0.75, 0.3, -0.75}], 
   Point3DBox[{0.75, -5., 0.75}], Point3DBox[{-0.75, -5., 0.75}], 
   Point3DBox[{0.75, -5., -0.75}], Point3DBox[{-0.75, -5., -0.75}], 
   Line3DBox[{{0, -0.011029200748807984`, 0}, {0., -2.0776428428177893`, 
     0.}}], Line3DBox[{{-0.35000000000000003`, -0.0020420192067499487`, \
-0.35000000000000003`}, {-0.4586677677090375, -2.089751725367939, \
-0.1860749280929821}}], 
   Line3DBox[{{-0.35000000000000003`, -0.0020420192067499487`, 
    0.35000000000000003`}, {-0.1860749280929821, -2.089751725367939, 
    0.4586677677090375}}], 
   Line3DBox[{{0.35000000000000003`, -0.0020420192067499487`, 
    0.35000000000000003`}, {0.4586677677090375, -2.089751725367939, 
    0.1860749280929821}}], 
   Line3DBox[{{
    0.35000000000000003`, -0.0020420192067499487`, -0.35000000000000003`}, {
    0.1860749280929821, -2.089751725367939, -0.4586677677090375}}], 
   Line3DBox[{{-0.125, -2.084781140372976, -0.015}, {-0.125, \
-4.170793881317836, -0.015}}], 
   Line3DBox[{{-0.125, -2.084781140372976, 
    0.015}, {-0.125, -4.170793881317836, 0.015}}], 
   Line3DBox[{{0.125, -2.084781140372976, 0.015}, {0.125, -4.170793881317836, 
    0.015}}], 
   Line3DBox[{{0.125, -2.084781140372976, -0.015}, {
    0.125, -4.170793881317836, -0.015}}], 
   Line3DBox[{{-0.27, -2.085336573358779, -0.03}, {-0.27, \
-4.1777413570259565`, -0.03}}], 
   Line3DBox[{{-0.27, -2.085336573358779, 0.03}, {-0.27, -4.1777413570259565`,
     0.03}}], 
   Line3DBox[{{0.27, -2.085336573358779, 0.03}, {0.27, -4.1777413570259565`, 
    0.03}}], Line3DBox[{{0.27, -2.085336573358779, -0.03}, {
    0.27, -4.1777413570259565`, -0.03}}], 
   GeometricTransformation3DBox[
    GeometricTransformation3DBox[
     GeometricTransformation3DBox[{
       Polygon3DBox[{{-0.3, -2.0336720435665976`, -0.3}, {-0.3, \
-2.0336720435665976`, 0.3}, {-0.3, -2.1336720435665972`, 
        0.3}, {-0.3, -2.1336720435665972`, -0.3}}], 
       Polygon3DBox[{{0.3, -2.0336720435665976`, -0.3}, {
        0.3, -2.0336720435665976`, 0.3}, {0.3, -2.1336720435665972`, 0.3}, {
        0.3, -2.1336720435665972`, -0.3}}], 
       Polygon3DBox[{{-0.3, -2.1336720435665972`, -0.3}, {-0.3, \
-2.0336720435665976`, -0.3}, {0.3, -2.0336720435665976`, -0.3}, {
        0.3, -2.1336720435665972`, -0.3}}], 
       Polygon3DBox[{{-0.3, -2.1336720435665972`, 
        0.3}, {-0.3, -2.0336720435665976`, 0.3}, {0.3, -2.0336720435665976`, 
        0.3}, {0.3, -2.1336720435665972`, 0.3}}], 
       Polygon3DBox[{{-0.3, -2.1336720435665972`, -0.3}, {-0.3, \
-2.1336720435665972`, 0.3}, {0.3, -2.1336720435665972`, 0.3}, {
        0.3, -2.1336720435665972`, -0.3}}], 
       Polygon3DBox[{{-0.3, -2.0336720435665976`, -0.3}, {-0.3, \
-2.0336720435665976`, 0.3}, {0.3, -2.0336720435665976`, 0.3}, {
        0.3, -2.0336720435665976`, -0.3}}]}, {{{1., 0., 0.}, {0., 1., 0.}, {
       0., 0., 1.}}, {0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 
      1.}}, {0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 1.}}, {0., 
     0., 0.}}], 
   GeometricTransformation3DBox[
    GeometricTransformation3DBox[
     GeometricTransformation3DBox[{
       Polygon3DBox[{{-0.35000000000000003`, -2.0177937445746887`, \
-0.35000000000000003`}, {-0.35000000000000003`, -2.0177937445746887`, 
        0.35000000000000003`}, {-0.35000000000000003`, -2.167793744574689, 
        0.35000000000000003`}, {-0.35000000000000003`, -2.167793744574689, \
-0.35000000000000003`}}], 
       Polygon3DBox[{{
        0.35000000000000003`, -2.0177937445746887`, -0.35000000000000003`}, {
        0.35000000000000003`, -2.0177937445746887`, 0.35000000000000003`}, {
        0.35000000000000003`, -2.167793744574689, 0.35000000000000003`}, {
        0.35000000000000003`, -2.167793744574689, -0.35000000000000003`}}], 
       Polygon3DBox[{{-0.35000000000000003`, -2.167793744574689, \
-0.35000000000000003`}, {-0.35000000000000003`, -2.0177937445746887`, \
-0.35000000000000003`}, {
        0.35000000000000003`, -2.0177937445746887`, -0.35000000000000003`}, {
        0.35000000000000003`, -2.167793744574689, -0.35000000000000003`}}], 
       Polygon3DBox[{{-0.35000000000000003`, -2.167793744574689, 
        0.35000000000000003`}, {-0.35000000000000003`, -2.0177937445746887`, 
        0.35000000000000003`}, {0.35000000000000003`, -2.0177937445746887`, 
        0.35000000000000003`}, {0.35000000000000003`, -2.167793744574689, 
        0.35000000000000003`}}]}, {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 
       1.}}, {0., 0., 0.}}], {{{0.9210609940028851, 0., 0.3894183423086505}, {
      0., 1., 0.}, {-0.3894183423086505, 0., 0.9210609940028851}}, {0., 0., 
      0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 1.}}, {0., 0., 0.}}], 
   GeometricTransformation3DBox[
    GeometricTransformation3DBox[
     GeometricTransformation3DBox[{
       Polygon3DBox[{{0.11888206453689419`, -4.134275853827346, 0.05}, {
        0.11888206453689419`, -4.134275853827346, -0.05}, {
        0.10112712429686843`, -4.099429821587656, -0.05}, {
        0.10112712429686843`, -4.099429821587656, 0.05}}], 
       Polygon3DBox[{{0.10112712429686843`, -4.099429821587656, 0.05}, {
        0.10112712429686843`, -4.099429821587656, -0.05}, {
        0.07347315653655914, -4.071775853827346, -0.05}, {
        0.07347315653655914, -4.071775853827346, 0.05}}], 
       Polygon3DBox[{{0.07347315653655914, -4.071775853827346, 0.05}, {
        0.07347315653655914, -4.071775853827346, -0.05}, {
        0.03862712429686843, -4.054020913587321, -0.05}, {
        0.03862712429686843, -4.054020913587321, 0.05}}], 
       Polygon3DBox[{{0.03862712429686843, -4.054020913587321, 0.05}, {
        0.03862712429686843, -4.054020913587321, -0.05}, {
        0., -4.047902978124215, -0.05}, {0., -4.047902978124215, 0.05}}], 
       Polygon3DBox[{{0., -4.047902978124215, 0.05}, {
        0., -4.047902978124215, -0.05}, {-0.03862712429686843, \
-4.054020913587321, -0.05}, {-0.03862712429686843, -4.054020913587321, 
        0.05}}], 
       Polygon3DBox[{{-0.03862712429686843, -4.054020913587321, 
        0.05}, {-0.03862712429686843, -4.054020913587321, -0.05}, \
{-0.07347315653655914, -4.071775853827346, -0.05}, {-0.07347315653655914, \
-4.071775853827346, 0.05}}], 
       Polygon3DBox[{{-0.07347315653655914, -4.071775853827346, 
        0.05}, {-0.07347315653655914, -4.071775853827346, -0.05}, \
{-0.10112712429686843`, -4.099429821587656, -0.05}, {-0.10112712429686843`, \
-4.099429821587656, 0.05}}], 
       Polygon3DBox[{{-0.10112712429686843`, -4.099429821587656, 
        0.05}, {-0.10112712429686843`, -4.099429821587656, -0.05}, \
{-0.11888206453689419`, -4.134275853827346, -0.05}, {-0.11888206453689419`, \
-4.134275853827346, 0.05}}], 
       Polygon3DBox[{{-0.11888206453689419`, -4.134275853827346, 
        0.05}, {-0.11888206453689419`, -4.134275853827346, -0.05}, {-0.125, \
-4.172902978124215, -0.05}, {-0.125, -4.172902978124215, 0.05}}], 
       Polygon3DBox[{{-0.125, -4.172902978124215, 
        0.05}, {-0.125, -4.172902978124215, -0.05}, {-0.11888206453689419`, \
-4.211530102421084, -0.05}, {-0.11888206453689419`, -4.211530102421084, 
        0.05}}], 
       Polygon3DBox[{{-0.11888206453689419`, -4.211530102421084, 
        0.05}, {-0.11888206453689419`, -4.211530102421084, -0.05}, \
{-0.10112712429686843`, -4.246376134660774, -0.05}, {-0.10112712429686843`, \
-4.246376134660774, 0.05}}], 
       Polygon3DBox[{{-0.10112712429686843`, -4.246376134660774, 
        0.05}, {-0.10112712429686843`, -4.246376134660774, -0.05}, \
{-0.07347315653655914, -4.274030102421084, -0.05}, {-0.07347315653655914, \
-4.274030102421084, 0.05}}], 
       Polygon3DBox[{{-0.07347315653655914, -4.274030102421084, 
        0.05}, {-0.07347315653655914, -4.274030102421084, -0.05}, \
{-0.03862712429686843, -4.291785042661109, -0.05}, {-0.03862712429686843, \
-4.291785042661109, 0.05}}], 
       Polygon3DBox[{{-0.03862712429686843, -4.291785042661109, 
        0.05}, {-0.03862712429686843, -4.291785042661109, -0.05}, {
        0., -4.297902978124215, -0.05}, {0., -4.297902978124215, 0.05}}], 
       Polygon3DBox[{{0., -4.297902978124215, 0.05}, {
        0., -4.297902978124215, -0.05}, {
        0.03862712429686843, -4.291785042661109, -0.05}, {
        0.03862712429686843, -4.291785042661109, 0.05}}], 
       Polygon3DBox[{{0.03862712429686843, -4.291785042661109, 0.05}, {
        0.03862712429686843, -4.291785042661109, -0.05}, {
        0.07347315653655914, -4.274030102421084, -0.05}, {
        0.07347315653655914, -4.274030102421084, 0.05}}], 
       Polygon3DBox[{{0.07347315653655914, -4.274030102421084, 0.05}, {
        0.07347315653655914, -4.274030102421084, -0.05}, {
        0.10112712429686843`, -4.246376134660774, -0.05}, {
        0.10112712429686843`, -4.246376134660774, 0.05}}], 
       Polygon3DBox[{{0.10112712429686843`, -4.246376134660774, 0.05}, {
        0.10112712429686843`, -4.246376134660774, -0.05}, {
        0.11888206453689419`, -4.211530102421084, -0.05}, {
        0.11888206453689419`, -4.211530102421084, 0.05}}], 
       Polygon3DBox[{{0.11888206453689419`, -4.211530102421084, 0.05}, {
        0.11888206453689419`, -4.211530102421084, -0.05}, {
        0.125, -4.172902978124215, -0.05}, {0.125, -4.172902978124215, 
        0.05}}], 
       Polygon3DBox[{{0.125, -4.172902978124215, 0.05}, {
        0.125, -4.172902978124215, -0.05}, {
        0.11888206453689419`, -4.134275853827346, -0.05}, {
        0.11888206453689419`, -4.134275853827346, 0.05}}]}, {{{1., 0., 0.}, {
       0., 1., 0.}, {0., 0., 1.}}, {0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 
      0.}, {0., 0., 1.}}, {0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 
     0., 1.}}, {0., 0., 0.}}], 
   GeometricTransformation3DBox[
    GeometricTransformation3DBox[
     GeometricTransformation3DBox[{
       Polygon3DBox[{{0.2567852593996915, -4.096971298336903, 0.1}, {
        0.2567852593996915, -4.096971298336903, -0.1}, {
        0.21843458848123581`, -4.021703868699171, -0.1}, {
        0.21843458848123581`, -4.021703868699171, 0.1}}], 
       Polygon3DBox[{{0.21843458848123581`, -4.021703868699171, 0.1}, {
        0.21843458848123581`, -4.021703868699171, -0.1}, {
        0.15870201811896775`, -3.9619712983369024`, -0.1}, {
        0.15870201811896775`, -3.9619712983369024`, 0.1}}], 
       Polygon3DBox[{{0.15870201811896775`, -3.9619712983369024`, 0.1}, {
        0.15870201811896775`, -3.9619712983369024`, -0.1}, {
        0.08343458848123582, -3.923620627418447, -0.1}, {
        0.08343458848123582, -3.923620627418447, 0.1}}], 
       Polygon3DBox[{{0.08343458848123582, -3.923620627418447, 0.1}, {
        0.08343458848123582, -3.923620627418447, -0.1}, {
        0., -3.9104058868181384`, -0.1}, {0., -3.9104058868181384`, 0.1}}], 
       Polygon3DBox[{{0., -3.9104058868181384`, 0.1}, {
        0., -3.9104058868181384`, -0.1}, {-0.08343458848123582, \
-3.923620627418447, -0.1}, {-0.08343458848123582, -3.923620627418447, 0.1}}], 
       Polygon3DBox[{{-0.08343458848123582, -3.923620627418447, 
        0.1}, {-0.08343458848123582, -3.923620627418447, -0.1}, \
{-0.15870201811896775`, -3.9619712983369024`, -0.1}, {-0.15870201811896775`, \
-3.9619712983369024`, 0.1}}], 
       Polygon3DBox[{{-0.15870201811896775`, -3.9619712983369024`, 
        0.1}, {-0.15870201811896775`, -3.9619712983369024`, -0.1}, \
{-0.21843458848123581`, -4.021703868699171, -0.1}, {-0.21843458848123581`, \
-4.021703868699171, 0.1}}], 
       Polygon3DBox[{{-0.21843458848123581`, -4.021703868699171, 
        0.1}, {-0.21843458848123581`, -4.021703868699171, -0.1}, \
{-0.2567852593996915, -4.096971298336903, -0.1}, {-0.2567852593996915, \
-4.096971298336903, 0.1}}], 
       Polygon3DBox[{{-0.2567852593996915, -4.096971298336903, 
        0.1}, {-0.2567852593996915, -4.096971298336903, -0.1}, {-0.27, \
-4.180405886818138, -0.1}, {-0.27, -4.180405886818138, 0.1}}], 
       Polygon3DBox[{{-0.27, -4.180405886818138, 
        0.1}, {-0.27, -4.180405886818138, -0.1}, {-0.2567852593996915, \
-4.263840475299374, -0.1}, {-0.2567852593996915, -4.263840475299374, 0.1}}], 
       Polygon3DBox[{{-0.2567852593996915, -4.263840475299374, 
        0.1}, {-0.2567852593996915, -4.263840475299374, -0.1}, \
{-0.21843458848123581`, -4.339107904937106, -0.1}, {-0.21843458848123581`, \
-4.339107904937106, 0.1}}], 
       Polygon3DBox[{{-0.21843458848123581`, -4.339107904937106, 
        0.1}, {-0.21843458848123581`, -4.339107904937106, -0.1}, \
{-0.15870201811896775`, -4.398840475299374, -0.1}, {-0.15870201811896775`, \
-4.398840475299374, 0.1}}], 
       Polygon3DBox[{{-0.15870201811896775`, -4.398840475299374, 
        0.1}, {-0.15870201811896775`, -4.398840475299374, -0.1}, \
{-0.08343458848123582, -4.43719114621783, -0.1}, {-0.08343458848123582, \
-4.43719114621783, 0.1}}], 
       Polygon3DBox[{{-0.08343458848123582, -4.43719114621783, 
        0.1}, {-0.08343458848123582, -4.43719114621783, -0.1}, {
        0., -4.450405886818139, -0.1}, {0., -4.450405886818139, 0.1}}], 
       Polygon3DBox[{{0., -4.450405886818139, 0.1}, {
        0., -4.450405886818139, -0.1}, {
        0.08343458848123582, -4.43719114621783, -0.1}, {
        0.08343458848123582, -4.43719114621783, 0.1}}], 
       Polygon3DBox[{{0.08343458848123582, -4.43719114621783, 0.1}, {
        0.08343458848123582, -4.43719114621783, -0.1}, {
        0.15870201811896775`, -4.398840475299374, -0.1}, {
        0.15870201811896775`, -4.398840475299374, 0.1}}], 
       Polygon3DBox[{{0.15870201811896775`, -4.398840475299374, 0.1}, {
        0.15870201811896775`, -4.398840475299374, -0.1}, {
        0.21843458848123581`, -4.339107904937106, -0.1}, {
        0.21843458848123581`, -4.339107904937106, 0.1}}], 
       Polygon3DBox[{{0.21843458848123581`, -4.339107904937106, 0.1}, {
        0.21843458848123581`, -4.339107904937106, -0.1}, {
        0.2567852593996915, -4.263840475299374, -0.1}, {
        0.2567852593996915, -4.263840475299374, 0.1}}], 
       Polygon3DBox[{{0.2567852593996915, -4.263840475299374, 0.1}, {
        0.2567852593996915, -4.263840475299374, -0.1}, {
        0.27, -4.180405886818138, -0.1}, {0.27, -4.180405886818138, 0.1}}], 
       Polygon3DBox[{{0.27, -4.180405886818138, 0.1}, {
        0.27, -4.180405886818138, -0.1}, {
        0.2567852593996915, -4.096971298336903, -0.1}, {
        0.2567852593996915, -4.096971298336903, 0.1}}]}, {{{1., 0., 0.}, {0., 
       1., 0.}, {0., 0., 1.}}, {0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {
      0., 0., 1.}}, {0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 
     1.}}, {0., 0., 0.}}], 
   GeometricTransformation3DBox[
    GeometricTransformation3DBox[
     GeometricTransformation3DBox[{
       Polygon3DBox[{{0.5706339097770922, 0, 0.1854101966249685}, {
         0.1902113032590307, 0, 0.06180339887498949}, {
         0.1618033988749895, 0, 0.11755705045849463`}, {
         0.48541019662496854`, 0, 0.35267115137548394`}}], 
       Polygon3DBox[{{0.48541019662496854`, 0, 0.35267115137548394`}, {
         0.1618033988749895, 0, 0.11755705045849463`}, {
         0.11755705045849463`, 0, 0.1618033988749895}, {
         0.35267115137548394`, 0, 0.48541019662496854`}}], 
       Polygon3DBox[{{0.35267115137548394`, 0, 0.48541019662496854`}, {
         0.11755705045849463`, 0, 0.1618033988749895}, {
         0.06180339887498949, 0, 0.1902113032590307}, {
         0.1854101966249685, 0, 0.5706339097770922}}], 
       Polygon3DBox[{{0.1854101966249685, 0, 0.5706339097770922}, {
         0.06180339887498949, 0, 0.1902113032590307}, {0., 0, 0.2}, {
         0., 0, 0.6000000000000001}}], 
       Polygon3DBox[{{0., 0, 0.6000000000000001}, {
         0., 0, 0.2}, {-0.06180339887498949, 0, 
          0.1902113032590307}, {-0.1854101966249685, 0, 0.5706339097770922}}],
        Polygon3DBox[{{-0.1854101966249685, 0, 
          0.5706339097770922}, {-0.06180339887498949, 0, 
          0.1902113032590307}, {-0.11755705045849463`, 0, 
          0.1618033988749895}, {-0.35267115137548394`, 0, 
          0.48541019662496854`}}], 
       Polygon3DBox[{{-0.35267115137548394`, 0, 
          0.48541019662496854`}, {-0.11755705045849463`, 0, 
          0.1618033988749895}, {-0.1618033988749895, 0, 
          0.11755705045849463`}, {-0.48541019662496854`, 0, 
          0.35267115137548394`}}], 
       Polygon3DBox[{{-0.48541019662496854`, 0, 
          0.35267115137548394`}, {-0.1618033988749895, 0, 
          0.11755705045849463`}, {-0.1902113032590307, 0, 
          0.06180339887498949}, {-0.5706339097770922, 0, 
          0.1854101966249685}}], 
       Polygon3DBox[{{-0.5706339097770922, 0, 
          0.1854101966249685}, {-0.1902113032590307, 0, 
          0.06180339887498949}, {-0.2, 0, 0.}, {-0.6000000000000001, 0, 0.}}],
        Polygon3DBox[{{-0.6000000000000001, 0, 0.}, {-0.2, 0, 
          0.}, {-0.1902113032590307, 
          0, -0.06180339887498949}, {-0.5706339097770922, 
          0, -0.1854101966249685}}], 
       Polygon3DBox[{{-0.5706339097770922, 
          0, -0.1854101966249685}, {-0.1902113032590307, 
          0, -0.06180339887498949}, {-0.1618033988749895, 
          0, -0.11755705045849463`}, {-0.48541019662496854`, 
          0, -0.35267115137548394`}}], 
       Polygon3DBox[{{-0.48541019662496854`, 
          0, -0.35267115137548394`}, {-0.1618033988749895, 
          0, -0.11755705045849463`}, {-0.11755705045849463`, 
          0, -0.1618033988749895}, {-0.35267115137548394`, 
          0, -0.48541019662496854`}}], 
       Polygon3DBox[{{-0.35267115137548394`, 
          0, -0.48541019662496854`}, {-0.11755705045849463`, 
          0, -0.1618033988749895}, {-0.06180339887498949, 
          0, -0.1902113032590307}, {-0.1854101966249685, 
          0, -0.5706339097770922}}], 
       Polygon3DBox[{{-0.1854101966249685, 
          0, -0.5706339097770922}, {-0.06180339887498949, 
          0, -0.1902113032590307}, {0., 0, -0.2}, {
         0., 0, -0.6000000000000001}}], 
       Polygon3DBox[{{0., 0, -0.6000000000000001}, {0., 0, -0.2}, {
         0.06180339887498949, 0, -0.1902113032590307}, {
         0.1854101966249685, 0, -0.5706339097770922}}], 
       Polygon3DBox[{{0.1854101966249685, 0, -0.5706339097770922}, {
         0.06180339887498949, 0, -0.1902113032590307}, {
         0.11755705045849463`, 0, -0.1618033988749895}, {
         0.35267115137548394`, 0, -0.48541019662496854`}}], 
       Polygon3DBox[{{0.35267115137548394`, 0, -0.48541019662496854`}, {
         0.11755705045849463`, 0, -0.1618033988749895}, {
         0.1618033988749895, 0, -0.11755705045849463`}, {
         0.48541019662496854`, 0, -0.35267115137548394`}}], 
       Polygon3DBox[{{0.48541019662496854`, 0, -0.35267115137548394`}, {
         0.1618033988749895, 0, -0.11755705045849463`}, {
         0.1902113032590307, 0, -0.06180339887498949}, {
         0.5706339097770922, 0, -0.1854101966249685}}], 
       Polygon3DBox[{{0.5706339097770922, 0, -0.1854101966249685}, {
         0.1902113032590307, 0, -0.06180339887498949}, {0.2, 0, 0.}, {
         0.6000000000000001, 0, 0.}}], 
       Polygon3DBox[{{0.6000000000000001, 0, 0.}, {0.2, 0, 0.}, {
         0.1902113032590307, 0, 0.06180339887498949}, {
         0.5706339097770922, 0, 0.1854101966249685}}]}, {{{1, 0, 0}, {0, 1, 
       0}, {0, 0, 1}}, {0, 0, 0}}], {{{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}, {0, 0,
       0}}], {{{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}, {0, 0, 0}}], 
   GeometricTransformation3DBox[
    GeometricTransformation3DBox[
     GeometricTransformation3DBox[{
       Polygon3DBox[{{0.019021130325903073`, -0.006029200748807984, 
        0.006180339887498949}, {0.019021130325903073`, -0.016029200748807985`,
         0.006180339887498949}, {
        0.016180339887498948`, -0.016029200748807985`, 
        0.011755705045849463`}, {0.016180339887498948`, -0.006029200748807984,
         0.011755705045849463`}}], 
       Polygon3DBox[{{0.016180339887498948`, -0.006029200748807984, 
        0.011755705045849463`}, {
        0.016180339887498948`, -0.016029200748807985`, 
        0.011755705045849463`}, {
        0.011755705045849463`, -0.016029200748807985`, 
        0.016180339887498948`}, {0.011755705045849463`, -0.006029200748807984,
         0.016180339887498948`}}], 
       Polygon3DBox[{{0.011755705045849463`, -0.006029200748807984, 
        0.016180339887498948`}, {
        0.011755705045849463`, -0.016029200748807985`, 
        0.016180339887498948`}, {0.006180339887498949, -0.016029200748807985`,
         0.019021130325903073`}, {0.006180339887498949, -0.006029200748807984,
         0.019021130325903073`}}], 
       Polygon3DBox[{{0.006180339887498949, -0.006029200748807984, 
        0.019021130325903073`}, {0.006180339887498949, -0.016029200748807985`,
         0.019021130325903073`}, {0., -0.016029200748807985`, 0.02}, {
        0., -0.006029200748807984, 0.02}}], 
       Polygon3DBox[{{0., -0.006029200748807984, 0.02}, {
        0., -0.016029200748807985`, 
        0.02}, {-0.006180339887498949, -0.016029200748807985`, 
        0.019021130325903073`}, {-0.006180339887498949, -0.006029200748807984,
         0.019021130325903073`}}], 
       Polygon3DBox[{{-0.006180339887498949, -0.006029200748807984, 
        0.019021130325903073`}, {-0.006180339887498949, \
-0.016029200748807985`, 
        0.019021130325903073`}, {-0.011755705045849463`, \
-0.016029200748807985`, 
        0.016180339887498948`}, {-0.011755705045849463`, \
-0.006029200748807984, 0.016180339887498948`}}], 
       Polygon3DBox[{{-0.011755705045849463`, -0.006029200748807984, 
        0.016180339887498948`}, {-0.011755705045849463`, \
-0.016029200748807985`, 
        0.016180339887498948`}, {-0.016180339887498948`, \
-0.016029200748807985`, 
        0.011755705045849463`}, {-0.016180339887498948`, \
-0.006029200748807984, 0.011755705045849463`}}], 
       Polygon3DBox[{{-0.016180339887498948`, -0.006029200748807984, 
        0.011755705045849463`}, {-0.016180339887498948`, \
-0.016029200748807985`, 
        0.011755705045849463`}, {-0.019021130325903073`, \
-0.016029200748807985`, 
        0.006180339887498949}, {-0.019021130325903073`, -0.006029200748807984,
         0.006180339887498949}}], 
       Polygon3DBox[{{-0.019021130325903073`, -0.006029200748807984, 
        0.006180339887498949}, {-0.019021130325903073`, \
-0.016029200748807985`, 0.006180339887498949}, {-0.02, -0.016029200748807985`,
         0.}, {-0.02, -0.006029200748807984, 0.}}], 
       Polygon3DBox[{{-0.02, -0.006029200748807984, 
        0.}, {-0.02, -0.016029200748807985`, 
        0.}, {-0.019021130325903073`, -0.016029200748807985`, \
-0.006180339887498949}, {-0.019021130325903073`, -0.006029200748807984, \
-0.006180339887498949}}], 
       Polygon3DBox[{{-0.019021130325903073`, -0.006029200748807984, \
-0.006180339887498949}, {-0.019021130325903073`, -0.016029200748807985`, \
-0.006180339887498949}, {-0.016180339887498948`, -0.016029200748807985`, \
-0.011755705045849463`}, {-0.016180339887498948`, -0.006029200748807984, \
-0.011755705045849463`}}], 
       Polygon3DBox[{{-0.016180339887498948`, -0.006029200748807984, \
-0.011755705045849463`}, {-0.016180339887498948`, -0.016029200748807985`, \
-0.011755705045849463`}, {-0.011755705045849463`, -0.016029200748807985`, \
-0.016180339887498948`}, {-0.011755705045849463`, -0.006029200748807984, \
-0.016180339887498948`}}], 
       Polygon3DBox[{{-0.011755705045849463`, -0.006029200748807984, \
-0.016180339887498948`}, {-0.011755705045849463`, -0.016029200748807985`, \
-0.016180339887498948`}, {-0.006180339887498949, -0.016029200748807985`, \
-0.019021130325903073`}, {-0.006180339887498949, -0.006029200748807984, \
-0.019021130325903073`}}], 
       Polygon3DBox[{{-0.006180339887498949, -0.006029200748807984, \
-0.019021130325903073`}, {-0.006180339887498949, -0.016029200748807985`, \
-0.019021130325903073`}, {0., -0.016029200748807985`, -0.02}, {
        0., -0.006029200748807984, -0.02}}], 
       Polygon3DBox[{{0., -0.006029200748807984, -0.02}, {
        0., -0.016029200748807985`, -0.02}, {
        0.006180339887498949, -0.016029200748807985`, \
-0.019021130325903073`}, {
        0.006180339887498949, -0.006029200748807984, \
-0.019021130325903073`}}], 
       Polygon3DBox[{{
        0.006180339887498949, -0.006029200748807984, -0.019021130325903073`}, \
{0.006180339887498949, -0.016029200748807985`, -0.019021130325903073`}, {
        0.011755705045849463`, -0.016029200748807985`, \
-0.016180339887498948`}, {
        0.011755705045849463`, -0.006029200748807984, \
-0.016180339887498948`}}], 
       Polygon3DBox[{{
        0.011755705045849463`, -0.006029200748807984, \
-0.016180339887498948`}, {
        0.011755705045849463`, -0.016029200748807985`, \
-0.016180339887498948`}, {
        0.016180339887498948`, -0.016029200748807985`, \
-0.011755705045849463`}, {
        0.016180339887498948`, -0.006029200748807984, \
-0.011755705045849463`}}], 
       Polygon3DBox[{{
        0.016180339887498948`, -0.006029200748807984, \
-0.011755705045849463`}, {
        0.016180339887498948`, -0.016029200748807985`, \
-0.011755705045849463`}, {
        0.019021130325903073`, -0.016029200748807985`, \
-0.006180339887498949}, {
        0.019021130325903073`, -0.006029200748807984, \
-0.006180339887498949}}], 
       Polygon3DBox[{{
        0.019021130325903073`, -0.006029200748807984, -0.006180339887498949}, \
{0.019021130325903073`, -0.016029200748807985`, -0.006180339887498949}, {
        0.02, -0.016029200748807985`, 0.}, {0.02, -0.006029200748807984, 
        0.}}], Polygon3DBox[{{0.02, -0.006029200748807984, 0.}, {
        0.02, -0.016029200748807985`, 0.}, {
        0.019021130325903073`, -0.016029200748807985`, 
        0.006180339887498949}, {0.019021130325903073`, -0.006029200748807984, 
        0.006180339887498949}}]}, {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 
       1.}}, {0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 1.}}, {0.,
       0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 1.}}, {0., 0., 0.}}]},
  AutomaticImageSize->True,
  ImageSize->{230.0223823701776, 531.3438179097376},
  ViewPoint->{-1.8637746307136254`, 1.1118173341348814`, -2.5961907367189907`},
  ViewVertical->{-0.8770344937746104, 
   0.8690013656791519, -1.5124350424338513`}]], "Output",
 CellChangeTimes->{
  3.5111617395012894`*^9, {3.511228226351862*^9, 3.511228247177899*^9}, 
   3.511228383444138*^9, 3.511228597258114*^9, 3.511228747564378*^9, 
   3.5112287993252687`*^9, 3.5112289130494685`*^9, {3.511228945216725*^9, 
   3.511228955434743*^9}, 3.5112292060803833`*^9, {3.5112293442498255`*^9, 
   3.51122939789832*^9}, 3.5112296699939976`*^9, 3.511229845541106*^9, {
   3.5112299408728733`*^9, 3.5112299460676827`*^9}, {3.5112300007925787`*^9, 
   3.5112300283266273`*^9}, {3.5112303683760242`*^9, 3.511230413304103*^9}, {
   3.5112305166542845`*^9, 3.5112305897248135`*^9}, 3.511231829458991*^9, 
   3.5112323163736506`*^9, 3.511232379912562*^9, 3.511232425199442*^9, 
   3.5112324586459007`*^9, 3.511232492997161*^9, 3.511232537472839*^9, 
   3.511232581215316*^9, 3.5112326569066486`*^9, 3.5112328454329796`*^9, 
   3.5112329079110894`*^9, 3.5112332290844536`*^9, 3.511329534437514*^9, 
   3.511329918135791*^9, {3.511330019067968*^9, 3.511330036477598*^9}, {
   3.5113300876612883`*^9, 3.511330164990624*^9}, 3.511330270368809*^9, 
   3.511330439254706*^9, 3.511330487630391*^9, 3.5113306319774446`*^9, 
   3.5113329875503817`*^9, 3.5114066981539755`*^9, 3.5114067825033236`*^9, 
   3.5114069498136177`*^9, {3.511407033226964*^9, 3.5114070948626723`*^9}, 
   3.51140717883762*^9, {3.511431565876172*^9, 3.511431582537001*^9}, 
   3.511490059584117*^9, 3.5114943360824537`*^9, 3.5114960912229385`*^9, 
   3.5123557887907486`*^9, {3.512355920904023*^9, 3.5123559369096513`*^9}, 
   3.51235600480097*^9, 3.512710814199548*^9, {3.512710909468915*^9, 
   3.512710934569359*^9}, 3.5127123264782057`*^9, 3.512712518342942*^9, 
   3.512712612567108*^9, {3.512730656244234*^9, 3.512730713636735*^9}, 
   3.5127329316658335`*^9, {3.5127748420464916`*^9, 3.5127748489105043`*^9}, {
   3.512774915460221*^9, 3.5127750431464453`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Transfer Function Plot", "Section",
 CellChangeTimes->{{3.510629600074376*^9, 3.5106296028355813`*^9}, {
  3.510922302451619*^9, 3.5109223035016794`*^9}}],

Cell[CellGroupData[{

Cell["Frequency List", "Subsection",
 CellChangeTimes->{{3.505457309098371*^9, 3.5054573134988823`*^9}, {
  3.505457392369238*^9, 3.505457393435096*^9}, {3.507903438769166*^9, 
  3.507903442019352*^9}, {3.5079450761978855`*^9, 3.5079450876015053`*^9}, {
  3.507945118302359*^9, 3.507945140813199*^9}, {3.5109227629829597`*^9, 
  3.510922763595995*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"fmin", "=", 
    RowBox[{"1", " ", 
     SuperscriptBox["10", 
      RowBox[{"-", "1"}]], " ", "hertz"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"minimum", " ", "frequency"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"fmax", "=", 
    RowBox[{"5", " ", 
     SuperscriptBox["10", "1"], "hertz"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"maximum", " ", "freqnency"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"ngrid", "=", "2001"}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"number", " ", "of", " ", "grids"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5079437907558236`*^9, 3.5079439830729613`*^9}, {
   3.5079440622431*^9, 3.5079440624147005`*^9}, {3.5079440956739593`*^9, 
   3.507944096875161*^9}, 3.5079441778237033`*^9, {3.5079445464225545`*^9, 
   3.5079445466097546`*^9}, {3.50794501640298*^9, 3.5079450202405868`*^9}, 
   3.5079450559646497`*^9, 3.507945701790184*^9, {3.507945743301857*^9, 
   3.5079457477166643`*^9}, {3.508056511433575*^9, 3.508056511776776*^9}, {
   3.5080566504610195`*^9, 3.50805665075742*^9}, {3.508117933375839*^9, 
   3.5081179356066427`*^9}, {3.5082143648046265`*^9, 3.508214371684239*^9}, {
   3.510405201938513*^9, 3.510405203850622*^9}, {3.5104081570565357`*^9, 
   3.510408158312608*^9}, {3.5104088030074825`*^9, 3.510408811078944*^9}, {
   3.5104129367289176`*^9, 3.5104129458664403`*^9}, {3.5104455061152945`*^9, 
   3.5104455111697035`*^9}, {3.5105372720362353`*^9, 
   3.5105372721766357`*^9}, {3.5105373756672173`*^9, 3.510537375854418*^9}, 
   3.5106333737914104`*^9, {3.5106358171925025`*^9, 3.510635817473303*^9}, {
   3.5109228206962614`*^9, 3.5109228592864685`*^9}, 3.5109676607265215`*^9, {
   3.5110038763463163`*^9, 3.5110038811043243`*^9}, {3.5113312825453873`*^9, 
   3.511331283746589*^9}, {3.5114943820713344`*^9, 3.5114943893565474`*^9}, {
   3.5114955948018646`*^9, 3.5114955949422646`*^9}, {3.511497703881379*^9, 
   3.5114977039749794`*^9}, {3.512727423163353*^9, 3.512727491881474*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"freq", "=", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"imin", ",", "imax", ",", "dgrid"}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"imin", "=", 
        RowBox[{"Log", "[", 
         RowBox[{"10", ",", "fmin"}], "]"}]}], ";", 
       RowBox[{"imax", "=", 
        RowBox[{"Log", "[", 
         RowBox[{"10", ",", "fmax"}], "]"}]}], ";", 
       RowBox[{"dgrid", "=", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"imax", "-", "imin"}], ")"}], "/", 
         RowBox[{"(", 
          RowBox[{"ngrid", "-", "1"}], ")"}]}]}], ";", "\n", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"10", "^", "i"}], ",", 
         RowBox[{"{", 
          RowBox[{"i", ",", "imin", ",", "imax", ",", "dgrid"}], "}"}]}], 
        "]"}]}]}], "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Frequency", " ", "list"}], "*)"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5079437907558236`*^9, 3.5079439830729613`*^9}, {
   3.5079440622431*^9, 3.5079440624147005`*^9}, {3.5079440956739593`*^9, 
   3.507944096875161*^9}, 3.5079441778237033`*^9, {3.5079445464225545`*^9, 
   3.5079445466097546`*^9}, {3.50794501640298*^9, 3.5079450202405868`*^9}, 
   3.5079450559646497`*^9, 3.507945701790184*^9, {3.507945743301857*^9, 
   3.5079457477166643`*^9}, {3.508056511433575*^9, 3.508056511776776*^9}, {
   3.5080566504610195`*^9, 3.50805665075742*^9}, {3.508117933375839*^9, 
   3.5081179356066427`*^9}, {3.5082143648046265`*^9, 3.508214371684239*^9}, {
   3.510405201938513*^9, 3.510405203850622*^9}, {3.5104081570565357`*^9, 
   3.510408158312608*^9}, {3.5104088030074825`*^9, 3.510408811078944*^9}, {
   3.5104129367289176`*^9, 3.5104129458664403`*^9}, {3.5104455061152945`*^9, 
   3.5104455111697035`*^9}, {3.5105372720362353`*^9, 
   3.5105372721766357`*^9}, {3.5105373756672173`*^9, 3.510537375854418*^9}, 
   3.5106333737914104`*^9, {3.5106358171925025`*^9, 3.510635817473303*^9}, {
   3.5109228206962614`*^9, 3.5109228592864685`*^9}, 3.5109676607265215`*^9, {
   3.5110038763463163`*^9, 3.5110038811043243`*^9}, {3.5113312825453873`*^9, 
   3.511331283746589*^9}, {3.5114943820713344`*^9, 3.5114943893565474`*^9}, {
   3.5114955948018646`*^9, 3.5114955949422646`*^9}, {3.511497703881379*^9, 
   3.5114977039749794`*^9}, {3.512727423163353*^9, 3.512727540428759*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Actuator Definition", "Subsection",
 CellChangeTimes->{{3.5109654612318583`*^9, 3.510965462635861*^9}, {
  3.5109690024068785`*^9, 3.510969004138481*^9}}],

Cell[CellGroupData[{

Cell["Actuator Name List", "Subsubsection",
 CellChangeTimes->{{3.5113307936249285`*^9, 3.5113308073529525`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"allacts", "=", 
    RowBox[{"{", 
     RowBox[{"aTM1", ",", "aTM2", ",", "aTM3", ",", "aTM4"}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5109727782843103`*^9, 3.5109728075967617`*^9}, 
   3.51133080025494*^9, {3.5113912910255685`*^9, 3.5113912956275764`*^9}, {
   3.5114124581364727`*^9, 3.511412463674483*^9}, {3.5114130712487497`*^9, 
   3.5114130854447746`*^9}, 3.5127114692567*^9, 3.5127751682898645`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making Actuator List", "Subsubsection",
 CellChangeTimes->{{3.5113308136397634`*^9, 3.511330818600572*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"actuatorlist", "=", 
    RowBox[{"Join", "[", "\[IndentingNewLine]", 
     RowBox[{"Table", "[", 
      RowBox[{"(*", "TM", "*)"}], "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{"posTM", ",", 
         RowBox[{"(*", 
          RowBox[{
          "coordinate", " ", "list", " ", "of", " ", "actuated", " ", 
           "body"}], "*)"}], "\[IndentingNewLine]", 
         RowBox[{"posTMa", "[", 
          RowBox[{"[", "n", "]"}], "]"}], ",", 
         RowBox[{"(*", 
          RowBox[{
          "position", " ", "of", " ", "actuator", " ", "at", " ", "actuated", 
           " ", "body"}], "*)"}], "\[IndentingNewLine]", "posRM", ",", 
         RowBox[{"(*", 
          RowBox[{
          "coordinate", " ", "list", " ", "of", " ", "recoil", " ", "body"}], 
          "*)"}], "\[IndentingNewLine]", 
         RowBox[{"posRMa", "[", 
          RowBox[{"[", "n", "]"}], "]"}], ",", 
         RowBox[{"(*", 
          RowBox[{
          "position", " ", "of", " ", "actuator", " ", "at", " ", "recoil", 
           " ", "body"}], "*)"}], "\[IndentingNewLine]", 
         RowBox[{"dirTMa", "[", 
          RowBox[{"[", "n", "]"}], "]"}]}], 
        RowBox[{"(*", 
         RowBox[{
         "working", " ", "direction", " ", "defined", " ", "at", " ", 
          "recoil", " ", "body"}], "*)"}], "\[IndentingNewLine]", "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"n", ",", "4"}], "}"}]}], "\[IndentingNewLine]", "]"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5109654818394947`*^9, 3.510965501292729*^9}, {
   3.510967724749034*^9, 3.510967731956247*^9}, {3.510969081296217*^9, 
   3.5109691312787046`*^9}, {3.5109691764251842`*^9, 3.510969496490946*^9}, {
   3.510969534071412*^9, 3.5109695354442143`*^9}, {3.5109728140863733`*^9, 
   3.510972844880827*^9}, 3.510974608651125*^9, {3.511003646932313*^9, 
   3.5110036534843245`*^9}, {3.5111471459271812`*^9, 
   3.5111471546631966`*^9}, {3.5111473903484106`*^9, 3.511147422125666*^9}, {
   3.5111552347923946`*^9, 3.5111552422648077`*^9}, {3.5113307607556705`*^9, 
   3.511330775575697*^9}, {3.5113321606397295`*^9, 3.511332172963751*^9}, {
   3.5113913047691927`*^9, 3.5113913416788573`*^9}, {3.5114131119804215`*^9, 
   3.5114131610581074`*^9}, 3.5127114769319134`*^9, {3.5127751755126777`*^9, 
   3.5127751759026785`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Transfer Function Plot", "Subsection",
 CellChangeTimes->{{3.5106327360154905`*^9, 3.510632742271102*^9}, {
  3.5109229122064953`*^9, 3.5109229133585615`*^9}, {3.510975022395052*^9, 
  3.510975046309894*^9}, {3.511431088725732*^9, 3.511431089770934*^9}}],

Cell[CellGroupData[{

Cell["Usage Notes", "Subsubsection",
 CellChangeTimes->{{3.512731063640951*^9, 3.5127310662461557`*^9}}],

Cell[BoxData[
 RowBox[{"?", "tfplot"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127276236861057`*^9, 3.5127276254801083`*^9}, 
   3.5127278088272305`*^9, 3.5127285744453754`*^9}],

Cell[BoxData[
 StyleBox["\<\"tfplot[{param},{var},freqlist] calculates transfer functions \
from input parameters' displacement to output variables' displacement.\"\>", 
  "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.512730831884943*^9},
 CellTags->"Info3512763231-3783408"],

Cell[BoxData[
 RowBox[{"?", "tfplotf"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127277012026415`*^9, 3.512727705445849*^9}, 
   3.5127278088272305`*^9, 3.5127285744609756`*^9}],

Cell[BoxData[
 StyleBox["\<\"tfplot[{varin},{varout},freq] calculates transfer functions \
from force exerted to the input variables, to output variables' displacement.\
\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.5127308323061438`*^9},
 CellTags->"Info3512763232-9783139"],

Cell[BoxData[
 RowBox[{"?", "tfplota"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.512727708347454*^9, 3.512727710375458*^9}, 
   3.5127278088272305`*^9, 3.5127285744609756`*^9}],

Cell[BoxData[
 StyleBox["\<\"tfplota[{act},{var},freqlist] shows transfer functions from \
input actuator force to output variables' displacement. Input can be a linear \
combination of actuators.\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.5127308328365445`*^9},
 CellTags->"Info3512763232-5043775"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Plots", "Subsubsection",
 CellChangeTimes->{{3.5127310719245653`*^9, 3.512731072782567*^9}}],

Cell[CellGroupData[{

Cell["Transfer function from disp. to disp.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511418345274813*^9, 3.511418367785653*^9}}],

Cell[BoxData[
 RowBox[{"tfplot", "[", 
  RowBox[{
   RowBox[{"{", "zg", "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM"}], "}"}], ",", "freq"}], "]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.510957748603794*^9, 3.5109577777446456`*^9}, {
   3.510957947894144*^9, 3.510957950312149*^9}, 3.5109588754873734`*^9, {
   3.510960251768591*^9, 3.5109602568386*^9}, {3.5109644688037157`*^9, 
   3.5109645325454273`*^9}, {3.5109646318084016`*^9, 3.510964636956411*^9}, {
   3.5109652306946535`*^9, 3.510965232036256*^9}, {3.5109653621872845`*^9, 
   3.510965374776507*^9}, {3.510965421685789*^9, 3.510965427161399*^9}, {
   3.5109750765583467`*^9, 3.5109750793195515`*^9}, {3.5109818646568904`*^9, 
   3.5109818950301437`*^9}, {3.5109829019735146`*^9, 
   3.5109829028627157`*^9}, {3.511147957815007*^9, 3.511147970092229*^9}, {
   3.5111480189515142`*^9, 3.5111480302771344`*^9}, {3.511152738053403*^9, 
   3.5111527568982363`*^9}, {3.5111614206837378`*^9, 
   3.5111614428905745`*^9}, {3.511232181917014*^9, 3.5112321830402164`*^9}, {
   3.5112326734114776`*^9, 3.5112326900723066`*^9}, {3.5112327680568438`*^9, 
   3.5112327702252474`*^9}, 3.51123286263981*^9, {3.5112329654595904`*^9, 
   3.5112329723704023`*^9}, {3.51123301048127*^9, 3.51123301626888*^9}, {
   3.511233061602559*^9, 3.511233087561005*^9}, {3.511233247133685*^9, 
   3.511233263388914*^9}, {3.511330838459407*^9, 3.511330842000613*^9}, {
   3.5113312692229633`*^9, 3.5113312709701667`*^9}, {3.5113904840985513`*^9, 
   3.511390509822996*^9}, {3.511414980348903*^9, 3.5114149810821047`*^9}, 
   3.511418367785653*^9, {3.512728004420374*^9, 3.5127280098959837`*^9}, {
   3.512733378388218*^9, 3.512733383676627*^9}, {3.5127354019145722`*^9, 
   3.512735408201383*^9}}],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[Large], LineBox[CompressedData["
1:eJwUWnc4198XR/beKxFllUr0bVj3Y68oyYjKLqUipYSWnSilQRGijCSzFDlX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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
       LineBox[CompressedData["
1:eJwUWHc8Vv8Xf57H3ltWpFIpo/BF1ucpSoqWjFIiFKmEyAqpUNKSmZVIJFuR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        "]]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{-2.302585092994045, -18.851625345860068`},
     BaseStyle->{FontSize -> Scaled[0.04]},
     CoordinatesToolOptions:>{"DisplayFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& ), "CopiedValueFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& )},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     Frame->True,
     FrameLabel->{
       FormBox["\"Frequency [Hz]\"", TraditionalForm]},
     FrameTicks->{{{{-18.420680743952367`, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "8"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-13.815510557964274`, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "6"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-9.210340371976182, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "4"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-4.605170185988091, 
          FormBox["0.01`", TraditionalForm]}, {0, 
          FormBox["1", TraditionalForm]}, {4.605170185988092, 
          FormBox["100", TraditionalForm]}, {-15.935774094164366`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-15.285186528023216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.894320219336205`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.614018254182046`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.395329053217216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.215988124561399`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.063971917262773`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-13.932044374220226`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.330603908176274`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.680016342035124`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.289150033348113`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.008848068193954`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.790158867229126, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.610817938573309, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.458801731274683, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.326874188232134, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.725433722188183, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.074846156047033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.683979847360021, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.403677882205863, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.184988681241033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.005647752585217, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.853631545286591, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.721704002244043, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.120263536200091, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.4696759700589417`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.07880966137193, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.7985076962177716, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.579818495252942, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.40047756659712525`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2484613592984996, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.11653381625595151`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.4849066497880004`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.1354942159291497`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.5263605246161616`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.8066624897703196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.02535169073515, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.204692619390966, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.356708826689592, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.48863636973214, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-18.420680743952367`, 
          FormBox["\"\"", TraditionalForm]}, {-13.815510557964274`, 
          FormBox["\"\"", TraditionalForm]}, {-9.210340371976182, 
          FormBox["\"\"", TraditionalForm]}, {-4.605170185988091, 
          FormBox["\"\"", TraditionalForm]}, {0, 
          FormBox["\"\"", TraditionalForm]}, {4.605170185988092, 
          FormBox["\"\"", TraditionalForm]}, {-15.935774094164366`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-15.285186528023216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.894320219336205`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.614018254182046`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.395329053217216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.215988124561399`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.063971917262773`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-13.932044374220226`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.330603908176274`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.680016342035124`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.289150033348113`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.008848068193954`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.790158867229126, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.610817938573309, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.458801731274683, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.326874188232134, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.725433722188183, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.074846156047033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.683979847360021, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.403677882205863, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.184988681241033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.005647752585217, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.853631545286591, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.721704002244043, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.120263536200091, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.4696759700589417`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.07880966137193, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.7985076962177716, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.579818495252942, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.40047756659712525`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2484613592984996, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.11653381625595151`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.4849066497880004`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.1354942159291497`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.5263605246161616`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.8066624897703196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.02535169073515, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.204692619390966, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.356708826689592, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.48863636973214, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}, {{{-2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-0.6931471805599453, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {0., 
          FormBox[
           TagBox[
            InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {1.6094379124341003`, 
          FormBox[
           TagBox[
            InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {3.912023005428146, 
          FormBox[
           TagBox[
            InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {4.605170185988092, 
          FormBox[
           TagBox[
            InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {6.214608098422191, 
          FormBox[
           TagBox[
            InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {-0.6931471805599453, 
          FormBox["\"\"", TraditionalForm]}, {0., 
          FormBox["\"\"", TraditionalForm]}, {1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm]}, {2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {3.912023005428146, 
          FormBox["\"\"", TraditionalForm]}, {4.605170185988092, 
          FormBox["\"\"", TraditionalForm]}, {6.214608098422191, 
          FormBox["\"\"", TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}},
     GridLines->{{-2.302585092994046, -0.6931471805599453, 0., 
       1.6094379124341003`, 2.302585092994046, 3.912023005428146, 
       4.605170185988092, 
       6.214608098422191, -1.6094379124341003`, -1.203972804325936, \
-0.916290731874155, -0.5108256237659907, -0.35667494393873245`, \
-0.2231435513142097, -0.10536051565782628`, 0.6931471805599453, 
       1.0986122886681098`, 1.3862943611198906`, 1.791759469228055, 
       1.9459101490553132`, 2.0794415416798357`, 2.1972245773362196`, 
       2.995732273553991, 3.4011973816621555`, 3.6888794541139363`, 
       4.0943445622221, 4.248495242049359, 4.382026634673881, 
       4.499809670330265, 5.298317366548036, 5.703782474656201, 
       5.991464547107982}, {-18.420680743952367`, -13.815510557964274`, \
-9.210340371976182, -4.605170185988091, 0, 
        4.605170185988092, -15.935774094164366`, -15.285186528023216`, \
-14.894320219336205`, -14.614018254182046`, -14.395329053217216`, \
-14.215988124561399`, -14.063971917262773`, -13.932044374220226`, \
-11.330603908176274`, -10.680016342035124`, -10.289150033348113`, \
-10.008848068193954`, -9.790158867229126, -9.610817938573309, \
-9.458801731274683, -9.326874188232134, -6.725433722188183, \
-6.074846156047033, -5.683979847360021, -5.403677882205863, \
-5.184988681241033, -5.005647752585217, -4.853631545286591, \
-4.721704002244043, -2.120263536200091, -1.4696759700589417`, \
-1.07880966137193, -0.7985076962177716, -0.579818495252942, \
-0.40047756659712525`, -0.2484613592984996, -0.11653381625595151`, 
        2.4849066497880004`, 3.1354942159291497`, 3.5263605246161616`, 
        3.8066624897703196`, 4.02535169073515, 4.204692619390966, 
        4.356708826689592, 4.48863636973214}},
     ImageSize->600,
     Method->{},
     PlotRange->{{-2.302585092994045, 
      3.9120230054281455`}, {-18.851625345860068`, 7.000523123323866}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], 
       Scaled[0.02]},
     Ticks->{{{-2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-0.6931471805599453, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {0., 
         FormBox[
          TagBox[
           InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {1.6094379124341003`, 
         FormBox[
          TagBox[
           InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {3.912023005428146, 
         FormBox[
          TagBox[
           InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {4.605170185988092, 
         FormBox[
          TagBox[
           InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {6.214608098422191, 
         FormBox[
          TagBox[
           InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-1.6094379124341003`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.203972804325936, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.916290731874155, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.5108256237659907, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.35667494393873245`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.2231435513142097, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.10536051565782628`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {0.6931471805599453, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.0986122886681098`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.3862943611198906`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.791759469228055, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.9459101490553132`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.0794415416798357`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.1972245773362196`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.995732273553991, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.4011973816621555`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.6888794541139363`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.0943445622221, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.248495242049359, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.382026634673881, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.499809670330265, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.298317366548036, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.703782474656201, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.991464547107982, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}, {{-18.420680743952367`, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "8"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-13.815510557964274`, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "6"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-9.210340371976182, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "4"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-4.605170185988091, 
         FormBox["0.01`", TraditionalForm]}, {0, 
         FormBox["1", TraditionalForm]}, {4.605170185988092, 
         FormBox["100", TraditionalForm]}, {-15.935774094164366`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-15.285186528023216`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.894320219336205`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.614018254182046`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.395329053217216`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.215988124561399`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.063971917262773`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-13.932044374220226`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-11.330603908176274`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-10.680016342035124`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-10.289150033348113`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-10.008848068193954`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.790158867229126, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.610817938573309, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.458801731274683, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.326874188232134, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-6.725433722188183, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-6.074846156047033, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.683979847360021, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.403677882205863, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.184988681241033, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.005647752585217, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-4.853631545286591, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-4.721704002244043, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-2.120263536200091, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.4696759700589417`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.07880966137193, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.7985076962177716, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.579818495252942, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.40047756659712525`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.2484613592984996, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.11653381625595151`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.4849066497880004`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.1354942159291497`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.5263605246161616`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.8066624897703196`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.02535169073515, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.204692619390966, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.356708826689592, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.48863636973214, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], GraphicsGroupBox[{
     {GrayLevel[0], RectangleBox[{0.35, 0.3}, {0.78, 0.45999999999999996`}]}, 
     {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
      GraphicsGroupBox[{
        RectangleBox[{0.35, 0.3}, {0.78, 0.45999999999999996`}], InsetBox[
         GraphicsBox[{{{InsetBox[
              GraphicsBox[{
                {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 0}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"pitchTM/zg\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[1, 2]}, {1.05, 0.5}], {-1, 0},
               Automatic, {1, 0}]}, {InsetBox[
              GraphicsBox[{
                {Hue[0.67, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 1}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"zTM/zg\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[3, 2]}, {1.05, 1.5}], {-1, 0},
               Automatic, {1, 0}]}}, {}},
          AspectRatio->0.372093023255814,
          FormatType->TraditionalForm,
          PlotRange->{{-0.1, 3.6}, {-0.1, 2.1}}], {0.35, 0.3}, {
         Left, Bottom}, {0.43, 0.16}]}]}}]},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->600.,
  PlotRange->All]], "Output",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511161422990941*^9, 3.5111614478109827`*^9}, 
   3.511230657179332*^9, 3.511230687973786*^9, 3.511231850987028*^9, 
   3.5112321855830207`*^9, 3.5112326978099203`*^9, 3.5112327732360525`*^9, {
   3.5112328563529987`*^9, 3.5112328659626155`*^9}, 3.5112329164599047`*^9, 
   3.5112329747416067`*^9, {3.5112330127276735`*^9, 3.5112330177040825`*^9}, {
   3.5112330645665646`*^9, 3.511233088918207*^9}, {3.5112332403788733`*^9, 
   3.511233264777316*^9}, 3.511330866367856*^9, 3.5113313123882394`*^9, 
   3.511390364321541*^9, 3.5113904650821176`*^9, {3.511390495814172*^9, 
   3.5113905162346077`*^9}, 3.511418367785653*^9, 3.5114185710696096`*^9, 
   3.511431634843893*^9, 3.512730750000399*^9, 3.5127333926310434`*^9, {
   3.512735394317359*^9, 3.5127354208530054`*^9}, 3.512775202353125*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Transfer function from force to disp.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5109750603967185`*^9, 3.5109750715975385`*^9}, {
  3.5114183861780853`*^9, 3.5114184005613103`*^9}}],

Cell[BoxData[
 RowBox[{"tfplotf", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"zIM", "-", "zIRM"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM"}], "}"}], ",", "freq"}], "]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.510957748603794*^9, 3.5109577777446456`*^9}, {
   3.510957947894144*^9, 3.510957950312149*^9}, 3.5109588754873734`*^9, {
   3.510960251768591*^9, 3.5109602568386*^9}, {3.5109644688037157`*^9, 
   3.5109645325454273`*^9}, {3.5109646318084016`*^9, 3.510964636956411*^9}, {
   3.5109652306946535`*^9, 3.510965232036256*^9}, {3.5109653621872845`*^9, 
   3.510965374776507*^9}, {3.510965421685789*^9, 3.510965427161399*^9}, {
   3.5109750765583467`*^9, 3.5109750893035693`*^9}, {3.5109828951875024`*^9, 
   3.510982897106306*^9}, {3.511002574321229*^9, 3.511002578689237*^9}, {
   3.511147126083946*^9, 3.511147126583147*^9}, {3.51133135800272*^9, 
   3.5113313625891275`*^9}, 3.5114184005613103`*^9, {3.511431802278987*^9, 
   3.511431817223813*^9}, {3.5114319322428155`*^9, 3.511431934083619*^9}, {
   3.511494423146207*^9, 3.511494426297412*^9}, {3.5114944629106765`*^9, 
   3.511494463940278*^9}, {3.511494528649192*^9, 3.511494529819194*^9}, {
   3.511494611407337*^9, 3.5114946157285447`*^9}, {3.5114947070511055`*^9, 
   3.51149470978111*^9}, {3.511495153274289*^9, 3.511495157252296*^9}, {
   3.511495906471814*^9, 3.5114959082502174`*^9}, 3.511495988091158*^9, {
   3.5114960625188885`*^9, 3.5114960693985004`*^9}, {3.5114976579860983`*^9, 
   3.5114976911673565`*^9}, {3.511497813627572*^9, 3.511497817184378*^9}, {
   3.5123560718342876`*^9, 3.512356077871499*^9}, {3.5123561129091606`*^9, 
   3.512356116699967*^9}, {3.5123595749764414`*^9, 3.51235961417931*^9}, 
   3.512359690713044*^9, {3.512423733837125*^9, 3.5124237421987395`*^9}, {
   3.512728016650796*^9, 3.5127280238736086`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Transfer function from actuator to disp.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.510969554959849*^9, 3.510969560841059*^9}, {
   3.5109751070876007`*^9, 3.5109751100048056`*^9}, {3.5114184059745197`*^9, 
   3.511418444678188*^9}, 3.5127279886643467`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"ca1", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"+", "0.25"}]}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"ca2", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"-", "0.25"}], ",", 
     RowBox[{"-", "0.25"}], ",", 
     RowBox[{"+", "0.25"}]}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"aTMlist", "=", 
   RowBox[{"{", 
    RowBox[{"aTM1", ",", "aTM2", ",", "aTM3", ",", "aTM4"}], "}"}]}], 
  ";"}]}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.512727879666955*^9, 3.5127279886643467`*^9}}],

Cell[BoxData[
 RowBox[{"tfplota", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"ca1", ".", "aTMlist"}], ",", 
     RowBox[{"ca2", ".", "aTMlist"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM"}], "}"}], ",", "freq"}], "]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.510974630101163*^9, 3.510974667307228*^9}, {
   3.5109751707513123`*^9, 3.510975230827018*^9}, {3.510975278219901*^9, 
   3.510975296534333*^9}, 3.510975617333297*^9, 3.5109762079503345`*^9, {
   3.510976711909239*^9, 3.5109767681005373`*^9}, {3.510981275918256*^9, 
   3.510981277462659*^9}, {3.5109815367911143`*^9, 3.510981543436726*^9}, {
   3.5110039773252935`*^9, 3.5110039774500937`*^9}, {3.5111476837225256`*^9, 
   3.511147683909726*^9}, {3.511149615115118*^9, 3.5111496153023186`*^9}, {
   3.5111518249681997`*^9, 3.5111518251085997`*^9}, {3.511155202125937*^9, 
   3.511155202281937*^9}, {3.511155261842842*^9, 3.511155266881651*^9}, {
   3.51133147809173*^9, 3.511331483863741*^9}, {3.5113317639466324`*^9, 
   3.511331775989854*^9}, {3.5113321818401666`*^9, 3.5113321940861883`*^9}, {
   3.511332324533617*^9, 3.511332347590458*^9}, {3.511332426900997*^9, 
   3.511332473513879*^9}, {3.511332686797454*^9, 3.51133270737389*^9}, {
   3.511332765078391*^9, 3.511332767293595*^9}, {3.511391414234585*^9, 
   3.511391414983386*^9}, {3.511418452665402*^9, 3.511418453414203*^9}, {
   3.51141849147827*^9, 3.5114185140983095`*^9}, {3.511418638882929*^9, 
   3.51141863970973*^9}, {3.5114187268670835`*^9, 3.511418727069884*^9}, {
   3.511418809126028*^9, 3.5114188156312394`*^9}, {3.5114318640550957`*^9, 
   3.5114318802947245`*^9}, {3.5127278701353383`*^9, 3.512727873208544*^9}, {
   3.5127279545158863`*^9, 3.5127279886643467`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["State-Space Model", "Section",
 CellChangeTimes->{{3.511335807091339*^9, 3.5113358106481447`*^9}, 
   3.511337659532192*^9, {3.512773070004176*^9, 3.5127730729993815`*^9}}],

Cell[CellGroupData[{

Cell["State-Space Matrices", "Subsection",
 CellChangeTimes->{{3.511338449601981*^9, 3.511338476356028*^9}}],

Cell[CellGroupData[{

Cell["Set Input", "Subsubsection",
 CellChangeTimes->{{3.511345101119687*^9, 3.5113451151285114`*^9}, 
   3.5113452765575953`*^9, {3.5113491519476023`*^9, 3.511349152649603*^9}}],

Cell[CellGroupData[{

Cell["Create parameter input vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, 
   3.5113919865527897`*^9, {3.5113932945782876`*^9, 3.511393300724698*^9}, 
   3.511393991963113*^9, {3.511394540569276*^9, 3.5113945443288827`*^9}, 
   3.512728164258255*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"inputp", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"inputvecp", "=", 
   RowBox[{"makeinputvecp", "[", "inputp", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, 3.511347728554302*^9, {
   3.51135156339744*^9, 3.5113515638186407`*^9}, 3.5113919865527897`*^9, {
   3.5113923432630167`*^9, 3.511392343497017*^9}, {3.511393282316666*^9, 
   3.5113932878546762`*^9}, 3.5113933213323345`*^9, 3.511393991963113*^9, {
   3.5127281565986414`*^9, 3.5127281605922484`*^9}, {3.5127308942850523`*^9, 
   3.51273090969788*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create force input vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, 
   3.5113919865527897`*^9, {3.5113932945782876`*^9, 3.511393313891122*^9}, 
   3.5113939948803177`*^9, {3.5113945468248873`*^9, 3.5113945475892887`*^9}, 
   3.5127281856926928`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"inputf", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"inputvecf", "=", 
   RowBox[{"makeinputvecf", "[", "inputf", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, 3.511347728554302*^9, {
   3.51135156339744*^9, 3.5113515638186407`*^9}, 3.5113919865527897`*^9, {
   3.5113923432630167`*^9, 3.511392343497017*^9}, {3.511393282316666*^9, 
   3.5113932878546762`*^9}, {3.5113933227831373`*^9, 3.511393333874757*^9}, 
   3.5113939948803177`*^9, {3.511419195257906*^9, 3.5114191953359065`*^9}, 
   3.51143273690463*^9, {3.5127281700770655`*^9, 3.5127281744762726`*^9}, {
   3.5127309033018684`*^9, 3.512730912833485*^9}, {3.512775346247778*^9, 
   3.512775358103799*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create actuator input vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, 
   3.5113919865527897`*^9, {3.5113932945782876`*^9, 3.511393346042778*^9}, {
   3.511394549476892*^9, 3.5113945509276943`*^9}, {3.5127281815742855`*^9, 
   3.5127281824634867`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"inputa", "=", 
   RowBox[{"{", 
    RowBox[{"aTM1", ",", "aTM2", ",", "aTM3", ",", "aTM4"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"inputveca", "=", 
   RowBox[{"makeinputveca", "[", "inputa", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, 3.511347728554302*^9, {
   3.51135156339744*^9, 3.5113515638186407`*^9}, 3.5113919865527897`*^9, {
   3.5113923432630167`*^9, 3.511392343497017*^9}, {3.511393282316666*^9, 
   3.5113932878546762`*^9}, {3.5113933227831373`*^9, 3.511393333874757*^9}, {
   3.5113934356805353`*^9, 3.511393441889346*^9}, {3.5113939599674563`*^9, 
   3.5113939625102606`*^9}, {3.5114327391510344`*^9, 
   3.5114327467638474`*^9}, {3.512728188204297*^9, 3.512728199093116*^9}, {
   3.5127309146586885`*^9, 3.51273092685791*^9}, {3.512775320070932*^9, 
   3.512775328604147*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Set Output", "Subsubsection",
 CellChangeTimes->{{3.5113471295288496`*^9, 3.511347130230851*^9}, {
  3.511349154038006*^9, 3.511349154693207*^9}}],

Cell[CellGroupData[{

Cell["Create output vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, {
   3.511347126221644*^9, 3.5113471378280644`*^9}, {3.511394532363662*^9, 
   3.5113945368876696`*^9}, {3.512728214271943*^9, 3.5127282158943453`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"outputv", "=", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM", ",", "yawTM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"outputvecv", "=", 
   RowBox[{"makeoutputvecv", "[", "outputv", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, {3.5113470309210763`*^9, 
   3.511347050530311*^9}, 3.511347126221644*^9, {3.5113926520031586`*^9, 
   3.51139265832117*^9}, {3.5113944608687363`*^9, 3.511394475579562*^9}, 
   3.5114190053119726`*^9, {3.512728211682338*^9, 3.5127282259251633`*^9}, {
   3.5127309369199276`*^9, 3.512730975311595*^9}, {3.512775368119016*^9, 
   3.512775390692256*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["State-Space Matrices", "Subsubsection",
 CellChangeTimes->{{3.5113428440609226`*^9, 3.5113428707837696`*^9}, {
  3.5114035961001205`*^9, 3.511403604336935*^9}, {3.511404827987484*^9, 
  3.511404832542692*^9}}],

Cell[CellGroupData[{

Cell["\<\
Damping frequency (for use in converting structure damping to viscous damping)\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511404853665129*^9, 3.5114048575495358`*^9}, {
  3.5114049021188145`*^9, 3.5114049368912754`*^9}, {3.5114050434862623`*^9, 
  3.5114050447498646`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"fdamp", "=", 
    RowBox[{"0.03", "hertz"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511404881448778*^9, 3.5114049021188145`*^9}, {
  3.5114059906771326`*^9, 3.511405991878335*^9}, {3.511406097755721*^9, 
  3.5114061003297253`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Create the \"A\" matrix. Structure damping is considered to be zero.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113491808076525`*^9}, {
  3.511349227935335*^9, 3.5113492291365376`*^9}, {3.5114048144466605`*^9, 
  3.51140481996907*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssAr", "=", 
    RowBox[{"ArrayFlatten", "[", "\[IndentingNewLine]", 
     RowBox[{"{", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"IdentityMatrix", "[", 
          RowBox[{"Length", "[", "allvars", "]"}], "]"}]}], 
        "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matKvv"}]}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matGvv"}]}]}], 
        "\[IndentingNewLine]", "}"}]}], "\[IndentingNewLine]", "}"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113426757990274`*^9, 3.5113426772342296`*^9}, {
   3.5113427437527466`*^9, 3.5113427464047513`*^9}, {3.511342836120509*^9, 
   3.511342836338909*^9}, {3.5113428792545843`*^9, 3.511342914588647*^9}, {
   3.5113433061181345`*^9, 3.5113433157745514`*^9}, {3.511343413555523*^9, 
   3.51134342885915*^9}, {3.511343479060038*^9, 3.5113435935642395`*^9}, {
   3.511343633001108*^9, 3.5113436752147827`*^9}, {3.5113438056778116`*^9, 
   3.5113438390618706`*^9}, 3.5113438807139435`*^9, 3.51134391846601*^9, {
   3.511347221849812*^9, 3.511347406320136*^9}, {3.511347441045797*^9, 
   3.5113474641026373`*^9}, 3.511347506503512*^9, 3.511349063230246*^9, 
   3.5113491808076525`*^9, {3.511352291731519*^9, 3.5113522926051207`*^9}, {
   3.5113527927143993`*^9, 3.5113528109664316`*^9}, {3.5113905926279416`*^9, 
   3.5113905982283516`*^9}, {3.5114050655291014`*^9, 3.511405068290306*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Create the \"A\" matrix. Structure damping is approximated by viscous damping \
such that the amount of phase lag is the same at damping frequency.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113491808076525`*^9}, {
  3.511349227935335*^9, 3.5113492291365376`*^9}, {3.51140477454179*^9, 
  3.5114048023878393`*^9}, {3.5114048635711465`*^9, 3.5114048754583673`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssAa", "=", 
    RowBox[{"ArrayFlatten", "[", "\[IndentingNewLine]", 
     RowBox[{"{", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"IdentityMatrix", "[", 
          RowBox[{"Length", "[", "allvars", "]"}], "]"}]}], 
        "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matKvv"}]}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{
            RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matGvv"}]}], "-", 
          RowBox[{
           RowBox[{
            RowBox[{"Inverse", "[", "matMvv", "]"}], ".", 
            RowBox[{"Im", "[", "matKvvc", "]"}]}], "/", 
           RowBox[{"(", 
            RowBox[{"2", 
             RowBox[{"N", "[", "Pi", "]"}], "fdamp"}], ")"}]}]}]}], 
        "\[IndentingNewLine]", "}"}]}], "\[IndentingNewLine]", "}"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113426757990274`*^9, 3.5113426772342296`*^9}, {
   3.5113427437527466`*^9, 3.5113427464047513`*^9}, {3.511342836120509*^9, 
   3.511342836338909*^9}, {3.5113428792545843`*^9, 3.511342914588647*^9}, {
   3.5113433061181345`*^9, 3.5113433157745514`*^9}, {3.511343413555523*^9, 
   3.51134342885915*^9}, {3.511343479060038*^9, 3.5113435935642395`*^9}, {
   3.511343633001108*^9, 3.5113436752147827`*^9}, {3.5113438056778116`*^9, 
   3.5113438390618706`*^9}, 3.5113438807139435`*^9, 3.51134391846601*^9, {
   3.511347221849812*^9, 3.511347406320136*^9}, {3.511347441045797*^9, 
   3.5113474641026373`*^9}, 3.511347506503512*^9, 3.511349063230246*^9, 
   3.5113491808076525`*^9, {3.511352291731519*^9, 3.5113522926051207`*^9}, {
   3.5113527927143993`*^9, 3.5113528109664316`*^9}, {3.5113905926279416`*^9, 
   3.5113905982283516`*^9}, 3.5114048023878393`*^9, {3.511405008043*^9, 
   3.5114051058707724`*^9}, {3.5114054963638587`*^9, 3.511405506831477*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create the \"B\" matrix.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113492256733313`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssB", "=", 
    RowBox[{"ArrayFlatten", "[", "\[IndentingNewLine]", 
     RowBox[{"{", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "inputp", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "inputf", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "inputa", "]"}], "}"}]}], "]"}]}], 
        "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matKpv", ".", 
           "inputvecp"}]}], ",", "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "inputvecf"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "inputveca"}]}], 
        "\[IndentingNewLine]", "}"}]}], "\[IndentingNewLine]", "}"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511343921024414*^9, 3.5113439340036373`*^9}, {
   3.5113475427891755`*^9, 3.5113477094598684`*^9}, {3.5113478660841436`*^9, 
   3.5113479747539344`*^9}, {3.511348254119225*^9, 3.5113482681436496`*^9}, {
   3.5113485186332893`*^9, 3.51134851882049*^9}, 3.511349064587448*^9, 
   3.511349196797681*^9, {3.5113528177836437`*^9, 3.5113528253184566`*^9}, {
   3.5113529173430185`*^9, 3.5113529199950233`*^9}, {3.511392437830383*^9, 
   3.511392440373187*^9}, {3.5113925180769234`*^9, 3.511392522725732*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create the \"C\" matrix.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113492162977147`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssC", "=", 
    RowBox[{"Join", "[", "\[IndentingNewLine]", 
     RowBox[{"outputvecv", ",", "\[IndentingNewLine]", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "allvars", "]"}], "}"}]}], "]"}], ",", "2"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511348273494459*^9, 3.5113482899836884`*^9}, {
   3.5113483981230783`*^9, 3.5113484770436163`*^9}, {3.5113485408165283`*^9, 
   3.511348569692179*^9}, 3.5113486013602347`*^9, {3.511348992686922*^9, 
   3.5113490063369465`*^9}, {3.511349044089012*^9, 3.5113490659446507`*^9}, 
   3.5113492071716986`*^9, {3.5113529435354643`*^9, 3.511352946218669*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create the \"D\" matrix.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113491808076525`*^9}, {
  3.5113492109469056`*^9, 3.5113492202133217`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssD", "=", 
    RowBox[{"Join", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "inputp", "]"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "inputf", "]"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "inputa", "]"}], "}"}]}], "]"}], ",", "2"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511349068035054*^9, 3.511349108813526*^9}, {
  3.5113492202133217`*^9, 3.511349221913725*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Export", "Subsection",
 CellChangeTimes->{{3.5113492434105625`*^9, 3.5113492506957755`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matlabexport", "[", 
    RowBox[{"\"\<matlab/ss110426.m\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssA\>\"", ",", "ssAa"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssB\>\"", ",", "ssB"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssC\>\"", ",", "ssC"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssD\>\"", ",", "ssD"}], "}"}]}], "]"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.511349255204183*^9, 3.511349323360703*^9}, 
   3.5114051453544416`*^9, {3.5114052158381658`*^9, 3.5114052352194*^9}, 
   3.5114055594347696`*^9, 3.511418896018181*^9, 3.511431975798092*^9, {
   3.5114323267943096`*^9, 3.5114323285415125`*^9}, {3.51277540460748*^9, 
   3.512775405574682*^9}}]
}, Open  ]]
}, Closed]]
}, Open  ]]
},
WindowSize->{1008, 632},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
ShowSelection->True,
FrontEndVersion->"7.0 for Microsoft Windows (32-bit) (2008\:5e7412\:670810\
\:65e5)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "Info3512807310-4917596"->{
  Cell[104155, 2810, 353, 6, 155, "Print",
   CellTags->"Info3512807310-4917596"]},
 "Info3512807310-2061953"->{
  Cell[112841, 3037, 599, 9, 243, "Print",
   CellTags->"Info3512807310-2061953"]},
 "Info3512807310-4632052"->{
  Cell[120326, 3221, 557, 8, 199, "Print",
   CellTags->"Info3512807310-4632052"]},
 "Info3512807310-2021627"->{
  Cell[126349, 3365, 368, 6, 111, "Print",
   CellTags->"Info3512807310-2021627"]},
 "Info3512763231-3783408"->{
  Cell[219479, 5324, 282, 5, 44, "Print",
   CellTags->"Info3512763231-3783408"]},
 "Info3512763232-9783139"->{
  Cell[219990, 5337, 289, 5, 44, "Print",
   CellTags->"Info3512763232-9783139"]},
 "Info3512763232-5043775"->{
  Cell[220506, 5350, 315, 5, 67, "Print",
   CellTags->"Info3512763232-5043775"]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"Info3512807310-4917596", 364114, 7972},
 {"Info3512807310-2061953", 364227, 7975},
 {"Info3512807310-4632052", 364340, 7978},
 {"Info3512807310-2021627", 364453, 7981},
 {"Info3512763231-3783408", 364566, 7984},
 {"Info3512763232-9783139", 364678, 7987},
 {"Info3512763232-5043775", 364790, 7990}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[567, 22, 591, 11, 105, "Subtitle"],
Cell[CellGroupData[{
Cell[1183, 37, 172, 2, 71, "Section"],
Cell[CellGroupData[{
Cell[1380, 43, 157, 2, 36, "Subsection"],
Cell[1540, 47, 622, 15, 65, "Text"]
}, Open  ]],
Cell[CellGroupData[{
Cell[2199, 67, 105, 1, 36, "Subsection"],
Cell[2307, 70, 346, 7, 47, "Text"]
}, Closed]],
Cell[CellGroupData[{
Cell[2690, 82, 160, 2, 28, "Subsection"],
Cell[2853, 86, 552, 11, 83, "Text"]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[3454, 103, 204, 3, 71, "Section"],
Cell[CellGroupData[{
Cell[3683, 110, 204, 3, 36, "Subsection"],
Cell[3890, 115, 337, 4, 29, "Text"],
Cell[4230, 121, 446, 8, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[4713, 134, 203, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[4941, 141, 265, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[5209, 149, 465, 10, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[5711, 164, 219, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[5933, 169, 306, 7, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[6288, 182, 110, 1, 36, "Subsection"],
Cell[6401, 185, 518, 9, 47, "Text"],
Cell[6922, 196, 279, 7, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[7238, 208, 205, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[7468, 215, 297, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[7768, 221, 302, 6, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[8107, 232, 242, 5, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[8352, 239, 447, 7, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8848, 252, 203, 3, 36, "Subsection"],
Cell[9054, 257, 417, 7, 29, "Text"],
Cell[9474, 266, 2734, 79, 155, "Input"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12257, 351, 300, 4, 41, "Section"],
Cell[CellGroupData[{
Cell[12582, 359, 99, 1, 36, "Subsection"],
Cell[12684, 362, 339, 7, 65, "Text"]
}, Open  ]],
Cell[CellGroupData[{
Cell[13060, 374, 107, 1, 36, "Subsection"],
Cell[13170, 377, 603, 14, 54, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[13810, 396, 109, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[13944, 401, 210, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[14157, 406, 785, 22, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[14979, 433, 184, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[15166, 438, 1448, 32, 74, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[16651, 475, 177, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[16831, 480, 1658, 35, 74, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[18526, 520, 203, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[18732, 525, 1507, 32, 74, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[20288, 563, 151, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[20464, 569, 136, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[20625, 575, 236, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[20864, 580, 981, 25, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[21882, 610, 241, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[22126, 616, 2669, 55, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[24832, 676, 168, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[25003, 681, 520, 11, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[25584, 699, 95, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[25704, 704, 177, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[25906, 710, 236, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[26145, 715, 1885, 52, 232, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[28067, 772, 145, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[28215, 776, 4206, 105, 492, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[32470, 887, 229, 3, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[32724, 894, 222, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[32949, 899, 748, 14, 83, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[33700, 915, 244, 6, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[33981, 926, 193, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[34177, 931, 1606, 38, 172, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[35820, 974, 306, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[36129, 980, 1256, 29, 98, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[37422, 1014, 169, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[37594, 1019, 1576, 38, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[39207, 1062, 223, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}],
Cell[39433, 1067, 1087, 28, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[40569, 1101, 138, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[40732, 1107, 171, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[40906, 1112, 412, 9, 65, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[41355, 1126, 258, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[41616, 1131, 322, 9, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[41975, 1145, 245, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[42223, 1150, 7660, 202, 412, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[49920, 1357, 170, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[50093, 1361, 7820, 206, 412, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[57974, 1574, 153, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[58152, 1580, 205, 3, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[58382, 1587, 225, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[58610, 1592, 675, 15, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[59322, 1612, 178, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[59503, 1617, 703, 18, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[60243, 1640, 181, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[60427, 1645, 1420, 43, 76, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[61896, 1694, 169, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[62090, 1700, 193, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[62286, 1704, 1818, 31, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[64141, 1740, 220, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[64364, 1745, 1736, 28, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[66149, 1779, 183, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[66357, 1785, 177, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[66537, 1790, 1389, 28, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[67963, 1823, 273, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[68239, 1829, 1467, 29, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[69743, 1863, 185, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[69931, 1868, 1339, 29, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[71331, 1904, 99, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[71455, 1909, 102, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[71582, 1914, 169, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[71754, 1919, 1357, 44, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[73148, 1968, 204, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[73355, 1973, 2334, 69, 272, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[75738, 2048, 163, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[75926, 2054, 221, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[76150, 2059, 758, 17, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[76969, 2083, 100, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[77094, 2088, 106, 1, 27, "Subsubsection"],
Cell[77203, 2091, 549, 11, 83, "Text"],
Cell[77755, 2104, 360, 8, 72, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[78152, 2117, 213, 3, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[78390, 2124, 173, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[78566, 2129, 1261, 28, 192, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[79864, 2162, 173, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}],
Cell[80040, 2167, 847, 20, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[80924, 2192, 178, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[81105, 2197, 1151, 32, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[82293, 2234, 180, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}],
Cell[82476, 2239, 1092, 30, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[83605, 2274, 186, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10005.}],
Cell[83794, 2279, 1007, 30, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10005.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[84838, 2314, 198, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10006.}],
Cell[85039, 2319, 718, 20, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10006.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[85794, 2344, 172, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10007.}],
Cell[85969, 2349, 847, 22, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10007.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[86853, 2376, 196, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10008.}],
Cell[87052, 2381, 605, 14, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10008.}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[87730, 2403, 224, 3, 41, "Section"],
Cell[CellGroupData[{
Cell[87979, 2410, 203, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[88207, 2417, 125, 1, 27, "Subsubsection"],
Cell[88335, 2420, 1888, 52, 152, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[90260, 2477, 174, 2, 27, "Subsubsection"],
Cell[90437, 2481, 2015, 35, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[92489, 2521, 105, 1, 27, "Subsubsection"],
Cell[92597, 2524, 803, 19, 92, "Input"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[93449, 2549, 204, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[93678, 2556, 186, 2, 27, "Subsubsection"],
Cell[93867, 2560, 270, 8, 31, "Input"],
Cell[94140, 2570, 2672, 61, 152, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[96849, 2636, 272, 5, 27, "Subsubsection"],
Cell[97124, 2643, 1163, 23, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[98324, 2671, 103, 1, 27, "Subsubsection"],
Cell[98430, 2674, 863, 21, 92, "Input"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[99342, 2701, 234, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[99601, 2708, 110, 1, 27, "Subsubsection"],
Cell[99714, 2711, 2813, 42, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[102564, 2758, 103, 1, 27, "Subsubsection"],
Cell[102670, 2761, 752, 20, 52, "Input"]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[103483, 2788, 160, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[103668, 2794, 207, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[103900, 2801, 106, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[104031, 2806, 121, 2, 31, "Input"],
Cell[104155, 2810, 353, 6, 155, "Print",
 CellTags->"Info3512807310-4917596"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[104557, 2822, 108, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[104690, 2827, 176, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[104869, 2832, 371, 9, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[105277, 2846, 250, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[105530, 2852, 2915, 61, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[108494, 2919, 101, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[108620, 2924, 169, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[108792, 2929, 1302, 31, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[110131, 2965, 223, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[110357, 2970, 1990, 45, 252, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[112408, 3022, 157, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[112590, 3028, 104, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[112719, 3033, 119, 2, 31, "Input"],
Cell[112841, 3037, 599, 9, 243, "Print",
 CellTags->"Info3512807310-2061953"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[113489, 3052, 107, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[113621, 3057, 278, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[113902, 3063, 641, 15, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[114580, 3083, 381, 5, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[114964, 3090, 4757, 108, 347, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[119782, 3205, 259, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[120066, 3212, 109, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[120200, 3217, 123, 2, 31, "Input"],
Cell[120326, 3221, 557, 8, 199, "Print",
 CellTags->"Info3512807310-4632052"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[120932, 3235, 107, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[121064, 3240, 305, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[121372, 3246, 549, 11, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[121958, 3262, 182, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[122143, 3267, 3756, 77, 252, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[125960, 3351, 104, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[126089, 3356, 109, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[126223, 3361, 123, 2, 31, "Input"],
Cell[126349, 3365, 368, 6, 111, "Print",
 CellTags->"Info3512807310-2021627"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[126766, 3377, 112, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[126903, 3382, 232, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[127138, 3387, 354, 8, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[127553, 3402, 264, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[127842, 3409, 167, 2, 27, "Subsubsection"],
Cell[128012, 3413, 1848, 32, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[129897, 3450, 115, 1, 27, "Subsubsection"],
Cell[130015, 3453, 795, 18, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[130847, 3476, 109, 1, 27, "Subsubsection"],
Cell[130959, 3479, 2007, 34, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[133003, 3518, 108, 1, 27, "Subsubsection"],
Cell[133114, 3521, 436, 9, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[133587, 3535, 96, 1, 27, "Subsubsection"],
Cell[133686, 3538, 875, 17, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[134610, 3561, 113, 1, 36, "Subsection"],
Cell[134726, 3564, 574, 11, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[135337, 3580, 118, 1, 36, "Subsection"],
Cell[135458, 3583, 281, 7, 52, "Input"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[135788, 3596, 139, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[135952, 3602, 235, 3, 36, "Subsection"],
Cell[136190, 3607, 1414, 22, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[137641, 3634, 138, 2, 36, "Subsection"],
Cell[137782, 3638, 269, 3, 29, "Text"],
Cell[CellGroupData[{
Cell[138076, 3645, 1917, 33, 132, "Input"],
Cell[139996, 3680, 4085, 70, 73, "Output"]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[144142, 3757, 132, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[144299, 3763, 99, 1, 36, "Subsection"],
Cell[144401, 3766, 1611, 34, 72, "Input"],
Cell[146015, 3802, 1594, 34, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[147646, 3841, 155, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[147826, 3847, 197, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[148026, 3852, 1874, 38, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[149903, 3892, 1648, 35, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[151554, 3929, 4305, 99, 292, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[155896, 4033, 204, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[156103, 4038, 1662, 35, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[157768, 4075, 1546, 34, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[159317, 4111, 4251, 98, 292, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[163617, 4215, 153, 2, 36, "Subsection"],
Cell[163773, 4219, 1711, 36, 72, "Input"],
Cell[165487, 4257, 1694, 35, 92, "Input"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[167230, 4298, 94, 1, 41, "Section"],
Cell[CellGroupData[{
Cell[167349, 4303, 168, 2, 36, "Subsection"],
Cell[167520, 4307, 1095, 29, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[168652, 4341, 160, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[168837, 4347, 509, 8, 31, "Input"],
Cell[169349, 4357, 7711, 156, 387, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[177109, 4519, 161, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[177295, 4525, 214, 3, 27, "Subsubsection"],
Cell[177512, 4530, 1662, 40, 112, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[179211, 4575, 115, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[179351, 4580, 2668, 37, 31, "Input"],
Cell[182022, 4619, 28113, 483, 546, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[210208, 5110, 160, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[210393, 5116, 352, 5, 36, "Subsection"],
Cell[CellGroupData[{
Cell[210770, 5125, 2119, 41, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[212892, 5168, 2457, 50, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[215398, 5224, 160, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[215583, 5230, 113, 1, 27, "Subsubsection"],
Cell[215699, 5233, 491, 10, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[216227, 5248, 113, 1, 27, "Subsubsection"],
Cell[216343, 5251, 2468, 51, 272, "Input"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[218860, 5308, 260, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[219145, 5315, 104, 1, 27, "Subsubsection"],
Cell[219252, 5318, 224, 4, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[219479, 5324, 282, 5, 44, "Print",
 CellTags->"Info3512763231-3783408"],
Cell[219764, 5331, 223, 4, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[219990, 5337, 289, 5, 44, "Print",
 CellTags->"Info3512763232-9783139"],
Cell[220282, 5344, 221, 4, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[220506, 5350, 315, 5, 67, "Print",
 CellTags->"Info3512763232-5043775"]
}, Open  ]],
Cell[CellGroupData[{
Cell[220858, 5360, 98, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[220981, 5365, 172, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[221156, 5369, 1791, 28, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[222950, 5399, 117283, 2003, 390, "Output",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[340270, 7407, 229, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[340502, 7412, 1879, 29, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[342418, 7446, 305, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[342726, 7452, 750, 23, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[343479, 7477, 1789, 29, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[345341, 7514, 178, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[345544, 7520, 108, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[345677, 7525, 178, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[345880, 7531, 416, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[346299, 7539, 764, 15, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[347100, 7559, 418, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[347521, 7567, 914, 17, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[348472, 7589, 421, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[348896, 7597, 1033, 20, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[349978, 7623, 152, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[350155, 7629, 383, 5, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[350541, 7636, 822, 17, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[351412, 7659, 215, 3, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[351652, 7666, 329, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[351984, 7674, 358, 8, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[352379, 7687, 314, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[352696, 7695, 2122, 42, 272, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[354855, 7742, 443, 7, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[355301, 7751, 2494, 51, 272, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[357832, 7807, 161, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[357996, 7811, 2297, 50, 312, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[360330, 7866, 161, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[360494, 7870, 910, 18, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[361441, 7893, 214, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[361658, 7898, 1136, 29, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[362855, 7934, 98, 1, 36, "Subsection"],
Cell[362956, 7937, 778, 18, 52, "Input"]
}, Open  ]]
}, Closed]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
