(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 7.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    633989,      12307]
NotebookOptionsPosition[    611800,      11631]
NotebookOutlinePosition[    613269,      11678]
CellTagsIndexPosition[    612917,      11667]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["\<\

Toy-Model of 4-wire Simple Pendulum\
\>", "Subtitle",
 CellChangeTimes->{{3.5109088633079133`*^9, 3.510908886523241*^9}, {
   3.5109089295327015`*^9, 3.5109089606324797`*^9}, {3.510998012935999*^9, 
   3.510998040953648*^9}, {3.511147199138874*^9, 3.511147201837679*^9}, 
   3.5111528655524273`*^9, {3.5112140422561007`*^9, 3.511214046727357*^9}, {
   3.511333214700381*^9, 3.5113332166191845`*^9}, {3.5113333850886836`*^9, 
   3.511333385931085*^9}, {3.512773766799003*^9, 3.51277377079261*^9}, {
   3.512775502577655*^9, 3.512775507866064*^9}, {3.5127757788237405`*^9, 
   3.5127758085885925`*^9}}],

Cell[CellGroupData[{

Cell["Discription", "Section",
 CellChangeTimes->{{3.5054543012647257`*^9, 3.505454323450515*^9}, {
   3.510907714373198*^9, 3.5109077167963367`*^9}, 3.510908178125723*^9}],

Cell[CellGroupData[{

Cell["About This Model", "Subsection",
 CellChangeTimes->{{3.510394061790189*^9, 3.5103940822418246`*^9}, {
  3.510907707872826*^9, 3.5109077115670376`*^9}}],

Cell["\<\
A toy-model with  a simple pendulum, suspending LIGO's mirror (25 cm \
diameter, 10 cm length and 10 kg weight) with four 50-cm wires.
You can see the effects of asymmetry by changing the geometry of suspension \
points and the length of each wire.
\
\>", "Text",
 CellChangeTimes->{{3.505454452255293*^9, 3.505454488921332*^9}, {
   3.5077129980500813`*^9, 3.5077130433695107`*^9}, {3.5078876676897573`*^9, 
   3.5078878698873224`*^9}, {3.507891500540996*^9, 3.5078915759671288`*^9}, {
   3.508056919436692*^9, 3.508056920357094*^9}, {3.510355123371209*^9, 
   3.51035514664645*^9}, {3.5103918876678514`*^9, 3.5103918944872417`*^9}, {
   3.510623942883439*^9, 3.510624004228547*^9}, {3.510624128264365*^9, 
   3.5106243303003197`*^9}, {3.510907735689417*^9, 3.510907763414003*^9}, {
   3.510909653179636*^9, 3.510909653834837*^9}, {3.5109584516502295`*^9, 
   3.5109584654874535`*^9}, {3.5109980702504997`*^9, 3.510998172181079*^9}, {
   3.5111293295678463`*^9, 3.5111293498580065`*^9}, {3.511129506365958*^9, 
   3.511129532635461*^9}, {3.511214054289789*^9, 3.511214075686013*^9}, {
   3.5127249187349544`*^9, 3.5127249762990556`*^9}, {3.512725013333521*^9, 
   3.5127252632303596`*^9}, {3.51277378189983*^9, 3.5127737981550584`*^9}, 
   3.5127758372306433`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Version History", "Subsection",
 CellChangeTimes->{{3.5109077795259247`*^9, 3.510907781974065*^9}}],

Cell["\<\
04/26/10 - First release.
\
\>", "Text",
 CellChangeTimes->{{3.510907788475436*^9, 3.510907825669564*^9}, {
   3.5109981953783193`*^9, 3.5109981973127227`*^9}, 3.5111293590005293`*^9, {
   3.511301582319485*^9, 3.5113015951115074`*^9}, {3.5127252375215144`*^9, 
   3.512725238395116*^9}, {3.512725273198777*^9, 3.5127252867084007`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Coordinate System", "Subsection",
 CellChangeTimes->{{3.5103935324149256`*^9, 3.510393535246088*^9}, {
  3.5103940971398506`*^9, 3.5103941199158907`*^9}}],

Cell["\<\
(VIRGO's Reference)
X-axis:  Transversal, Y-axis: Vertical, Z-axis: Longitudinal (Beam-axis)
Pitch: Rotation around x-axis, Yaw: Rotation around y-axis, Roll: Rotation \
around z-axis
\
\>", "Text",
 CellChangeTimes->{{3.5103935528940973`*^9, 3.510393696350746*^9}, {
   3.510394123098296*^9, 3.5103943553983045`*^9}, {3.510395802459265*^9, 
   3.510395835515723*^9}, {3.510624429500894*^9, 3.510624435179304*^9}, {
   3.5109079621453695`*^9, 3.510908048918333*^9}, {3.5109081088107586`*^9, 
   3.5109081217995014`*^9}, 3.512725284446397*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Preliminaries", "Section",
 CellChangeTimes->{{3.509358217720311*^9, 3.5093582194831142`*^9}, {
  3.509358283131226*^9, 3.5093582840048275`*^9}, {3.5109081602447004`*^9, 
  3.5109081665040584`*^9}}],

Cell[CellGroupData[{

Cell["Clear Values", "Subsection",
 CellChangeTimes->{{3.5106377661991253`*^9, 3.5106377726731367`*^9}, {
  3.510908300193705*^9, 3.510908303030867*^9}, {3.510908796062067*^9, 
  3.5109088095318375`*^9}}],

Cell["Clear values and definitions of all the symbols.", "Text",
 CellChangeTimes->{{3.5109082594663754`*^9, 3.510908268789909*^9}, {
  3.5109083050389824`*^9, 3.5109083170736704`*^9}, {3.510908356790942*^9, 
  3.5109083909098935`*^9}, {3.5109084458170342`*^9, 3.510908446235058*^9}, {
  3.5112987930589027`*^9, 3.5112987971351357`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"Clear", "[", "\"\<Global`*\>\"", "]"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5101923438572636`*^9, 3.510192346852469*^9}, {
   3.5105415211971025`*^9, 3.5105415220863047`*^9}, {3.5106246850761433`*^9, 
   3.5106246954033613`*^9}, {3.5106377478222933`*^9, 3.510637755653507*^9}, 
   3.5109583818557067`*^9, {3.511298860191743*^9, 3.511298870315322*^9}, {
   3.512776682689728*^9, 3.5127766927361455`*^9}, {3.5181489348507547`*^9, 
   3.518148938059938*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Alart Switch OFF", "Subsection",
 CellChangeTimes->{{3.5093582266123266`*^9, 3.509358235441942*^9}, {
  3.510908785327453*^9, 3.510908786812538*^9}, {3.512712672876814*^9, 
  3.51271269337525*^9}}],

Cell[CellGroupData[{

Cell["\<\
Disable spell-checking. There are so many variables with similar names that \
it' s useless.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5109082158268795`*^9, 3.5109082167409315`*^9}, 
   3.512712667838005*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"Off", "[", 
   RowBox[{
    RowBox[{"General", "::", "spell"}], ",", 
    RowBox[{"General", "::", "spell1"}]}], "]"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5106245299962707`*^9, 3.510624543927095*^9}, {
   3.5106247451362486`*^9, 3.5106247454950495`*^9}, {3.5106377976799808`*^9, 
   3.5106377985379825`*^9}, 3.510958382947708*^9, 3.512712667838005*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Disable the LinerSolve alart.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511418292811921*^9, 3.5114183319367895`*^9}, {
  3.5114185384811525`*^9, 3.5114185527395773`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"Off", "[", 
   RowBox[{"LinearSolve", "::", "luc"}], "]"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5110035674345737`*^9, 3.5110035736277847`*^9}, {
  3.5114183319367895`*^9, 3.511418334307994*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Set Kernel Directory", "Subsection",
 CellChangeTimes->{{3.5101923522344785`*^9, 3.510192358162489*^9}}],

Cell["\<\
Set the current working directory to the directory where this notebook exists.
Imported data and packages should be placed in this directory, or the parent \
directory.\
\>", "Text",
 CellChangeTimes->{{3.5109084715615067`*^9, 3.510908486584366*^9}, {
   3.510908577266553*^9, 3.510908645640464*^9}, {3.5112989682869253`*^9, 
   3.5112990157796416`*^9}, {3.5113016406323876`*^9, 3.5113016472467995`*^9}, 
   3.5114313091697197`*^9, 3.5114315039128633`*^9, {3.512725476732334*^9, 
   3.5127254897115574`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"SetDirectory", "[", 
    RowBox[{"NotebookDirectory", "[", "]"}], "]"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{3.510192369066908*^9, 3.510958384320511*^9, 
  3.5114313091697197`*^9, 3.5114315039128633`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Import Packages", "Subsection",
 CellChangeTimes->{{3.509358549500699*^9, 3.5093585508579016`*^9}, {
  3.510908771531664*^9, 3.510908776462946*^9}, {3.511431144979431*^9, 
  3.5114311459622326`*^9}}],

Cell[CellGroupData[{

Cell["Standard package for use in plotting transfer functions.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109086828285904`*^9, 3.510908700501601*^9}, {
  3.5109087550517216`*^9, 3.5109087629341726`*^9}, {3.5114311527794447`*^9, 
  3.511431169050273*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"<<", "\"\<PlotLegends`\>\""}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{
  3.5093585572227125`*^9, 3.5109583868009152`*^9, 3.5114311527794447`*^9, {
   3.5114314680016003`*^9, 3.5114314701388035`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Manual package for modeling and simulating vibration isolation systems.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5114311609694595`*^9, 3.511431194384718*^9}, 
   3.511431248501213*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"<<", "\"\<../visUtil.m\>\""}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5114311961163206`*^9, 3.5114312118099484`*^9}, {
   3.5114312446636057`*^9, 3.511431248501213*^9}, {3.51143132645455*^9, 
   3.511431328217353*^9}, {3.5114314592187843`*^9, 3.51143146218279*^9}, {
   3.511432205581296*^9, 3.5114322437545633`*^9}, 3.5126997613310375`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Unit System", "Subsection",
 CellChangeTimes->{{3.507762673125991*^9, 3.5077626748989697`*^9}, {
  3.507887916125967*^9, 3.5078879248934684`*^9}, {3.510909011354381*^9, 
  3.5109090133324947`*^9}}],

Cell["\<\
MKS unit system is set as defalut. If you want to use another unit system or \
add units, please edit this subsection.\
\>", "Text",
 CellChangeTimes->{{3.510909029719432*^9, 3.5109090347067165`*^9}, {
  3.5112991343644247`*^9, 3.511299152603468*^9}, {3.5112992219044313`*^9, 
  3.5112992467198505`*^9}, {3.5112993190699887`*^9, 3.5112993260563884`*^9}, {
  3.5112996874180574`*^9, 3.5112997335146937`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"kg", "=", "1.0"}], ";", 
   RowBox[{"meter", "=", "1.0"}], ";", 
   RowBox[{"rad", "=", "1.0"}], ";", 
   RowBox[{"sec", "=", "1.0"}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Basical", " ", "Set"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"cm", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "2"}]], "meter"}]}], ";", 
   RowBox[{"mm", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "meter"}]}], ";", " ", 
   RowBox[{"km", "=", 
    RowBox[{
     SuperscriptBox["10", "3"], "meter"}]}], ";"}], 
  RowBox[{"(*", "Length", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"gram", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "kg"}]}], ";", " ", 
   RowBox[{"mg", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "6"}]], "kg"}]}], ";"}], 
  RowBox[{"(*", "Mass", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"deg", "=", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"N", "[", "Pi", "]"}], "/", "180"}], ")"}], "rad"}]}], ";", 
   " ", 
   RowBox[{"mrad", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "rad"}]}], ";"}], 
  RowBox[{"(*", "Angle", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"hertz", "=", 
    RowBox[{"1", "/", "sec"}]}], ";", 
   RowBox[{"mHz", "=", 
    RowBox[{
     SuperscriptBox["10", 
      RowBox[{"-", "3"}]], "hertz"}]}], ";", " ", 
   RowBox[{"kHz", "=", 
    RowBox[{
     SuperscriptBox["10", "3"], "hertz"}]}], ";"}], 
  RowBox[{"(*", "Frequency", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"newton", "=", 
    RowBox[{"kg", " ", 
     RowBox[{"meter", "/", 
      SuperscriptBox["sec", "2"]}]}]}], ";", 
   RowBox[{"pascal", "=", 
    RowBox[{"newton", "/", 
     SuperscriptBox["meter", "2"]}]}], ";"}], 
  RowBox[{"(*", "Force", "*)"}]}], "\n"}], "Input",
 CellChangeTimes->{{3.505459898591352*^9, 3.5054599656528482`*^9}, {
   3.505460214618761*^9, 3.505460246823139*^9}, {3.505460353839837*^9, 
   3.505460386329871*^9}, {3.5054609432258244`*^9, 3.505460943823134*^9}, {
   3.507717502489213*^9, 3.50771751561602*^9}, {3.507763954546257*^9, 
   3.50776398513273*^9}, {3.507887947930786*^9, 3.5078880664505653`*^9}, {
   3.507888099557459*^9, 3.50788814661615*^9}, {3.507888389647051*^9, 
   3.5078883993246045`*^9}, {3.507888436328721*^9, 3.507888448082393*^9}, 
   3.5109090416871157`*^9, 3.5109090778881865`*^9, {3.510909120506624*^9, 
   3.5109091273490157`*^9}, {3.5109092151620665`*^9, 3.510909215271267*^9}, {
   3.511300808916927*^9, 3.511300836123375*^9}, {3.511300869803834*^9, 
   3.5113008723622384`*^9}}]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Constant Values", "Section",
 CellChangeTimes->{{3.51142744973974*^9, 3.5114274556677504`*^9}, {
  3.511427538098295*^9, 3.5114275409375*^9}, {3.5126981805490613`*^9, 
  3.512698220251131*^9}, {3.5126995238362207`*^9, 3.5126995252714233`*^9}, {
  3.512709794472556*^9, 3.5127098002289667`*^9}}],

Cell[CellGroupData[{

Cell["Physical Constant", "Subsection",
 CellChangeTimes->{{3.5126995351930404`*^9, 3.512699537813845*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"g", "=", 
    RowBox[{"9.81", " ", 
     RowBox[{"meter", "/", 
      SuperscriptBox["sec", "2"]}]}]}], ";"}], " ", 
  RowBox[{"(*", 
   RowBox[{"local", " ", "gravity", " ", "constant"}], "*)"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.507762704553597*^9, 3.5077627272070923`*^9}, {
   3.507763249921364*^9, 3.507763251940638*^9}, 3.507765884859458*^9, {
   3.5078741427757235`*^9, 3.5078741430097237`*^9}, {3.5078882875472107`*^9, 
   3.5078883798324895`*^9}, 3.510910789651633*^9, {3.5127255934829397`*^9, 
   3.512725594387741*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Material Property", "Subsection",
 CellChangeTimes->{{3.5078885954778233`*^9, 3.5078886008591313`*^9}}],

Cell[CellGroupData[{

Cell["Usage note for symbols", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127058900508986`*^9, 3.5127058993329153`*^9}, {
  3.512706096907262*^9, 3.512706100963269*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "::", "usage"}], "=", 
   "\"\<rho['mat'] is volume density of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"young", "::", "usage"}], "=", 
   "\"\<young['mat'] is Young's modulus of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "::", "usage"}], "=", 
   "\"\<phi['mat'] is loss angle of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"poisson", "::", "usage"}], "=", 
   "\"\<poisson['mat'] is Poisson ratio of the material.\>\""}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127059638078284`*^9, 3.512706105643277*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Maraging Steel (MS)", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5103965360817537`*^9, 3.510396544037768*^9}, 
   3.5127059285829663`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "[", "\"\<MS\>\"", "]"}], "=", 
   RowBox[{"8.0", 
    RowBox[{"gram", "/", 
     SuperscriptBox["cm", "3"]}]}]}], ";", 
  RowBox[{
   RowBox[{"young", "[", "\"\<MS\>\"", "]"}], "=", 
   RowBox[{"195", " ", 
    SuperscriptBox["10", "9"], " ", "pascal"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "[", "\"\<MS\>\"", "]"}], "=", 
   RowBox[{"1.0", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "3"}]]}]}], ";", 
  RowBox[{
   RowBox[{"poisson", "[", "\"\<MS\>\"", "]"}], "=", "0.9"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5078886139238787`*^9, 3.507888680536689*^9}, {
   3.507888827152075*^9, 3.5078888913997498`*^9}, {3.5078889230655603`*^9, 
   3.507889061793495*^9}, {3.507889125559222*^9, 3.5078891537796717`*^9}, {
   3.507889406024915*^9, 3.507889441421377*^9}, {3.5078960325775614`*^9, 
   3.5078960964908733`*^9}, {3.5103965183445225`*^9, 3.510396588887847*^9}, {
   3.510909272975768*^9, 3.510909298450613*^9}, 3.510910792100837*^9, {
   3.510917538160895*^9, 3.5109175628401384`*^9}, {3.5110012657717195`*^9, 
   3.5110013428284593`*^9}, {3.5113005011767845`*^9, 
   3.5113005170108128`*^9}, {3.511300658220662*^9, 3.511300688422315*^9}, {
   3.512355711763343*^9, 3.512355728764315*^9}, {3.512699581587522*^9, 
   3.5126995827419243`*^9}, 3.5127059285829663`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Tungsten (W)", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5103965944882565`*^9, 3.5103965978578625`*^9}, 
   3.512705931312971*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "[", "\"\<W\>\"", "]"}], "=", 
   RowBox[{"19.25", 
    RowBox[{"gram", "/", 
     SuperscriptBox["cm", "3"]}]}]}], ";", " ", 
  RowBox[{
   RowBox[{"young", "[", "\"\<W\>\"", "]"}], "=", 
   RowBox[{"411", " ", 
    SuperscriptBox["10", "9"], " ", "pascal"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "[", "\"\<W\>\"", "]"}], "=", 
   RowBox[{"1.0", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "3"}]]}]}], ";", 
  RowBox[{
   RowBox[{"poisson", "[", "\"\<W\>\"", "]"}], "=", "0.3"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5078886139238787`*^9, 3.507888680536689*^9}, {
   3.507888827152075*^9, 3.5078888913997498`*^9}, {3.5078889230655603`*^9, 
   3.507889061793495*^9}, {3.507889125559222*^9, 3.5078891537796717`*^9}, {
   3.507889406024915*^9, 3.5078894378489714`*^9}, {3.507896109594897*^9, 
   3.507896113026903*^9}, {3.5103966082162805`*^9, 3.5103966252515106`*^9}, 
   3.5107455826730013`*^9, {3.5109092790285788`*^9, 3.5109092799021807`*^9}, 
   3.510910793146039*^9, {3.510917571108153*^9, 3.510917577987765*^9}, {
   3.5110013469000664`*^9, 3.5110013527812767`*^9}, {3.5110014676084857`*^9, 
   3.5110014906029263`*^9}, {3.5113002673217673`*^9, 
   3.5113002845129976`*^9}, {3.5113004092804227`*^9, 3.511300412774829*^9}, {
   3.511300542797658*^9, 3.5113005435620594`*^9}, {3.5113006258066044`*^9, 
   3.511300626805006*^9}, 3.51130069099632*^9, {3.512355660889433*^9, 
   3.5123556611114454`*^9}, {3.512699585487529*^9, 3.5126995863299303`*^9}, 
   3.512705931312971*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["C-70 Steel (C70)", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.5107455881642113`*^9, 3.5107455934682207`*^9}, 
   3.510910365408888*^9, 3.512705934167776*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"rho", "[", "\"\<C70\>\"", "]"}], "=", 
   RowBox[{"7.8", 
    RowBox[{"gram", "/", 
     SuperscriptBox["cm", "3"]}]}]}], ";", " ", 
  RowBox[{
   RowBox[{"young", "[", "\"\<C70\>\"", "]"}], "=", 
   RowBox[{"200", " ", 
    SuperscriptBox["10", "9"], " ", "pascal"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"phi", "[", "\"\<C70\>\"", "]"}], "=", 
   RowBox[{"1.0", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "3"}]]}]}], ";", 
  RowBox[{
   RowBox[{"poisson", "[", "\"\<C70\>\"", "]"}], "=", "0.3"}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.5078886139238787`*^9, 3.507888680536689*^9}, {
   3.507888827152075*^9, 3.5078888913997498`*^9}, {3.5078889230655603`*^9, 
   3.507889061793495*^9}, {3.507889125559222*^9, 3.5078891537796717`*^9}, {
   3.507889406024915*^9, 3.5078894378489714`*^9}, {3.507896109594897*^9, 
   3.507896113026903*^9}, {3.5103966082162805`*^9, 3.5103966252515106`*^9}, {
   3.5107455826730013`*^9, 3.5107456317818875`*^9}, {3.5109092830533857`*^9, 
   3.5109092838801875`*^9}, 3.510910794362841*^9, {3.510917573572957*^9, 
   3.510917582168572*^9}, {3.5110014921005287`*^9, 3.5110015006025434`*^9}, {
   3.5113005549188795`*^9, 3.511300557196483*^9}, {3.5113006740078897`*^9, 
   3.5113006841791077`*^9}, {3.512355689126048*^9, 3.512355699808659*^9}, {
   3.512699588825935*^9, 3.512699589590336*^9}, 3.512705934167776*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Bodies", "Subsection",
 CellChangeTimes->{{3.5106251018152747`*^9, 3.5106251044516797`*^9}, {
  3.5109104340490084`*^9, 3.5109104353594103`*^9}}],

Cell[CellGroupData[{

Cell["Standard Bodies", "Subsubsection",
 CellChangeTimes->{{3.511311704314874*^9, 3.5113117105236855`*^9}, 
   3.5127091783182745`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"mTM", "=", 
   RowBox[{"10.7", "kg"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"ixTM", "=", 
   RowBox[{".051", "kg", " ", 
    SuperscriptBox["meter", "2"]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"iyTM", "=", 
   RowBox[{".051", "kg", " ", 
    SuperscriptBox["meter", "2"]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"izTM", "=", 
   RowBox[{".084", "kg", " ", 
    SuperscriptBox["meter", "2"]}]}], ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511304710127389*^9, 3.511304728535421*^9}, {
   3.5113047674418893`*^9, 3.5113048860488977`*^9}, {3.5113050074015107`*^9, 
   3.51130514904976*^9}, {3.5113051832138195`*^9, 3.51130538269137*^9}, {
   3.511305417479431*^9, 3.511305551904867*^9}, {3.5113058796614428`*^9, 
   3.511305880269844*^9}, {3.5113061359454927`*^9, 3.5113061641503425`*^9}, {
   3.511306624819152*^9, 3.5113066315271635`*^9}, 3.511307473382642*^9, {
   3.511308315877722*^9, 3.511308321212931*^9}, {3.511386760794197*^9, 
   3.511386828105047*^9}, {3.5113869415443287`*^9, 3.511386973617985*^9}, {
   3.5114939721454124`*^9, 3.511493984048234*^9}, {3.5114942197842493`*^9, 
   3.511494225041459*^9}, {3.51149502317006*^9, 3.5114950728873477`*^9}, {
   3.512355081541296*^9, 3.5123550980172386`*^9}, {3.5123562660702295`*^9, 
   3.512356447654548*^9}, {3.5126972323489933`*^9, 3.512697253565031*^9}, {
   3.5126973286319647`*^9, 3.5126973856968646`*^9}, {3.5126974251337337`*^9, 
   3.512697450421378*^9}, {3.512697521946704*^9, 3.512697534910327*^9}, {
   3.5126975666875825`*^9, 3.5126976978682127`*^9}, {3.512697732281873*^9, 
   3.512697844960871*^9}, {3.5126983687633915`*^9, 3.5126983919918327`*^9}, {
   3.5127063777233553`*^9, 3.51270638635017*^9}, {3.512711677657466*^9, 
   3.5127117195591393`*^9}, {3.5127117556108027`*^9, 
   3.5127119120322776`*^9}, {3.5127256372098165`*^9, 3.5127256429818263`*^9}, 
   3.512730602065339*^9, {3.512772450558288*^9, 3.5127724661115155`*^9}, {
   3.512773848387147*^9, 3.5127738638467736`*^9}, {3.5127758895219345`*^9, 
   3.5127759473824368`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Wires", "Subsection",
 CellChangeTimes->{{3.511306120572666*^9, 3.5113061311874847`*^9}}],

Cell[CellGroupData[{

Cell["Properties and Geometric Values", "Subsubsection",
 CellChangeTimes->{{3.511311536146579*^9, 3.5113115410137873`*^9}, {
  3.5113118928256054`*^9, 3.5113118955244102`*^9}}],

Cell[CellGroupData[{

Cell["Usage note for symbols.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113114603616457`*^9, 3.5113114747136707`*^9}, {
   3.511311505554925*^9, 3.511311549718603*^9}, 3.5126982410615673`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"nw", "::", "usage"}], "=", "\"\<number of wires\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"matw", "::", "usage"}], "=", "\"\<wire material\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"lNw", "::", "usage"}], "=", "\"\<natural length of wires\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dw", "::", "usage"}], "=", "\"\<wire diameters\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dxu", "::", "usage"}], "=", 
   "\"\<x-separation of wires & COM at upper suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dzu", "::", "usage"}], "=", 
   "\"\<z-separation of wires & COM at upper suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dxl", "::", "usage"}], "=", 
   "\"\<x-separation of wires & COM at lower suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dzl", "::", "usage"}], "=", 
   "\"\<z-separation of wires & COM at lower suspension points\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dyu", "::", "usage"}], "=", 
   "\"\<height of wire break-off below COM at upper body\>\""}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dyl", "::", "usage"}], "=", 
   "\"\<height of wire break-off above COM at lower body\>\""}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113096085915933`*^9, 3.5113098531844225`*^9}, {
   3.511309893479294*^9, 3.5113099178465366`*^9}, 3.511309950247793*^9, {
   3.511311488410495*^9, 3.511311526942563*^9}, 3.5113153674308243`*^9, 
   3.5126982410615673`*^9, 3.512706179431407*^9, {3.512711537023219*^9, 
   3.512711552982047*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Values", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113115557402134`*^9, 3.5113115654278307`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", "      ", "TM", "     ", "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"matw", "=", 
     RowBox[{"{", "\"\<W\>\"", "}"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"lNw", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", 
        RowBox[{"50", ",", "50", ",", "50", ",", "50"}], "}"}], "}"}], 
      "cm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dw", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", 
        RowBox[{"0.20", ",", "0.20", ",", "0.20", ",", "0.20"}], "}"}], "}"}],
       "mm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dyu", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", 
        RowBox[{"0.00", ",", "0.00", ",", "0.00", ",", "0.00"}], "}"}], "}"}],
       "mm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dyl", "=", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", 
        RowBox[{"1.00", ",", "1.00", ",", "1.00", ",", "1.00"}], "}"}], "}"}],
       "mm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dxu", "=", 
     RowBox[{
      RowBox[{"{", "12.5", "}"}], "cm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dzu", "=", 
     RowBox[{
      RowBox[{"{", "1.50", "}"}], "cm"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"dxl", "=", "dxu"}], ";", 
    RowBox[{"dzl", "=", "dzu"}], ";"}], 
   RowBox[{"(*", 
    RowBox[{"vertical", " ", "suspension"}], "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"nw", "=", 
     RowBox[{"{", "4", "}"}]}], ";"}], "\[IndentingNewLine]"}]}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5126979552998652`*^9, 3.5126979725066953`*^9}, {
   3.512698487807201*^9, 3.512698593512986*^9}, 3.512699268042571*^9, {
   3.512706425599839*^9, 3.5127064636951065`*^9}, {3.512707714364903*^9, 
   3.512707749839365*^9}, {3.5127079340132885`*^9, 3.512707950642918*^9}, {
   3.5127115595340586`*^9, 3.5127115689564753`*^9}, {3.5127119288491073`*^9, 
   3.5127120018884354`*^9}, {3.5127120337436914`*^9, 3.512712065645747*^9}, {
   3.5127121919123693`*^9, 3.5127122302416363`*^9}, {3.5127259305527315`*^9, 
   3.5127259383683453`*^9}, 3.512730513737984*^9, {3.512773898042034*^9, 
   3.5127739340936975`*^9}, {3.5127744525294075`*^9, 3.5127744565542145`*^9}, 
   3.5127745474399743`*^9, {3.51277597774009*^9, 3.5127760515438194`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Effective Bending Point Compensation", "Subsubsection",
 CellChangeTimes->{{3.511310011805502*^9, 3.511310013724305*^9}, {
  3.5113101034712625`*^9, 3.511310109976474*^9}, {3.511312045081873*^9, 
  3.5113120493874807`*^9}}],

Cell[CellGroupData[{

Cell["Discription", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511310024020323*^9, 3.5113100245975237`*^9}, {
   3.5113101204128923`*^9, 3.511310123595298*^9}, 3.511311097317808*^9}],

Cell["\<\
Bending point of wire is apart from the suspension point because of the wire \
bending stiffness.
In this subsubsection, the separation between effective bending point & \
suspension point is estimated (1/lambda),
and then wire length and wire break-off are compensated. If you want to \
disable this compensation, 
please make the following symbol \"False\".\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511310128790107*^9, 3.511310207960246*^9}, {
   3.511310336208071*^9, 3.511310459167487*^9}, {3.511310496030352*^9, 
   3.511310656320634*^9}, {3.51131089921306*^9, 3.5113110646045504`*^9}, 
   3.511311097317808*^9, 3.5113135489584303`*^9, {3.512725999270852*^9, 
   3.51272600367006*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"wirecompensation", "=", "True"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511311074011367*^9, 3.511311097317808*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Load weight", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511308651590311*^9, 3.511308660326327*^9}, {
  3.51131116687833*^9, 3.51131117236954*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"loadw", "=", 
      RowBox[{"{", "}"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"AppendTo", "[", 
      RowBox[{"loadw", ",", "mTM"}], "]"}], ";"}]}], "\[IndentingNewLine]", 
   "]"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113086367702856`*^9, 3.5113087745029273`*^9}, {
   3.5113088097589893`*^9, 3.5113088188382053`*^9}, {3.5113088861367235`*^9, 
   3.511308890411131*^9}, {3.51131110959503*^9, 3.5113111384394803`*^9}, 
   3.51131117236954*^9, {3.512706614687772*^9, 3.512706633719805*^9}, {
   3.512706853539791*^9, 3.5127068600294027`*^9}, {3.5127069021182766`*^9, 
   3.512706931742728*^9}, 3.5127739589133406`*^9, {3.512776715215785*^9, 
   3.5127767234838*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Second moment of area", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113089605956545`*^9, 3.511309018003755*^9}, 
   3.5113093541395464`*^9, {3.511309392125613*^9, 3.511309434089687*^9}, {
   3.511311157487114*^9, 3.5113111858791637`*^9}, 3.511311299977764*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"iw", "=", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"N", "[", "Pi", "]"}], 
         RowBox[{
          SuperscriptBox[
           RowBox[{"dw", "[", 
            RowBox[{"[", 
             RowBox[{"n", ",", "m"}], "]"}], "]"}], "4"], "/", "64"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "dw", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"nw", "[", 
           RowBox[{"[", "n", "]"}], "]"}]}], "}"}]}], "]"}]}], ";"}]}], 
   "\[IndentingNewLine]", "]"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113091838496475`*^9, 3.5113092569513755`*^9}, {
   3.5113093129398737`*^9, 3.5113093236882925`*^9}, 3.5113093541395464`*^9, {
   3.511311150248701*^9, 3.5113111520583043`*^9}, {3.5113111858791637`*^9, 
   3.51131121202481*^9}, 3.511311299977764*^9, {3.5113151870237074`*^9, 
   3.5113152264761767`*^9}, {3.511315400799283*^9, 3.511315403076887*^9}, {
   3.512358626588375*^9, 3.512358627446377*^9}, 3.512706501493973*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Lambda", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511311218998022*^9, 3.5113112216032267`*^9}, 
   3.511311306186575*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"lambdaw", "=", 
      RowBox[{"Table", "[", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Sqrt", "[", 
         RowBox[{
          RowBox[{"loadw", "[", 
           RowBox[{"[", "n", "]"}], "]"}], 
          RowBox[{
           RowBox[{
            RowBox[{"g", "/", 
             RowBox[{"nw", "[", 
              RowBox[{"[", "n", "]"}], "]"}]}], "/", 
            RowBox[{"young", "[", 
             RowBox[{"matw", "[", 
              RowBox[{"[", "n", "]"}], "]"}], "]"}]}], "/", 
           RowBox[{"iw", "[", 
            RowBox[{"[", 
             RowBox[{"n", ",", "m"}], "]"}], "]"}]}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "dw", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"nw", "[", 
           RowBox[{"[", "n", "]"}], "]"}]}], "}"}]}], "\[IndentingNewLine]", 
       "]"}]}], ";"}]}], "\[IndentingNewLine]", "]"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511309025819369*^9, 3.5113091071423116`*^9}, {
   3.511309263518987*^9, 3.5113093858388023`*^9}, 3.5113111858791637`*^9, {
   3.5113112437708654`*^9, 3.5113112539732833`*^9}, 3.511311306186575*^9, {
   3.5113134454678483`*^9, 3.5113134500074563`*^9}, {3.5113152729798584`*^9, 
   3.5113152861306815`*^9}, {3.511315413669306*^9, 3.511315416804911*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Compensation", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.511309454182522*^9, 3.511309456428926*^9}, {
   3.5113113155153913`*^9, 3.511311321724202*^9}, 3.5113115879074697`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"If", "[", 
   RowBox[{"wirecompensation", ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"lNw", "=", 
      RowBox[{"lNw", "-", 
       RowBox[{"2", "/", "lambdaw"}]}]}], ";", 
     RowBox[{"(*", 
      RowBox[{"wire", " ", "length"}], "*)"}], "\[IndentingNewLine]", 
     RowBox[{"dyu", "=", 
      RowBox[{"dyu", "+", 
       RowBox[{"1", "/", "lambdaw"}]}]}], ";", 
     RowBox[{"(*", 
      RowBox[{
       RowBox[{"wire", " ", "break"}], "-", 
       RowBox[{"off", " ", "at", " ", "upper", " ", "body"}]}], "*)"}], 
     "\[IndentingNewLine]", 
     RowBox[{"dyl", "=", 
      RowBox[{"dyl", "+", 
       RowBox[{"1", "/", "lambdaw"}]}]}], ";"}]}], 
   RowBox[{"(*", 
    RowBox[{
     RowBox[{"wire", " ", "break"}], "-", 
     RowBox[{"off", " ", "at", " ", "lower", " ", "body"}]}], "*)"}], 
   "\[IndentingNewLine]", "]"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.},
 CellChangeTimes->{{3.5113113332994223`*^9, 3.5113114085071545`*^9}, {
  3.5113115879074697`*^9, 3.5113116037102976`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Suspension Points", "Subsubsection",
 CellChangeTimes->{{3.511313811132491*^9, 3.5113138177781024`*^9}, 
   3.5113138856070213`*^9}],

Cell[CellGroupData[{

Cell["Description", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511327852568959*^9, 3.511327855345764*^9}, 
   3.51132791554627*^9}],

Cell["\<\
The clamp positions of wires in local coordinate of the attached mass.
If you want to add geometric asymmetry to the system, please arrange the \
following definition.
\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113160716784616`*^9, 3.511316135513774*^9}, {
   3.511316199583086*^9, 3.511316246585969*^9}, 3.51132791554627*^9, 
   3.5126992435505285`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Body corrdinates for all objects originate at COM", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.510968232951127*^9, 3.510968236383133*^9}, 
   3.511327918369875*^9, {3.5113279682587624`*^9, 3.511327969631565*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"posCOM", "=", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5109682272727165`*^9, 3.5109682511563587`*^9}, 
   3.511327918369875*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Suspension Points at upper bodies", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113278633641787`*^9, 3.511327866530984*^9}, 
   3.5113279214898806`*^9, {3.51269895991083*^9, 3.5126989766652594`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"spu", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"(*", "Ground", "*)"}], "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{"dxu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dyu", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dzu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{"dxu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dyu", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "2"}], "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dzu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"+", 
          RowBox[{"dxu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dyu", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "3"}], "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dzu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"+", 
          RowBox[{"dxu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dyu", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "4"}], "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dzu", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}]}], "}"}], 
     "\[IndentingNewLine]", "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.510541271815065*^9, 3.5105413275383625`*^9}, {
   3.510541364338827*^9, 3.5105414659262056`*^9}, {3.5105415729735937`*^9, 
   3.5105416770257764`*^9}, {3.510541718537449*^9, 3.510541745213496*^9}, {
   3.5105418057416024`*^9, 3.5105418656925077`*^9}, {3.5105419541758633`*^9, 
   3.510541964097481*^9}, {3.510562740817388*^9, 3.51056276442023*^9}, {
   3.5106273753944683`*^9, 3.5106273856904864`*^9}, {3.5106361554478965`*^9, 
   3.5106361572418995`*^9}, {3.510636263328886*^9, 3.5106362654972897`*^9}, {
   3.510636399979926*^9, 3.510636408606741*^9}, {3.510655831710415*^9, 
   3.510655836608824*^9}, {3.510910690123458*^9, 3.5109107397159452`*^9}, {
   3.5109112296192055`*^9, 3.5109112741416836`*^9}, {3.510999339027935*^9, 
   3.5109993921616287`*^9}, {3.511154589829262*^9, 3.51115459458727*^9}, {
   3.511219031273158*^9, 3.5112190316787586`*^9}, {3.5113138771362066`*^9, 
   3.511313877448207*^9}, {3.511313922329486*^9, 3.511314076176956*^9}, {
   3.5113144895308824`*^9, 3.511314559107004*^9}, {3.511315592929221*^9, 
   3.5113156070628457`*^9}, {3.5113156392145023`*^9, 
   3.5113160247691793`*^9}, {3.5113268353067727`*^9, 3.5113268435123873`*^9}, 
   3.5113279214898806`*^9, {3.5126986628863087`*^9, 3.512698934170785*^9}, 
   3.512699128297526*^9, {3.512699214409677*^9, 3.5126992208680887`*^9}, {
   3.512707995851797*^9, 3.5127080201878405`*^9}, {3.512712331173814*^9, 
   3.512712357038659*^9}, 3.512773990581396*^9, {3.5127745616203995`*^9, 
   3.5127745896224484`*^9}, {3.512776524989051*^9, 3.5127765520082984`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Suspension points at lower bodies", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.5126989944024906`*^9, 3.512699019736935*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"spl", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"(*", 
       RowBox[{"Test", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{"dxl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dyl", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dzl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{"dxl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dyl", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "2"}], "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dzl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"+", 
          RowBox[{"dxl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dyl", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "3"}], "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dzl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"+", 
          RowBox[{"dxl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}], ",", 
         RowBox[{"+", 
          RowBox[{"dyl", "[", 
           RowBox[{"[", 
            RowBox[{"1", ",", "4"}], "]"}], "]"}]}], ",", 
         RowBox[{"-", 
          RowBox[{"dzl", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}]}], "}"}], 
     "\[IndentingNewLine]", "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.510541271815065*^9, 3.5105413275383625`*^9}, {
   3.510541364338827*^9, 3.5105414659262056`*^9}, {3.5105415729735937`*^9, 
   3.5105416770257764`*^9}, {3.510541718537449*^9, 3.510541745213496*^9}, {
   3.5105418057416024`*^9, 3.5105418656925077`*^9}, {3.5105419541758633`*^9, 
   3.510541964097481*^9}, {3.510562740817388*^9, 3.51056276442023*^9}, {
   3.5106273753944683`*^9, 3.5106273856904864`*^9}, {3.5106361554478965`*^9, 
   3.5106361572418995`*^9}, {3.510636263328886*^9, 3.5106362654972897`*^9}, {
   3.510636399979926*^9, 3.510636408606741*^9}, {3.510655831710415*^9, 
   3.510655836608824*^9}, {3.510910690123458*^9, 3.5109107397159452`*^9}, {
   3.5109112296192055`*^9, 3.5109112741416836`*^9}, {3.510999339027935*^9, 
   3.5109993921616287`*^9}, {3.511154589829262*^9, 3.51115459458727*^9}, {
   3.511219031273158*^9, 3.5112190316787586`*^9}, {3.5113138771362066`*^9, 
   3.511313877448207*^9}, {3.511313922329486*^9, 3.511314076176956*^9}, {
   3.5113144895308824`*^9, 3.511314559107004*^9}, {3.511315592929221*^9, 
   3.5113156070628457`*^9}, {3.5113156392145023`*^9, 
   3.5113160247691793`*^9}, {3.5113268353067727`*^9, 3.5113268435123873`*^9}, 
   3.5113279214898806`*^9, {3.5126986628863087`*^9, 3.512698934170785*^9}, {
   3.512699019736935*^9, 3.512699078798639*^9}, {3.51269913639394*^9, 
   3.5126992265620985`*^9}, {3.5127080281594543`*^9, 3.512708062214314*^9}, {
   3.5127123629042697`*^9, 3.512712393121523*^9}, 3.5127740004250135`*^9, {
   3.5127745972664623`*^9, 3.51277462470691*^9}, {3.5127765616959157`*^9, 
   3.5127765778263435`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Actuators", "Subsection",
 CellChangeTimes->{{3.5113201245923815`*^9, 3.511320127103986*^9}}],

Cell[CellGroupData[{

Cell["Positions", "Subsubsection",
 CellChangeTimes->{{3.5113204481369495`*^9, 3.511320449650152*^9}}],

Cell[CellGroupData[{

Cell["Values", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113202091757298`*^9, 3.511320210205332*^9}, 
   3.511320410462883*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"dxTMa", "=", 
    RowBox[{"12", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"x", "-", 
    RowBox[{
    "separation", " ", "of", " ", "actuators", " ", "at", " ", "TM"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dyTMa", "=", 
    RowBox[{"12", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"y", "-", 
    RowBox[{
    "separation", " ", "of", " ", "actators", " ", "at", " ", "TM"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"dzTMa", "=", 
    RowBox[{"5", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{
    RowBox[{
     RowBox[{"z", "-", 
      RowBox[{
      "separation", " ", "between", " ", "COM", " ", "of", " ", "TM"}]}], " ",
      "&"}], " ", "actuators"}], "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113202124517355`*^9, 3.5113202217649517`*^9}, {
  3.51132026614703*^9, 3.511320410462883*^9}, {3.5113204870590177`*^9, 
  3.5113205365111046`*^9}, {3.5127760805754704`*^9, 3.5127760807626705`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Position of actuators in local coordinate", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113204715057907`*^9, 3.511320480023405*^9}, 
   3.511320711652612*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"posTMa", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "dxTMa"}], "/", "2"}], ",", 
        RowBox[{
         RowBox[{"-", "dyTMa"}], "/", "2"}], ",", 
        RowBox[{"-", "dzTMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "dxTMa"}], "/", "2"}], ",", 
        RowBox[{
         RowBox[{"+", "dyTMa"}], "/", "2"}], ",", 
        RowBox[{"-", "dzTMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"+", "dxTMa"}], "/", "2"}], ",", 
        RowBox[{
         RowBox[{"+", "dyTMa"}], "/", "2"}], ",", 
        RowBox[{"-", "dzTMa"}]}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"+", "dxTMa"}], "/", "2"}], ",", 
        RowBox[{
         RowBox[{"-", "dyTMa"}], "/", "2"}], ",", 
        RowBox[{"-", "dzTMa"}]}], "}"}]}], "\[IndentingNewLine]", "}"}]}], 
   ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511320549069127*^9, 3.5113207534450855`*^9}, {
   3.5113912026046133`*^9, 3.5113912502938967`*^9}, 3.5127760737894583`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Working Direction", "Subsubsection",
 CellChangeTimes->{{3.511320420556101*^9, 3.5113204221161036`*^9}, {
  3.5113204602893705`*^9, 3.5113204617401733`*^9}}],

Cell[CellGroupData[{

Cell["Working direction in local coordinate system at recoil body.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511320810213585*^9, 3.511320827560816*^9}, 
   3.511323594038475*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"dirTMa", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "\[IndentingNewLine]", 
     "}"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511320767765911*^9, 3.5113207953779593`*^9}, {
   3.5113208372172327`*^9, 3.5113208425680423`*^9}, 3.511323594038475*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Body Shape", "Subsection",
 CellChangeTimes->{{3.511303371161437*^9, 3.5113033797726517`*^9}}],

Cell[CellGroupData[{

Cell["Discription", "Subsubsection",
 CellChangeTimes->{{3.5113191638334937`*^9, 3.5113191686227016`*^9}}],

Cell["\<\
This subsection contains the definition of the body appearance for eigen-mode \
3D graphics.
Containing nothing important for eigen-mode calculation or transfer function \
plots.
The 3D graphics shows not only standard bodies, but also mass-less bodies; \
the ground and key stones.
If you do not want to show them, please make the following symbols \"False\".\
\>", "Text",
 CellChangeTimes->{{3.51131927562329*^9, 3.511319695373227*^9}, {
  3.5113201748088694`*^9, 3.5113201768836727`*^9}, {3.51277273048518*^9, 
  3.51277274766081*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"showground", "=", "True"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.511319700568036*^9, 3.511319716386464*^9}, {
  3.5113208734716964`*^9, 3.511320878869306*^9}, {3.5113305238848543`*^9, 
  3.5113305240252547`*^9}, {3.512776096971099*^9, 3.5127761011987066`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Shape Definition", "Subsubsection",
 CellChangeTimes->{{3.5113192607252636`*^9, 3.5113192614740653`*^9}, {
  3.511319726401682*^9, 3.5113197279772844`*^9}, {3.512708700551835*^9, 
  3.5127087296770864`*^9}}],

Cell[CellGroupData[{

Cell["Ground", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113059380991454`*^9, 3.5113059389259467`*^9}, 
   3.5113197356836977`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{"showground", ",", "\[IndentingNewLine]", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"shapeg", "=", "\"\<Doughnut\>\""}], ";", "\[IndentingNewLine]", 
      RowBox[{"rstage", "=", 
       RowBox[{"11", "cm"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"rgi", "=", 
       RowBox[{"rstage", "-", 
        RowBox[{"10", "cm"}]}]}], ";", 
      RowBox[{"(*", 
       RowBox[{"inner", " ", "radius"}], "*)"}], "\[IndentingNewLine]", 
      RowBox[{"rgo", "=", 
       RowBox[{"rstage", "+", 
        RowBox[{"10", "cm"}]}]}], ";", 
      RowBox[{"(*", 
       RowBox[{"outer", " ", "radius"}], "*)"}], "\[IndentingNewLine]", 
      RowBox[{"dimg", "=", 
       RowBox[{"{", 
        RowBox[{"rgi", ",", "rgo"}], "}"}]}], ";"}]}], "\[IndentingNewLine]", 
    "]"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113034020962915`*^9, 3.5113035097988806`*^9}, 
   3.5113035528549557`*^9, {3.511303899549965*^9, 3.5113039129503884`*^9}, {
   3.5113197356836977`*^9, 3.511319750269724*^9}, 3.511321098112091*^9, {
   3.512708410016925*^9, 3.5127084313577623`*^9}, {3.512774886069769*^9, 
   3.5127748932145815`*^9}, {3.5127761392627735`*^9, 
   3.5127761437399817`*^9}, {3.5127768399208045`*^9, 3.512776842619609*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Test Mass", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10006.},
 CellChangeTimes->{{3.5113042574613934`*^9, 3.5113042620946016`*^9}, {
  3.5113200209769993`*^9, 3.51132002713901*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"shapeTM", "=", "\"\<zCylinder\>\""}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"rTM", "=", 
    RowBox[{"12.5", "cm"}]}], ";"}], 
  RowBox[{"(*", "radius", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"lTM", "=", 
    RowBox[{"10", "cm"}]}], ";"}], 
  RowBox[{"(*", "length", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dimTM", "=", 
   RowBox[{"{", 
    RowBox[{"rTM", ",", "lTM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10006.},
 CellChangeTimes->{{3.5113042642630053`*^9, 3.511304330968723*^9}, 
   3.51132002713901*^9, {3.512708577545619*^9, 3.512708587061636*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Variables", "Section",
 CellChangeTimes->{{3.5106252414511204`*^9, 3.5106252530107403`*^9}, {
   3.510625426015044*^9, 3.5106254276374474`*^9}, {3.5106259504567657`*^9, 
   3.510625951330367*^9}, 3.510911145254257*^9}],

Cell[CellGroupData[{

Cell["Variable List", "Subsection",
 CellChangeTimes->{{3.510626076005786*^9, 3.510626107814242*^9}, {
  3.510911153662672*^9, 3.5109111566890774`*^9}, {3.510911385167079*^9, 
  3.5109113901902876`*^9}}],

Cell[CellGroupData[{

Cell["Variables Specifying Each Body", "Subsubsection",
 CellChangeTimes->{{3.5109114063519163`*^9, 3.5109114182391367`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"varg", "=", 
    RowBox[{"{", 
     RowBox[{
     "xg", ",", "yg", ",", "zg", ",", "pitchg", ",", "yawg", ",", "rollg"}], 
     "}"}]}], ";"}], 
  RowBox[{"(*", "Ground", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"varTM", "=", 
    RowBox[{"{", 
     RowBox[{
     "xTM", ",", "yTM", ",", "zTM", ",", "pitchTM", ",", "yawTM", ",", 
      "rollTM"}], "}"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Test", " ", "Mass"}], "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->CompressedData["
1:eJxTTMoPSmViYGAQBWIQfe6w/os/6q8cNbIdXoHoNt3tEh0arxyvHfghBaKP
VfFrgmgDUUUDEC00wyEURNtlbQDTJo8+ZW0G0t49ufkgWvFN/TYQ/ez8ETC9
R5I96gaQvmV6Jx5ECwneygPRW+5E5oPoB8kLC28C6XeF24tAtNuM4goQvYcl
rQlEM2zx3wCi/0ZqbwTRQV7VRz8B6W8fP4HpTv7aqyD6n8ZOMC0j/2jBHyBd
NOcFmPZT5bn3F0j/10t9AKLrVsV6/gPSbdFyXiC683RyN4hmm/gJTK9qepkr
p/nKUazTqwBEc/3UCwgA0lGceRkguqEo+YiozivHWd1xR0G0HtvE3WJAOuDj
YTANAG18qjI=
  "]]
}, Open  ]],

Cell[CellGroupData[{

Cell["All the Variables & Parameters", "Subsubsection",
 CellChangeTimes->{{3.510911424416748*^9, 3.5109114350559664`*^9}, {
  3.5109114803272457`*^9, 3.510911485444055*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"allparams", "=", 
    RowBox[{"Join", "[", "varg", "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"allvars", "=", 
    RowBox[{"Join", "[", "varTM", "]"}]}], ";"}], 
  RowBox[{"(*", "Variables", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"allfloats", "=", 
    RowBox[{"Join", "[", "]"}]}], ";"}], 
  RowBox[{"(*", "Floats", "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5105436285940065`*^9, 3.5105436779524927`*^9}, {
   3.5105505824294252`*^9, 3.510550583427827*^9}, {3.5105609245649977`*^9, 
   3.5105609275602026`*^9}, {3.5106256594538546`*^9, 
   3.5106256981731224`*^9}, {3.51062573105798*^9, 3.5106257692624474`*^9}, {
   3.510626150183916*^9, 3.5106261590603323`*^9}, {3.5106556411248803`*^9, 
   3.510655642092082*^9}, {3.5109113733266582`*^9, 3.510911378552667*^9}, {
   3.5109114473799877`*^9, 3.51091146552282*^9}, {3.5109116508113503`*^9, 
   3.5109116689697824`*^9}, {3.5109119407846594`*^9, 
   3.5109119478358717`*^9}, {3.510999937866187*^9, 3.510999943482197*^9}, {
   3.511075588996784*^9, 3.5110755901199865`*^9}, {3.511075685716954*^9, 
   3.511075686060155*^9}, {3.5110777999846897`*^9, 3.511077809531906*^9}, {
   3.5110778570807896`*^9, 3.511077864069602*^9}, {3.511077912663687*^9, 
   3.51107791422369*^9}, {3.5110782844591403`*^9, 3.5110782845839405`*^9}, {
   3.5111296569725723`*^9, 3.5111296786588125`*^9}, {3.511129758521381*^9, 
   3.5111297593664293`*^9}, {3.5111498442951202`*^9, 
   3.5111498693955646`*^9}, {3.5111523947904005`*^9, 
   3.5111523988152075`*^9}, {3.511153235678677*^9, 3.5111532386738825`*^9}, {
   3.511153792901862*^9, 3.5111537970826693`*^9}, {3.5112190816924467`*^9, 
   3.5112190848436522`*^9}, {3.5113214435902977`*^9, 
   3.5113214588627243`*^9}, {3.511321620947009*^9, 3.5113216299794254`*^9}, 
   3.5113217686324687`*^9, {3.5127741928045516`*^9, 3.512774194177354*^9}, {
   3.5127761646908183`*^9, 3.5127761692616262`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Velocities", "Subsubsection",
 CellChangeTimes->{{3.5109115271741285`*^9, 3.5109115296233325`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"velparams", "=", 
    RowBox[{"makevel", "[", "allparams", "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"velvars", "=", 
    RowBox[{"makevel", "[", "allvars", "]"}]}], ";"}], 
  RowBox[{"(*", "Variables", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"velfloats", "=", 
    RowBox[{"makevel", "[", "allfloats", "]"}]}], ";"}], 
  RowBox[{"(*", "Floats", "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5109116827914066`*^9, 3.510911693009424*^9}, {
  3.5109118265768595`*^9, 3.510911843393689*^9}, {3.5109119242330303`*^9, 
  3.5109119576326895`*^9}, {3.511129681916999*^9, 3.511129693044636*^9}, {
  3.511129762019581*^9, 3.5111297628596287`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Initial Values", "Subsection",
 CellChangeTimes->{{3.510560357878402*^9, 3.5105603606396074`*^9}, {
  3.510625979738017*^9, 3.5106259827956223`*^9}, {3.510911978505526*^9, 
  3.510911979644328*^9}}],

Cell[CellGroupData[{

Cell["Initial Position & Orientation of Each Body", "Subsubsection",
 CellChangeTimes->{{3.51091199199955*^9, 3.510912024151206*^9}, {
  3.5113222648381405`*^9, 3.5113222658053417`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"lw", "=", 
    RowBox[{"50", "cm"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"typical", " ", "length", " ", "of", " ", "wires"}], 
   "*)"}]}]], "Input",
 CellChangeTimes->{{3.5113219241803417`*^9, 3.511321944241977*^9}, {
  3.512776183176851*^9, 3.5127761875760584`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"initg", "=", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}]}],
    ";"}], 
  RowBox[{"(*", "Ground", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"initTM", "=", 
    RowBox[{"{", 
     RowBox[{"0", ",", 
      RowBox[{"-", "lw"}], ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}]}], 
   ";"}], 
  RowBox[{"(*", 
   RowBox[{"Test", " ", "Mass"}], "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.510542002925949*^9, 3.5105420494920306`*^9}, {
   3.5105436066915674`*^9, 3.510543608516771*^9}, {3.510543698622529*^9, 
   3.5105437015397344`*^9}, {3.51055031780616*^9, 3.510550319256963*^9}, {
   3.5106259640599895`*^9, 3.510625996414446*^9}, {3.510912034182024*^9, 
   3.510912037130429*^9}, {3.510912170073862*^9, 3.5109122070927277`*^9}, {
   3.5109999557906184`*^9, 3.510999980454262*^9}, {3.511075660164109*^9, 
   3.511075679258543*^9}, {3.5110757102869973`*^9, 3.511075712408601*^9}, {
   3.5110779207601013`*^9, 3.5110779389341335`*^9}, 3.511078289373149*^9, {
   3.5111296990169773`*^9, 3.511129702435173*^9}, {3.5111297677139063`*^9, 
   3.5111297678259125`*^9}, {3.5111524153668365`*^9, 3.511152441465682*^9}, {
   3.5111532486735*^9, 3.5111532518247056`*^9}, {3.5111537772238345`*^9, 
   3.5111537806558404`*^9}, {3.511219101894482*^9, 3.511219113048502*^9}, {
   3.5112290480209055`*^9, 3.5112290640733337`*^9}, {3.511231809771756*^9, 
   3.5112318099277563`*^9}, {3.511321800238124*^9, 3.511321902168703*^9}, {
   3.5113219541479945`*^9, 3.511322052240967*^9}, {3.512774222678604*^9, 
   3.512774234550225*^9}, {3.5127748105656366`*^9, 3.5127748144188433`*^9}, {
   3.5127761943932705`*^9, 3.512776198043677*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Default Values of Parameters & Initial Values of Variables and Floats\
\>", "Subsubsection",
 CellChangeTimes->{{3.5109123598637958`*^9, 3.5109123642786036`*^9}, {
  3.510912508001656*^9, 3.510912540184512*^9}, {3.5113222758205595`*^9, 
  3.511322278378964*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"defparams", "=", 
    RowBox[{"Join", "[", "initg", "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"initvars", "=", 
    RowBox[{"Join", "[", "initTM", "]"}]}], ";"}], 
  RowBox[{"(*", "Variables", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"initfloats", "=", 
    RowBox[{"Join", "[", "]"}]}], ";"}], 
  RowBox[{"(*", "Floats", "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.510912583599389*^9, 3.510912631117072*^9}, {
  3.5109188304827647`*^9, 3.5109188378615775`*^9}, {3.5109999852746706`*^9, 
  3.5109999912026806`*^9}, {3.5110757227670193`*^9, 3.511075724280222*^9}, {
  3.511077968542986*^9, 3.511077968683386*^9}, {3.5111297121007257`*^9, 
  3.5111297534580913`*^9}, {3.511149882561988*^9, 3.5111498881779976`*^9}, {
  3.5111532547107105`*^9, 3.5111532656775303`*^9}, {3.51115380350988*^9, 
  3.5111538083926888`*^9}, {3.511219133266137*^9, 3.511219137290944*^9}, {
  3.511322074003005*^9, 3.5113220940958405`*^9}, {3.5127742447214427`*^9, 
  3.512774245361044*^9}, {3.512776204751689*^9, 3.512776206701692*^9}, {
  3.5127767572250595`*^9, 3.512776759923864*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Substitution", "Subsubsection",
 CellChangeTimes->{{3.510913689415735*^9, 3.510913714516179*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"subdefparams", "=", 
    RowBox[{"makesub", "[", 
     RowBox[{"allparams", ",", "defparams"}], "]"}]}], ";"}], 
  RowBox[{"(*", "Parameters", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"subinitvars", "=", 
    RowBox[{"makesub", "[", 
     RowBox[{"allvars", ",", "initvars"}], "]"}]}], ";"}], 
  RowBox[{"(*", "Variables", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"subinitfloats", "=", 
    RowBox[{"makesub", "[", 
     RowBox[{"allvars", ",", "initvars"}], "]"}]}], ";"}], 
  RowBox[{"(*", "Floats", "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5109137313954086`*^9, 3.510913798990327*^9}, {
  3.5111524709497337`*^9, 3.511152475504942*^9}, {3.5111532804195557`*^9, 
  3.511153306814802*^9}, {3.5113221093838673`*^9, 3.5113221168250804`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Position of Bodies", "Subsection",
 CellChangeTimes->{{3.5111299640521364`*^9, 3.5111299739127007`*^9}, {
   3.511215303957964*^9, 3.5112153043635645`*^9}, 3.511219171330204*^9, {
   3.511322411587598*^9, 3.511322413272401*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"posg", "=", "varg"}], ";", 
   RowBox[{"posTM", "=", "varTM"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5110785212363563`*^9, 3.5110785817020626`*^9}, {
   3.511078660653801*^9, 3.5110786610750017`*^9}, {3.5110786937726593`*^9, 
   3.511078720760707*^9}, {3.511078819072079*^9, 3.5110788518009367`*^9}, 
   3.511129970563509*^9, {3.5111345040973763`*^9, 3.5111345247986126`*^9}, {
   3.511139023940115*^9, 3.5111390369505377`*^9}, {3.5111411593030653`*^9, 
   3.5111411653558764`*^9}, {3.5111478027663345`*^9, 
   3.5111478096615467`*^9}, {3.5111479049621143`*^9, 
   3.5111479063817167`*^9}, {3.5111506315505033`*^9, 3.511150631644103*^9}, {
   3.5111539126944723`*^9, 3.5111539127880726`*^9}, {3.5112153117735777`*^9, 
   3.51121536986808*^9}, {3.51121718335091*^9, 3.511217211243759*^9}, {
   3.5112191753238115`*^9, 3.5112192073038673`*^9}, {3.511219240344725*^9, 
   3.5112193137740545`*^9}, {3.5112199675092163`*^9, 3.511220015697701*^9}, {
   3.5112200468197556`*^9, 3.5112200990954475`*^9}, {3.511220139171918*^9, 
   3.5112201564879484`*^9}, {3.511223344655672*^9, 3.5112233853405437`*^9}, {
   3.5112234306586227`*^9, 3.511223499641944*^9}, 3.511223584864894*^9, {
   3.511223625081764*^9, 3.5112238683953915`*^9}, {3.511223913994272*^9, 
   3.511223934586308*^9}, {3.511224317972582*^9, 3.5112243645854635`*^9}, {
   3.511224438295593*^9, 3.5112245600694065`*^9}, {3.511224590973061*^9, 
   3.511224591238261*^9}, 3.511225501827461*^9, {3.5112257523951006`*^9, 
   3.5112257752959414`*^9}, {3.511225857430085*^9, 3.5112258621412935`*^9}, {
   3.5112259063985715`*^9, 3.511225914682186*^9}, {3.511228887559024*^9, 
   3.5112288890878263`*^9}, {3.511228928821096*^9, 3.511228934031505*^9}, {
   3.511229587438653*^9, 3.5112295923682613`*^9}, {3.5112299221528406`*^9, 
   3.5112299244304447`*^9}, {3.5112302659306445`*^9, 3.511230266133445*^9}, {
   3.511230339812374*^9, 3.5112303404519753`*^9}, {3.511322430869232*^9, 
   3.511322506700965*^9}, {3.511322597196724*^9, 3.51132271461813*^9}, {
   3.51132295932016*^9, 3.5113230738555613`*^9}, {3.5113231260064526`*^9, 
   3.5113231777829437`*^9}, {3.511323242959858*^9, 3.51132324969907*^9}, {
   3.5113233113035784`*^9, 3.5113233186355915`*^9}, 3.511323521420347*^9, {
   3.511324086125739*^9, 3.5113241183397956`*^9}, {3.5113875058597193`*^9, 
   3.5113876400667553`*^9}, {3.5113876748080163`*^9, 
   3.5113877253989053`*^9}, {3.5113891555846176`*^9, 3.511389165412635*^9}, {
   3.5114067382148457`*^9, 3.5114067398216486`*^9}, {3.5114068871327076`*^9, 
   3.511406911905551*^9}, 3.512726595160699*^9, {3.5127742616630726`*^9, 
   3.512774262583474*^9}, {3.5127762312873354`*^9, 3.512776232987738*^9}}]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Energy Calculation", "Section",
 CellChangeTimes->{{3.5106262518492947`*^9, 3.5106262674337225`*^9}, {
  3.5109170152635765`*^9, 3.5109170473996325`*^9}}],

Cell[CellGroupData[{

Cell["Body Definition", "Subsection",
 CellChangeTimes->{{3.5109140124455023`*^9, 3.51091404542396*^9}, {
  3.5109169162814026`*^9, 3.5109169184810066`*^9}, {3.5109662739308863`*^9, 
  3.51096627628649*^9}}],

Cell[CellGroupData[{

Cell["About bodylist", "Subsubsection",
 CellChangeTimes->{{3.5127269557397323`*^9, 3.51272696006094*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "bodylist"}]], "Input",
 CellChangeTimes->{{3.5127269619485435`*^9, 3.5127269651465487`*^9}}],

Cell[BoxData[
 StyleBox["\<\"bodylist is a list of structures defining bodies. Each \
structure containing the following items:\\n\tcoordinate list defining the \
body,\\n\tmass,\\n\tmoment of inertia tensor,\\n\tshape,\\n\tdimension\"\>", 
  "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.5181496107606726`*^9},
 CellTags->"Info3518182010-7910842"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Standard Bodies", "Subsubsection",
 CellChangeTimes->{{3.5113252072217083`*^9, 3.511325209982913*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"bodylist", "=", 
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"(*", 
       RowBox[{"Test", " ", "Mass"}], "*)"}], "\[IndentingNewLine]", 
      RowBox[{"posTM", ",", "mTM", ",", "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"ixTM", ",", "0", ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "iyTM", ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "0", ",", "izTM"}], "}"}]}], "}"}], ",", 
       RowBox[{"(*", 
        RowBox[{"moment", " ", "of", " ", "inertia"}], "*)"}], 
       "\[IndentingNewLine]", "shapeTM", ",", 
       RowBox[{"(*", 
        RowBox[{"type", " ", "of", " ", "shape"}], "*)"}], "dimTM"}], 
      RowBox[{"(*", 
       RowBox[{"size", " ", "information"}], "*)"}], "\[IndentingNewLine]", 
      "}"}], "\[IndentingNewLine]", "}"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5109140609147873`*^9, 3.5109140779968176`*^9}, {
   3.5109141081360703`*^9, 3.5109142185842648`*^9}, {3.5109143213572445`*^9, 
   3.5109145997085333`*^9}, 3.510917211730322*^9, {3.5109176320262594`*^9, 
   3.5109176547555*^9}, 3.510917743488456*^9, {3.5109662907321157`*^9, 
   3.510966396843502*^9}, {3.510967091044721*^9, 3.510967168249257*^9}, {
   3.5109955112540817`*^9, 3.5109955697864294`*^9}, {3.510997044967098*^9, 
   3.5109970462151003`*^9}, {3.5110002784771852`*^9, 
   3.5110003227344627`*^9}, {3.511000359144927*^9, 3.5110003704549465`*^9}, {
   3.511078476823078*^9, 3.511078512422341*^9}, 3.5111541017668047`*^9, {
   3.5112152637410936`*^9, 3.511215297234352*^9}, {3.511217240369011*^9, 
   3.511217254159435*^9}, {3.5112246532795706`*^9, 3.5112248007154293`*^9}, {
   3.51122485224232*^9, 3.5112249190416374`*^9}, {3.5112249915817647`*^9, 
   3.511225023593021*^9}, {3.5112253387915745`*^9, 3.5112253422859807`*^9}, {
   3.5112260225719757`*^9, 3.5112260227123756`*^9}, {3.511226094581702*^9, 
   3.5112260955489035`*^9}, {3.5113242138899636`*^9, 3.511324447937175*^9}, 
   3.511324596262235*^9, {3.511324730375671*^9, 3.5113247350088787`*^9}, 
   3.5113247888913736`*^9, {3.5127093008876896`*^9, 3.5127093655186033`*^9}, {
   3.5127094492907505`*^9, 3.5127095235312805`*^9}, {3.512710355012741*^9, 
   3.512710367024762*^9}, {3.512776269320202*^9, 3.512776321829894*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Others", "Subsubsection",
 CellChangeTimes->{{3.5113252238201375`*^9, 3.5113252269869432`*^9}}],

Cell[CellGroupData[{

Cell["Ground", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511325241245368*^9, 3.511325249388582*^9}, 
   3.5113253530819645`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{"showground", ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"AppendTo", "[", 
       RowBox[{"bodylist", ",", 
        RowBox[{"{", "\[IndentingNewLine]", 
         RowBox[{"posg", ",", 
          RowBox[{"(*", "coordinate", "*)"}], "0", ",", 
          RowBox[{"(*", "mass", "*)"}], "\[IndentingNewLine]", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "}"}], ",", 
          RowBox[{"(*", 
           RowBox[{"moment", " ", "of", " ", "inertia"}], "*)"}], 
          "\[IndentingNewLine]", "shapeg", ",", 
          RowBox[{"(*", 
           RowBox[{"type", " ", "of", " ", "shape"}], "*)"}], "dimg"}], 
         RowBox[{"(*", 
          RowBox[{"size", " ", "information"}], "*)"}], "}"}]}], "]"}], 
      ";"}]}], "\[IndentingNewLine]", "]"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113252534913893`*^9, 3.5113253530819645`*^9}, {
  3.512709563763751*^9, 3.5127095645437527`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Wire Definition", "Subsection",
 CellChangeTimes->{{3.5109140124455023`*^9, 3.51091404542396*^9}, {
  3.5109169162814026`*^9, 3.5109169184810066`*^9}}],

Cell[CellGroupData[{

Cell["About wirelist", "Subsubsection",
 CellChangeTimes->{{3.512726995441802*^9, 3.51272700010621*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "wirelist"}]], "Input",
 CellChangeTimes->{{3.5127270107922287`*^9, 3.512727011993431*^9}}],

Cell[BoxData[
 StyleBox["\<\"wirelist is a list of structures defining wires. Each \
structure containing the following items:\\n\tcoordinate list defining first \
mass,\\n\tattachment point for first mass (local coordinates),\\n\tcoordinate \
list defining second mass,\\n\tattachment point for second mass (local \
coordinates),\\n\tunstretched length,\\n\tlongitudinal elasticity,\\n\tloss \
angle of the material,\\n\tshear modulus of the material,\\n\tsecond moment \
of area\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.518149610932273*^9},
 CellTags->"Info3518182010-5560469"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making wirelist", "Subsubsection",
 CellChangeTimes->{{3.5127270034446163`*^9, 3.51272700581582*^9}}],

Cell[CellGroupData[{

Cell["Coordinate list of wire-attaching bodies", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5113271026756425`*^9, 3.5113271134708614`*^9}, {
   3.511327177774174*^9, 3.5113271910185976`*^9}, 3.511327439480234*^9, 
   3.51271000708533*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"posub", "=", 
   RowBox[{"{", "posg", "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"poslb", "=", 
   RowBox[{"{", "posTM", "}"}]}], ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511327117714069*^9, 3.5113272142158384`*^9}, 
   3.511327439480234*^9, {3.512710003122923*^9, 3.51271000708533*^9}, {
   3.5127743561992383`*^9, 3.512774359724845*^9}, {3.5127746732229953`*^9, 
   3.512774673488196*^9}, {3.5127766357492456`*^9, 3.5127766429096584`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making wirelist", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113264636323147`*^9, 3.511326468936324*^9}, 
   3.511326909453703*^9, {3.5113272309858675`*^9, 3.51132723226507*^9}, {
   3.5113274460322456`*^9, 3.5113274471086473`*^9}, {3.5113277295939436`*^9, 
   3.5113277327919493`*^9}, {3.5127270146298356`*^9, 3.512727014754636*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"wirelist", "=", "\[IndentingNewLine]", 
    RowBox[{"Flatten", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"Table", "[", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"posub", "[", 
           RowBox[{"[", "m", "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "coordinate", " ", "list", " ", "defining", " ", "upper", " ", 
            "body"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{"spu", "[", 
           RowBox[{"[", 
            RowBox[{"m", ",", "n"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "attachment", " ", "point", " ", "at", " ", "upper", " ", "body"}],
            "*)"}], "\[IndentingNewLine]", 
          RowBox[{"poslb", "[", 
           RowBox[{"[", "m", "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "coordinate", " ", "list", " ", "defining", " ", "lower", " ", 
            "body"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{"spl", "[", 
           RowBox[{"[", 
            RowBox[{"m", ",", "n"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{
           "attachment", " ", "point", " ", "at", " ", "lower", " ", "body"}],
            "*)"}], "\[IndentingNewLine]", 
          RowBox[{"lNw", "[", 
           RowBox[{"[", 
            RowBox[{"m", ",", "n"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{"Natural", " ", "Length"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"N", "[", "Pi", "]"}], 
           SuperscriptBox[
            RowBox[{"dw", "[", 
             RowBox[{"[", 
              RowBox[{"m", ",", "n"}], "]"}], "]"}], "2"], 
           RowBox[{
            RowBox[{
             RowBox[{"young", "[", 
              RowBox[{"matw", "[", 
               RowBox[{"[", "m", "]"}], "]"}], "]"}], "/", 
             RowBox[{"lNw", "[", 
              RowBox[{"[", 
               RowBox[{"m", ",", "n"}], "]"}], "]"}]}], "/", "4"}]}], ",", 
          RowBox[{"(*", "Stiffness", "*)"}], "\[IndentingNewLine]", 
          RowBox[{"phi", "[", 
           RowBox[{"matw", "[", 
            RowBox[{"[", "m", "]"}], "]"}], "]"}], ",", 
          RowBox[{"(*", 
           RowBox[{"loss", " ", "angle"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{
            RowBox[{"young", "[", 
             RowBox[{"matw", "[", 
              RowBox[{"[", "m", "]"}], "]"}], "]"}], "/", "2"}], "/", 
           RowBox[{"(", 
            RowBox[{"1", "+", 
             RowBox[{"poisson", "[", 
              RowBox[{"matw", "[", 
               RowBox[{"[", "m", "]"}], "]"}], "]"}]}], ")"}]}], ",", 
          RowBox[{"(*", 
           RowBox[{"shear", " ", "modulus"}], "*)"}], "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"N", "[", "Pi", "]"}], 
           RowBox[{
            SuperscriptBox[
             RowBox[{"dw", "[", 
              RowBox[{"[", 
               RowBox[{"m", ",", "n"}], "]"}], "]"}], "4"], "/", "32"}]}]}], 
         "}"}], ",", 
        RowBox[{"(*", 
         RowBox[{"second", " ", "moment", " ", "of", " ", "area"}], "*)"}], 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "nw", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"nw", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "}"}]}], "\[IndentingNewLine]", 
       "]"}], ",", "1"}], "\[IndentingNewLine]", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109140609147873`*^9, 3.5109140779968176`*^9}, {
   3.5109141081360703`*^9, 3.5109142185842648`*^9}, {3.5109143213572445`*^9, 
   3.5109145997085333`*^9}, 3.510917211730322*^9, {3.5109176320262594`*^9, 
   3.5109176547555*^9}, 3.510917743488456*^9, {3.511000024399539*^9, 
   3.5110000885468516`*^9}, {3.5110001243021145`*^9, 3.51100016160178*^9}, {
   3.5110001988390455`*^9, 3.5110002427687225`*^9}, {3.511001607649932*^9, 
   3.511001780966236*^9}, {3.5111538584687767`*^9, 3.511153869965997*^9}, {
   3.511155069015303*^9, 3.511155069842105*^9}, {3.5112246193027105`*^9, 
   3.511224643903954*^9}, {3.5112258457456646`*^9, 3.5112258461824656`*^9}, {
   3.5112259909039197`*^9, 3.511226005833146*^9}, {3.5113264112630234`*^9, 
   3.5113264144922285`*^9}, {3.5113264759095364`*^9, 
   3.5113266364338183`*^9}, {3.5113267140789595`*^9, 3.511326734109395*^9}, {
   3.511326898596084*^9, 3.511326924086529*^9}, {3.511327235587876*^9, 
   3.5113272773335495`*^9}, {3.511327366737306*^9, 3.511327419044198*^9}, {
   3.5113275248903837`*^9, 3.5113275334391985`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Spring Definition", "Subsection",
 CellChangeTimes->{{3.5109140124455023`*^9, 3.51091404542396*^9}, {
  3.5109169162814026`*^9, 3.5109169184810066`*^9}, {3.5109662739308863`*^9, 
  3.51096627628649*^9}, {3.5110788859025965`*^9, 3.5110788867761984`*^9}}],

Cell[CellGroupData[{

Cell["About springlist", "Subsubsection",
 CellChangeTimes->{{3.5127270198558445`*^9, 3.512727025378255*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "springlist"}]], "Input",
 CellChangeTimes->{{3.5127270268134575`*^9, 3.5127270288258605`*^9}}],

Cell[BoxData[
 StyleBox["\<\"springlist is a list of structures defining GAS springs. Each \
structure containing the following items:\\n\tcoordinate list defining first \
mass,\\n\tworking point at first mass (local coordinate)\\n\tcoordinate list \
defining second mass,\\n\tattached point at second mass (local coordinate)\\n\
\tspring constant tensor (6x6),\\n\tstoring force element vector (6x1),\\n\t\
spring constant tensor with imaginary part (6x6)\"\>", "MSG"]], "Print", \
"PrintUsage",
 CellChangeTimes->{3.518149611088273*^9},
 CellTags->"Info3518182010-4499236"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making springlist", "Subsubsection",
 CellChangeTimes->{{3.512727031540265*^9, 3.51272703420787*^9}}],

Cell[CellGroupData[{

Cell["Making springlist", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511327747565175*^9, 3.5113277509503813`*^9}, 
   3.5113280529669113`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"springlist", "=", 
   RowBox[{"{", "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"no", " ", "spring"}], "*)"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5109140609147873`*^9, 3.5109140779968176`*^9}, {
   3.5109141081360703`*^9, 3.5109142185842648`*^9}, {3.5109143213572445`*^9, 
   3.5109145997085333`*^9}, 3.510917211730322*^9, {3.5109176320262594`*^9, 
   3.5109176547555*^9}, 3.510917743488456*^9, {3.5109662907321157`*^9, 
   3.510966396843502*^9}, {3.510967091044721*^9, 3.510967168249257*^9}, {
   3.5109955112540817`*^9, 3.5109955697864294`*^9}, {3.510997044967098*^9, 
   3.5109970462151003`*^9}, {3.5110002784771852`*^9, 
   3.5110003227344627`*^9}, {3.511000359144927*^9, 3.5110003704549465`*^9}, {
   3.511078476823078*^9, 3.511078512422341*^9}, {3.511078893874211*^9, 
   3.5110789163538504`*^9}, {3.5111301717180142`*^9, 
   3.5111302177266455`*^9}, {3.511130264499321*^9, 3.5111303154552355`*^9}, {
   3.5111303524563518`*^9, 3.511130387641364*^9}, {3.5111305329186735`*^9, 
   3.5111305565030227`*^9}, {3.5111306659992857`*^9, 
   3.5111307500230913`*^9}, {3.5111310998197966`*^9, 3.51113113022425*^9}, 
   3.511132179263693*^9, {3.511134281157385*^9, 3.511134313043841*^9}, {
   3.511134822821536*^9, 3.511134849169983*^9}, 3.51113914042552*^9, 
   3.5111393361746635`*^9, {3.51113936804552*^9, 3.5111393805567417`*^9}, {
   3.5111525121650066`*^9, 3.5111525123522067`*^9}, {3.511154131282056*^9, 
   3.51115413343486*^9}, {3.5112291050234056`*^9, 3.511229108252611*^9}, {
   3.511327629769368*^9, 3.5113276327957735`*^9}, {3.5113277552715883`*^9, 
   3.511327803865674*^9}, {3.51132798381199*^9, 3.511327991175203*^9}, {
   3.511328024902462*^9, 3.5113280529669113`*^9}, {3.512726923057675*^9, 
   3.5127269360212975`*^9}, {3.5127764501401196`*^9, 3.51277645023372*^9}, {
   3.518149288590987*^9, 3.5181492924072056`*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Damper Definition", "Subsection",
 CellChangeTimes->{{3.511388575091998*^9, 3.51138859902244*^9}}],

Cell[CellGroupData[{

Cell["About damperlist", "Subsubsection",
 CellChangeTimes->{{3.512727052319502*^9, 3.5127270548623066`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"?", "damperlist"}]], "Input",
 CellChangeTimes->{{3.5127270626779203`*^9, 3.5127270641287227`*^9}}],

Cell[BoxData[
 StyleBox["\<\"damperlist is is a list of structures defining dampers. Each \
structure containing the following items:\\n\tcoordinate list defining first \
mass,\\n\tcoordinate list defining second mass,\\n\tdamping coefficient \
matrix (6x6)\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.518149611291074*^9},
 CellTags->"Info3518182011-4076727"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making damperlist", "Subsubsection",
 CellChangeTimes->{{3.5127270573115106`*^9, 3.5127270610087175`*^9}}],

Cell[CellGroupData[{

Cell["Making damperlist", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113885915032268`*^9, 3.5113885957776346`*^9}, 
   3.511388709876235*^9, {3.5127270675607285`*^9, 3.5127270704467335`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"damperlist", "=", 
   RowBox[{"{", "}"}]}], ";", 
  RowBox[{"(*", 
   RowBox[{"no", " ", "damper"}], "*)"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511388612048463*^9, 3.511388709876235*^9}, {
  3.511389274160426*^9, 3.511389300805273*^9}, {3.5127104680973396`*^9, 
  3.5127104738693495`*^9}, {3.5181492958043995`*^9, 3.518149298974581*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Potential Energy Calculation", "Subsection",
 CellChangeTimes->{{3.510560448249361*^9, 3.510560448670562*^9}, {
  3.5106264014535575`*^9, 3.510626411702776*^9}, {3.5109166230008874`*^9, 
  3.510916633390506*^9}, {3.510917225723546*^9, 3.510917232306758*^9}}],

Cell[CellGroupData[{

Cell["Wire Stretching Potential", "Subsubsection",
 CellChangeTimes->{{3.511219454212905*^9, 3.511219459080114*^9}, {
  3.5113282965753393`*^9, 3.511328305264555*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"wirepot", "=", 
    RowBox[{"makewirepot", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"only", " ", "real", " ", "part"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"wirepotc", "=", 
    RowBox[{"makewirepotc", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"with", " ", "imaginary", " ", "part"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.510542091144104*^9, 3.510542093936509*^9}, {
   3.510542888481906*^9, 3.5105428899327087`*^9}, {3.510542932146383*^9, 
   3.5105429338311863`*^9}, {3.5105434500672927`*^9, 3.510543476493739*^9}, {
   3.510543521063017*^9, 3.5105435385974483`*^9}, {3.5105435839311275`*^9, 
   3.5105435894847374`*^9}, {3.510545463687629*^9, 3.5105454866664696`*^9}, {
   3.5105498942966166`*^9, 3.510550005088011*^9}, {3.5105505425245547`*^9, 
   3.5105505701990037`*^9}, {3.5105506566075554`*^9, 3.510550672535183*^9}, {
   3.5105509738653126`*^9, 3.5105509743645134`*^9}, {3.510551048137043*^9, 
   3.5105511440304117`*^9}, 3.510557050919793*^9, 3.5105572108200736`*^9, 
   3.5105577896122904`*^9, 3.510557847488392*^9, {3.5105583630068974`*^9, 
   3.510558405704172*^9}, {3.5105608134148026`*^9, 3.510560853085672*^9}, 
   3.5105608836617255`*^9, {3.5106264247287984`*^9, 3.5106264690796766`*^9}, {
   3.5109166369473124`*^9, 3.510916652516139*^9}, {3.510916956872674*^9, 
   3.510916976934309*^9}, {3.5109172133839245`*^9, 3.510917261400809*^9}, {
   3.51091783730702*^9, 3.5109178806750965`*^9}, 3.511002137801263*^9, {
   3.5111411385862293`*^9, 3.5111411749810934`*^9}, 3.511141210159155*^9, {
   3.5111541843533497`*^9, 3.5111541968957715`*^9}, {3.5112194905765696`*^9, 
   3.5112195044761934`*^9}, {3.5113282322408266`*^9, 3.511328247341653*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Wire Torsional Potential", "Subsubsection",
 CellChangeTimes->{{3.511328281521313*^9, 3.511328293689335*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"torsionpot", "=", 
    RowBox[{"maketorsionpot", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"only", " ", "real", " ", "part"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"torsionpotc", "=", 
    RowBox[{"maketorsionpotc", "[", "wirelist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"with", " ", "imaginary", " ", "part"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511002097319192*^9, 3.511002109783613*^9}, 
   3.5110021423564706`*^9, {3.5110022020421753`*^9, 3.5110022106377907`*^9}, {
   3.5110034113562994`*^9, 3.5110034150379057`*^9}, {3.511233156981127*^9, 
   3.5112331628935375`*^9}, {3.511328253441264*^9, 3.511328271802496*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Spring Potential", "Subsubsection",
 CellChangeTimes->{{3.5112195186878185`*^9, 3.511219521293023*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"springpot", "=", 
    RowBox[{"makespringpot", "[", "springlist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"only", " ", "real", " ", "part"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"springpotc", "=", 
    RowBox[{"makespringpotc", "[", "springlist", "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"with", " ", "imaginary", " ", "part"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.510542091144104*^9, 3.510542093936509*^9}, {
   3.510542888481906*^9, 3.5105428899327087`*^9}, {3.510542932146383*^9, 
   3.5105429338311863`*^9}, {3.5105434500672927`*^9, 3.510543476493739*^9}, {
   3.510543521063017*^9, 3.5105435385974483`*^9}, {3.5105435839311275`*^9, 
   3.5105435894847374`*^9}, {3.510545463687629*^9, 3.5105454866664696`*^9}, {
   3.5105498942966166`*^9, 3.510550005088011*^9}, {3.5105505425245547`*^9, 
   3.5105505701990037`*^9}, {3.5105506566075554`*^9, 3.510550672535183*^9}, {
   3.5105509738653126`*^9, 3.5105509743645134`*^9}, {3.510551048137043*^9, 
   3.5105511440304117`*^9}, 3.510557050919793*^9, 3.5105572108200736`*^9, 
   3.5105577896122904`*^9, 3.510557847488392*^9, {3.5105583630068974`*^9, 
   3.510558405704172*^9}, {3.5105608134148026`*^9, 3.510560853085672*^9}, 
   3.5105608836617255`*^9, {3.5106264247287984`*^9, 3.5106264690796766`*^9}, {
   3.5109166369473124`*^9, 3.510916652516139*^9}, {3.510916956872674*^9, 
   3.510916976934309*^9}, {3.5109172133839245`*^9, 3.510917261400809*^9}, {
   3.51091783730702*^9, 3.5109178806750965`*^9}, 3.511002137801263*^9, {
   3.5111301323777637`*^9, 3.511130142680353*^9}, {3.5111389035391035`*^9, 
   3.511138929201149*^9}, {3.511139340261871*^9, 3.511139364129913*^9}, 
   3.511139422817216*^9, {3.51114012523965*^9, 3.511140134584066*^9}, {
   3.5111409838339577`*^9, 3.5111409926791735`*^9}, {3.5111542019033804`*^9, 
   3.5111542101869946`*^9}, {3.5113283226741858`*^9, 3.5113283456218257`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Gravity Potential", "Subsubsection",
 CellChangeTimes->{{3.511219530216239*^9, 3.511219534256646*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"gravpot", "=", 
    RowBox[{"makegravpot", "[", "bodylist", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5109172647548146`*^9, 3.5109172699028234`*^9}, {
   3.510966585884634*^9, 3.5109665906738424`*^9}, {3.5109669988485594`*^9, 
   3.5109670018593645`*^9}, 3.5110021586428995`*^9, {3.511227259181363*^9, 
   3.5112272796485996`*^9}, 3.5113283557930436`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Total", "Subsubsection",
 CellChangeTimes->{{3.511219549981474*^9, 3.511219550652275*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"potential", "=", 
   RowBox[{"wirepot", "+", "gravpot", "+", "torsionpot", "+", "springpot"}]}],
   ";"}], "\n", 
 RowBox[{
  RowBox[{"potentialc", "=", 
   RowBox[{
   "wirepotc", "+", "gravpot", "+", "torsionpot", "+", "springpotc"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109172740680313`*^9, 3.5109172945196667`*^9}, 
   3.5109179087707458`*^9, 3.5110004500306864`*^9, {3.5110004867843513`*^9, 
   3.5110005204336104`*^9}, 3.5110021501720843`*^9, {3.5110022152553988`*^9, 
   3.5110022192646055`*^9}, {3.5111433491457124`*^9, 3.511143363653738*^9}, {
   3.511152543224661*^9, 3.511152549573872*^9}, {3.511226445838719*^9, 
   3.511226472530366*^9}, {3.5113289728677273`*^9, 3.511328987375753*^9}, {
   3.5127744110957346`*^9, 3.5127744184121475`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Kinetic Energy Calculation", "Subsection",
 CellChangeTimes->{{3.51091730216368*^9, 3.510917310494095*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"kinetic", "=", 
    RowBox[{"makekinetic", "[", "bodylist", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.510917352099368*^9, 3.510917508349242*^9}, {
   3.5109668339562693`*^9, 3.51096683429947*^9}, {3.5109669425480604`*^9, 
   3.5109669488504715`*^9}, 3.510967004105768*^9, 3.510967051093051*^9, {
   3.511000390828583*^9, 3.511000401436601*^9}, {3.51114029066234*^9, 
   3.511140299008355*^9}, {3.5111504745986276`*^9, 3.5111504779058332`*^9}, {
   3.511227322049474*^9, 3.511227329007086*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Damping Energy Calculation", "Subsection",
 CellChangeTimes->{{3.5113890192871785`*^9, 3.5113890229843845`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"damping", "=", 
    RowBox[{"makedamping", "[", "damperlist", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5113890305347977`*^9, 3.511389047632428*^9}, {
  3.511389361348979*^9, 3.5113893730333996`*^9}}]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Find Equilibrium Point", "Section",
 CellChangeTimes->{{3.5106265435854073`*^9, 3.5106265504962196`*^9}, 
   3.5106554796021967`*^9}],

Cell[CellGroupData[{

Cell["Potential with Default Values of Parameters", "Subsection",
 CellChangeTimes->{{3.5106268136062813`*^9, 3.5106268456331377`*^9}, {
  3.510627087792363*^9, 3.5106271045155926`*^9}, {3.510917973136459*^9, 
  3.510917992106092*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"potential0", "=", 
    RowBox[{"potential", "/.", "subdefparams"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5105539888460145`*^9, 3.510554070636958*^9}, {
   3.510554121430648*^9, 3.5105542109124045`*^9}, 3.5105546485935736`*^9, {
   3.5105546974216595`*^9, 3.510554704051671*^9}, {3.5105571731928077`*^9, 
   3.5105571828960247`*^9}, 3.51055786907883*^9, 3.510557938810952*^9, {
   3.5105584149237885`*^9, 3.5105584273414106`*^9}, {3.510559192210754*^9, 
   3.5105592145655932`*^9}, {3.510560255901023*^9, 3.5105602574922256`*^9}, {
   3.5105606405976987`*^9, 3.51056068084577*^9}, 3.5105607235898447`*^9, 
   3.5105611168353353`*^9, {3.5106265647702446`*^9, 3.5106265760490646`*^9}, {
   3.510627120958021*^9, 3.51062714858567*^9}, {3.5106273594512405`*^9, 
   3.5106273673448544`*^9}, {3.5106274131777344`*^9, 3.510627420806148*^9}, {
   3.510636250108863*^9, 3.5106362563176737`*^9}, {3.5106557863611355`*^9, 
   3.51065579467595*^9}, {3.5108253325205708`*^9, 3.5108253359993763`*^9}, {
   3.5109179503760185`*^9, 3.5109179592680345`*^9}, {3.5109180139149303`*^9, 
   3.510918040434977*^9}, 3.5109183532935266`*^9, 3.510918580757526*^9, 
   3.510918631816416*^9, 3.51091887893645*^9, {3.511150554049567*^9, 
   3.5111505637215843`*^9}, {3.511150600678049*^9, 3.5111506134388714`*^9}, 
   3.511154234382637*^9, 3.5111613806292715`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Find Equilibrium Point", "Subsection",
 CellChangeTimes->{{3.5106274478565955`*^9, 3.510627536183951*^9}, 
   3.510655478166994*^9}],

Cell["Find optimal values that make the potential energy minimum.", "Text",
 CellChangeTimes->{{3.510627542611162*^9, 3.5106275966652565`*^9}, {
   3.5109180810106487`*^9, 3.510918084910655*^9}, {3.5109182507701464`*^9, 
   3.510918255060154*^9}, 3.510975865186532*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"subminvars", "=", 
  RowBox[{
   RowBox[{"FindMinimum", "[", "\[IndentingNewLine]", 
    RowBox[{"potential0", ",", "\[IndentingNewLine]", 
     RowBox[{"Join", "[", 
      RowBox[{
       RowBox[{"comb", "[", 
        RowBox[{"allvars", ",", "initvars"}], "]"}], ",", 
       RowBox[{"comb", "[", 
        RowBox[{"allfloats", ",", "initfloats"}], "]"}]}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"Method", "\[Rule]", "\"\<Newton\>\""}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"MaxIterations", "\[Rule]", "100"}]}], "\[IndentingNewLine]", 
    "]"}], "[", 
   RowBox[{"[", "2", "]"}], "]"}]}]], "Input",
 CellChangeTimes->{{3.51055781213873*^9, 3.510557830281562*^9}, {
   3.510557969761407*^9, 3.5105579803850255`*^9}, {3.510558511285158*^9, 
   3.510558539942408*^9}, {3.5105590865985684`*^9, 3.5105591272054396`*^9}, {
   3.510559279305707*^9, 3.510559289632925*^9}, 3.5105605175134826`*^9, 
   3.51056060141043*^9, {3.510560695213395*^9, 3.5105606957281957`*^9}, 
   3.5105609712558794`*^9, 3.510561120158141*^9, {3.510627334818797*^9, 
   3.510627350855625*^9}, {3.5106274406493826`*^9, 3.510627443504188*^9}, 
   3.510627615010889*^9, {3.5106295313562555`*^9, 3.5106295448814793`*^9}, 
   3.5106361885199547`*^9, 3.51063623703604*^9, {3.510636431164381*^9, 
   3.5106364370455914`*^9}, {3.510655802725564*^9, 3.5106558188091927`*^9}, 
   3.510655856608059*^9, {3.510655900756136*^9, 3.510655922518174*^9}, 
   3.510918005334915*^9, {3.510918228306107*^9, 3.510918233828517*^9}, {
   3.5109182690221786`*^9, 3.5109182811278*^9}, {3.511004376030794*^9, 
   3.5110044189620695`*^9}, {3.5110044623613453`*^9, 3.511004473421765*^9}, {
   3.5111403358556194`*^9, 3.5111403660104723`*^9}, {3.5111412252287817`*^9, 
   3.511141227818386*^9}, 3.5111413265977592`*^9, {3.5111413751606445`*^9, 
   3.5111414888988447`*^9}, {3.5111549058546166`*^9, 3.5111549297538586`*^9}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"xTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yTM", "\[Rule]", 
    RowBox[{"-", "0.5020116729045931`"}]}], ",", 
   RowBox[{"zTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"pitchTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"yawTM", "\[Rule]", "0.`"}], ",", 
   RowBox[{"rollTM", "\[Rule]", "0.`"}]}], "}"}]], "Output",
 CellChangeTimes->{
  3.510918281798601*^9, {3.5109185842987328`*^9, 3.510918588183139*^9}, 
   3.5109188021843147`*^9, 3.5109188818380547`*^9, 3.510956282795619*^9, 
   3.510958553471608*^9, 3.5109760020455723`*^9, 3.5109762224427595`*^9, 
   3.510981584761199*^9, 3.510997021052256*^9, 3.5109970696775417`*^9, 
   3.51100053174363*^9, 3.5110022334138308`*^9, 3.5110032212855654`*^9, 
   3.511003477625216*^9, 3.5110044026288404`*^9, {3.5110044639837484`*^9, 
   3.511004474498167*^9}, 3.511004507944626*^9, 3.5110674021996913`*^9, 
   3.5111403773048925`*^9, 3.511141010588004*^9, 3.5111412287075877`*^9, 
   3.5111413274869614`*^9, {3.5111413840838604`*^9, 3.5111414391659575`*^9}, {
   3.511141482034833*^9, 3.5111414896320467`*^9}, 3.511143384729375*^9, 
   3.5111469195239835`*^9, 3.5111472172193065`*^9, 3.511147933229364*^9, {
   3.511149501266118*^9, 3.5111495096589327`*^9}, 3.511150531741528*^9, 
   3.511150641955722*^9, {3.511150687227001*^9, 3.5111507000658236`*^9}, 
   3.5111525605562916`*^9, 3.511152660521267*^9, 3.5111541628097115`*^9, 
   3.5111546461141605`*^9, 3.511154858804934*^9, 3.511154922016245*^9, 
   3.5111550870333347`*^9, 3.511227338569903*^9, 3.5112285791620817`*^9, 
   3.5112287411059666`*^9, 3.5112289115986657`*^9, 3.511228943719122*^9, 
   3.5112291311846514`*^9, 3.5112291859407473`*^9, 3.511229342253022*^9, 
   3.511229667950394*^9, 3.51122993873567*^9, 3.51123036609842*^9, 
   3.511230514938282*^9, 3.5112317294784155`*^9, {3.511231796870534*^9, 
   3.5112318180085707`*^9}, 3.511232314782448*^9, 3.5112324235614386`*^9, 
   3.511232456883097*^9, 3.511232491608758*^9, 3.5112325360376368`*^9, 
   3.511232579889313*^9, 3.511232655409046*^9, 3.511232843857377*^9, 
   3.5112329061326866`*^9, 3.5112332275400505`*^9, 3.5113291732188797`*^9, 
   3.511329411712098*^9, 3.511330414918663*^9, 3.511330611135808*^9, 
   3.5113309898420725`*^9, 3.511331145155946*^9, 3.511332932014284*^9, 
   3.511343444537177*^9, 3.5113893904430304`*^9, 3.5114066658619184`*^9, 
   3.5114067613964863`*^9, 3.5114069281139793`*^9, 3.5114315309789104`*^9, 
   3.5114900390388813`*^9, 3.511494313353214*^9, 3.5114947992160673`*^9, 
   3.5114950951797867`*^9, 3.5114976128240194`*^9, 3.5123557673055196`*^9, 
   3.5123559844585347`*^9, 3.512357720304784*^9, 3.5123578229841638`*^9, 
   3.5123586976933002`*^9, 3.512358797330675*^9, 3.5123589226144953`*^9, 
   3.5124237581731677`*^9, 3.5124275277636547`*^9, 3.5127107827030926`*^9, 
   3.5127123057457695`*^9, 3.5127124974389057`*^9, 3.5127125912262707`*^9, 
   3.5127130589306917`*^9, 3.5127131447932425`*^9, 3.51273061964657*^9, 
   3.5127326937654157`*^9, 3.512733364067393*^9, 3.5127350758739996`*^9, 
   3.512774776027176*^9, 3.5127748237320595`*^9, 3.512776779689099*^9, 
   3.518149611603074*^9}]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Equations of Motion", "Section",
 CellChangeTimes->{{3.510627640017733*^9, 3.510627674649794*^9}, 
   3.5109189734102154`*^9}],

Cell[CellGroupData[{

Cell["Mass Matrix", "Subsection",
 CellChangeTimes->{{3.510561334346517*^9, 3.510561337388523*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matMvv", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"kinetic", ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellChangeTimes->{{3.510561339822127*^9, 3.5105613828470025`*^9}, {
   3.5105614843563805`*^9, 3.51056150657082*^9}, {3.510561663541896*^9, 
   3.5105617389524283`*^9}, {3.5105619183683434`*^9, 
   3.5105619553404083`*^9}, {3.5106285477833276`*^9, 
   3.5106286902115774`*^9}, {3.510629347182332*^9, 3.510629358086751*^9}, {
   3.5106295553334975`*^9, 3.5106295558638983`*^9}, {3.5106573586878967`*^9, 
   3.5106574988697433`*^9}, 3.5106575498350325`*^9, {3.5106575855122952`*^9, 
   3.510657586838298*^9}, 3.510657701592099*^9, {3.510918992301849*^9, 
   3.510919033813522*^9}, {3.510919072236389*^9, 3.5109190853248124`*^9}, {
   3.511002629373726*^9, 3.511002636112938*^9}, {3.511142697495368*^9, 
   3.511142697651368*^9}, {3.511148104096464*^9, 3.5111481386037245`*^9}, {
   3.5112308639264946`*^9, 3.5112308672337008`*^9}, 3.5127273081131516`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matMpv", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"kinetic", ",", 
           RowBox[{"velvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"velparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "velparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.510629277247409*^9, 3.5106293107406683`*^9}, {
   3.510629367384368*^9, 3.510629370301573*^9}, {3.5106295599043055`*^9, 
   3.5106295603567066`*^9}, {3.510657365037108*^9, 3.5106574369532347`*^9}, 
   3.510657709126912*^9, {3.5106577452409763`*^9, 3.510657747034979*^9}, {
   3.5109190411611347`*^9, 3.510919114247263*^9}, 3.51091916922176*^9, {
   3.5109193418204627`*^9, 3.51091934586087*^9}, 3.5109555482566385`*^9, {
   3.5110026383125415`*^9, 3.5110026412609468`*^9}, {3.5111427006777735`*^9, 
   3.511142701020974*^9}, {3.5111428283327975`*^9, 3.5111428289723988`*^9}, {
   3.51114814166133*^9, 3.5111481490869427`*^9}, {3.511230607696045*^9, 
   3.511230615230858*^9}, {3.5112308557832804`*^9, 3.5112308586848855`*^9}, 
   3.5127273090023527`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness Matrix", "Subsection",
 CellChangeTimes->{{3.510628701318797*^9, 3.510628705218804*^9}, {
  3.510628763250906*^9, 3.5106287646393085`*^9}}],

Cell[CellGroupData[{

Cell["Only Real", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5109555066872606`*^9, 3.510955522479164*^9}, {
  3.5111459241018353`*^9, 3.511145929171844*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matKvv", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"potential", ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5105619627036214`*^9, 3.510561989847669*^9}, 
   3.510562426586036*^9, {3.510562624565984*^9, 3.5105626282787905`*^9}, 
   3.510562823575534*^9, {3.5106287756061277`*^9, 3.5106287991777687`*^9}, 
   3.510629372735177*^9, {3.5106295648963146`*^9, 3.5106295735387297`*^9}, 
   3.510634023423352*^9, 3.5106340995670853`*^9, 3.510634705986951*^9, 
   3.5106357421095705`*^9, {3.5106577195477304`*^9, 3.5106577496869836`*^9}, {
   3.510919174588169*^9, 3.5109192067710257`*^9}, {3.51091935724889*^9, 
   3.5109193604780955`*^9}, 3.510955522479164*^9, {3.511000578777713*^9, 
   3.5110005872953277`*^9}, 3.511002643460551*^9, {3.51100273993112*^9, 
   3.511002769945573*^9}, {3.5110030522904687`*^9, 3.511003069060498*^9}, {
   3.511141726877263*^9, 3.511141728328066*^9}, 3.511141847169074*^9, {
   3.51114310883689*^9, 3.511143116262503*^9}, {3.511143405602212*^9, 
   3.5111434129186244`*^9}, {3.5111459241018353`*^9, 3.511145929171844*^9}, {
   3.5112308780601196`*^9, 3.5112308853609324`*^9}, 3.512727244075039*^9, 
   3.512727310281555*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matKpv", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"potential", ",", 
           RowBox[{"allvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"allparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "allparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.510629298089046*^9, 3.510629328524699*^9}, {
   3.5106295791391397`*^9, 3.510629587048353*^9}, 3.5106341033890924`*^9, 
   3.5106343797903776`*^9, 3.510634742647015*^9, 3.510635745759977*^9, {
   3.510657728892147*^9, 3.510657756722596*^9}, 3.5109191710313625`*^9, {
   3.5109192125430355`*^9, 3.5109192243990564`*^9}, {3.510919348949675*^9, 
   3.5109193550336857`*^9}, {3.510955522479164*^9, 3.5109555416132584`*^9}, {
   3.5110027536747446`*^9, 3.511002764969164*^9}, {3.5111417317132716`*^9, 
   3.5111417374696817`*^9}, 3.511141852644684*^9, {3.5111428317960033`*^9, 
   3.5111428323108044`*^9}, {3.5111459241018353`*^9, 3.511145929171844*^9}, {
   3.511230887888137*^9, 3.5112309006801596`*^9}, 3.5127272467426434`*^9, 
   3.5127273111395564`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Length", "[", "allfloats", "]"}], "\[GreaterEqual]", "1"}], 
     ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"matKfv", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potential", ",", 
              RowBox[{"allvars", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKff", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potential", ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKfp", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potential", ",", 
              RowBox[{"allparams", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allparams", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKvv", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfv", ".", 
          RowBox[{"Inverse", "[", "matKff", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfv", "]"}]}], "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"matKpv", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfv", ".", 
          RowBox[{"Inverse", "[", "matKff", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfp", "]"}]}], "]"}]}]}], ";"}]}], 
    "\[IndentingNewLine]", "]"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5111416497975273`*^9, 3.5111417209960527`*^9}, {
  3.5111417526641083`*^9, 3.5111418619267*^9}, {3.511142246202175*^9, 
  3.5111422962938633`*^9}, {3.5111425664863377`*^9, 3.5111425873747745`*^9}, {
  3.5111428355400105`*^9, 3.5111428761468816`*^9}, {3.511142909967741*^9, 
  3.511142930606577*^9}, {3.511142981868267*^9, 3.5111430139887238`*^9}, {
  3.5111430674656177`*^9, 3.511143067824418*^9}, {3.511143466379918*^9, 
  3.51114346733152*^9}, {3.5111435471100597`*^9, 3.5111435507604666`*^9}, {
  3.5111436040657597`*^9, 3.5111436094009695`*^9}, {3.511145929171844*^9, 
  3.5111459456454725`*^9}, {3.511149406308751*^9, 3.5111494064335513`*^9}, {
  3.5112309057189684`*^9, 3.5112309636574697`*^9}, {3.5112312731776133`*^9, 
  3.5112313016164637`*^9}, {3.5127272495194483`*^9, 3.5127272553538585`*^9}, {
  3.512727288207516*^9, 3.5127273062099476`*^9}, {3.512735314694819*^9, 
  3.512735322167232*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["With Imaginary Part", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.51095551455171*^9, 3.510955530382616*^9}, {
  3.5111481601941624`*^9, 3.511148165466972*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matKvvc", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"potentialc", ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"allvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "allvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5105619627036214`*^9, 3.510561989847669*^9}, 
   3.510562426586036*^9, {3.510562624565984*^9, 3.5105626282787905`*^9}, 
   3.510562823575534*^9, {3.5106287756061277`*^9, 3.5106287991777687`*^9}, 
   3.510629372735177*^9, {3.5106295648963146`*^9, 3.5106295735387297`*^9}, 
   3.510634023423352*^9, 3.5106340995670853`*^9, 3.510634705986951*^9, 
   3.5106357421095705`*^9, {3.5106577195477304`*^9, 3.5106577496869836`*^9}, {
   3.510919174588169*^9, 3.5109192067710257`*^9}, {3.51091935724889*^9, 
   3.5109193604780955`*^9}, {3.510922241814151*^9, 3.5109222467374325`*^9}, 
   3.510955530382616*^9, {3.5110005587316775`*^9, 3.5110005689340954`*^9}, {
   3.511002971747527*^9, 3.511002992214763*^9}, {3.5111437697380514`*^9, 
   3.5111437766956635`*^9}, 3.511143821826543*^9, {3.5111481601941624`*^9, 
   3.511148165466972*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matKpvc", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"potentialc", ",", 
           RowBox[{"allvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"allparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "allparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.510629298089046*^9, 3.510629328524699*^9}, {
   3.5106295791391397`*^9, 3.510629587048353*^9}, 3.5106341033890924`*^9, 
   3.5106343797903776`*^9, 3.510634742647015*^9, 3.510635745759977*^9, {
   3.510657728892147*^9, 3.510657756722596*^9}, 3.5109191710313625`*^9, {
   3.5109192125430355`*^9, 3.5109192243990564`*^9}, {3.510919348949675*^9, 
   3.5109193550336857`*^9}, {3.5109222445573077`*^9, 
   3.5109222483205233`*^9}, {3.510955530382616*^9, 3.510955540563198*^9}, {
   3.511002993930766*^9, 3.5110029968011713`*^9}, {3.511143772499256*^9, 
   3.511143780985671*^9}, 3.5111438228093443`*^9, {3.5111481601941624`*^9, 
   3.511148165466972*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Length", "[", "allfloats", "]"}], "\[GreaterEqual]", "1"}], 
     ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"matKfvc", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potentialc", ",", 
              RowBox[{"allvars", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allvars", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKffc", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potentialc", ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKfpc", "=", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"potentialc", ",", 
              RowBox[{"allparams", "[", 
               RowBox[{"[", "n", "]"}], "]"}], ",", 
              RowBox[{"allfloats", "[", 
               RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"n", ",", 
              RowBox[{"Length", "[", "allparams", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"m", ",", 
              RowBox[{"Length", "[", "allfloats", "]"}]}], "}"}]}], "]"}], "/.",
           "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"matKvvc", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfvc", ".", 
          RowBox[{"Inverse", "[", "matKffc", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfvc", "]"}]}], "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"matKpvc", "+=", 
       RowBox[{"-", 
        RowBox[{"Chop", "[", 
         RowBox[{"matKfvc", ".", 
          RowBox[{"Inverse", "[", "matKffc", "]"}], ".", 
          RowBox[{"Transpose", "[", "matKfpc", "]"}]}], "]"}]}]}], ";"}]}], 
    "\[IndentingNewLine]", "]"}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5111416497975273`*^9, 3.5111417209960527`*^9}, {
   3.5111417526641083`*^9, 3.5111418619267*^9}, {3.511142246202175*^9, 
   3.5111422962938633`*^9}, {3.5111425664863377`*^9, 
   3.5111425873747745`*^9}, {3.5111428355400105`*^9, 
   3.5111428761468816`*^9}, {3.511142909967741*^9, 3.511142930606577*^9}, {
   3.511142981868267*^9, 3.5111430139887238`*^9}, {3.5111430674656177`*^9, 
   3.511143067824418*^9}, {3.511143466379918*^9, 3.51114346733152*^9}, {
   3.5111435471100597`*^9, 3.5111435507604666`*^9}, {3.5111436040657597`*^9, 
   3.5111436094009695`*^9}, {3.5111438011565065`*^9, 3.511143815914132*^9}, 
   3.511148165466972*^9, {3.5111494314871955`*^9, 3.511149431689996*^9}, {
   3.512727323993979*^9, 3.5127273242591796`*^9}, {3.5127352324358745`*^9, 
   3.512735240625889*^9}, {3.5127353013099957`*^9, 3.5127353103736115`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Damping Matrix", "Subsection",
 CellChangeTimes->{{3.510561334346517*^9, 3.510561337388523*^9}, {
  3.5113900124941225`*^9, 3.511390015442528*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"matGvv", "=", 
   RowBox[{"Chop", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{"damping", ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "n", "]"}], "]"}], ",", 
          RowBox[{"velvars", "[", 
           RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"n", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", ",", 
          RowBox[{"Length", "[", "velvars", "]"}]}], "}"}]}], "]"}], "/.", 
      "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}], 
  ";"}]], "Input",
 CellChangeTimes->{{3.510561339822127*^9, 3.5105613828470025`*^9}, {
   3.5105614843563805`*^9, 3.51056150657082*^9}, {3.510561663541896*^9, 
   3.5105617389524283`*^9}, {3.5105619183683434`*^9, 
   3.5105619553404083`*^9}, {3.5106285477833276`*^9, 
   3.5106286902115774`*^9}, {3.510629347182332*^9, 3.510629358086751*^9}, {
   3.5106295553334975`*^9, 3.5106295558638983`*^9}, {3.5106573586878967`*^9, 
   3.5106574988697433`*^9}, 3.5106575498350325`*^9, {3.5106575855122952`*^9, 
   3.510657586838298*^9}, 3.510657701592099*^9, {3.510918992301849*^9, 
   3.510919033813522*^9}, {3.510919072236389*^9, 3.5109190853248124`*^9}, {
   3.511002629373726*^9, 3.511002636112938*^9}, {3.511142697495368*^9, 
   3.511142697651368*^9}, {3.511148104096464*^9, 3.5111481386037245`*^9}, {
   3.5112308639264946`*^9, 3.5112308672337008`*^9}, {3.511390018359733*^9, 
   3.511390025722946*^9}, {3.5113900655654163`*^9, 3.511390079979841*^9}, 
   3.512727336255601*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matGpv", "=", 
    RowBox[{"Chop", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{"damping", ",", 
           RowBox[{"velvars", "[", 
            RowBox[{"[", "n", "]"}], "]"}], ",", 
           RowBox[{"velparams", "[", 
            RowBox[{"[", "m", "]"}], "]"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"n", ",", 
           RowBox[{"Length", "[", "velvars", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"m", ",", 
           RowBox[{"Length", "[", "velparams", "]"}]}], "}"}]}], "]"}], "/.", 
       "\[IndentingNewLine]", "subminvars"}], "/.", "subdefparams"}], "]"}]}],
    ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.510629277247409*^9, 3.5106293107406683`*^9}, {
   3.510629367384368*^9, 3.510629370301573*^9}, {3.5106295599043055`*^9, 
   3.5106295603567066`*^9}, {3.510657365037108*^9, 3.5106574369532347`*^9}, 
   3.510657709126912*^9, {3.5106577452409763`*^9, 3.510657747034979*^9}, {
   3.5109190411611347`*^9, 3.510919114247263*^9}, 3.51091916922176*^9, {
   3.5109193418204627`*^9, 3.51091934586087*^9}, 3.5109555482566385`*^9, {
   3.5110026383125415`*^9, 3.5110026412609468`*^9}, {3.5111427006777735`*^9, 
   3.511142701020974*^9}, {3.5111428283327975`*^9, 3.5111428289723988`*^9}, {
   3.51114814166133*^9, 3.5111481490869427`*^9}, {3.511230607696045*^9, 
   3.511230615230858*^9}, {3.5112308557832804`*^9, 3.5112308586848855`*^9}, {
   3.511390021198938*^9, 3.5113900346461616`*^9}, {3.5113900824290457`*^9, 
   3.511390094378667*^9}, 3.5127273373944025`*^9}]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Eigen Mode", "Section",
 CellChangeTimes->{{3.510639974173004*^9, 3.51063997762061*^9}}],

Cell[CellGroupData[{

Cell["Eigen Frequencies & Vectors", "Subsection",
 CellChangeTimes->{{3.5106399863410254`*^9, 3.510639995685442*^9}, {
  3.5109193175936203`*^9, 3.510919324769633*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eigen", "=", 
   RowBox[{"Eigensystem", "[", 
    RowBox[{
     RowBox[{"Inverse", "[", "matMvv", "]"}], ".", 
     RowBox[{"Re", "[", "matKvv", "]"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"eigenf", "=", 
   RowBox[{"Reverse", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", 
      RowBox[{"eigen", "[", 
       RowBox[{"[", "1", "]"}], "]"}], "]"}], "/", 
     RowBox[{"(", 
      RowBox[{"2", 
       RowBox[{"N", "[", "Pi", "]"}]}], ")"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"eigenv", "=", 
   RowBox[{"Reverse", "[", 
    RowBox[{"eigen", "[", 
     RowBox[{"[", "2", "]"}], "]"}], "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5106404812986946`*^9, 3.5106405612644353`*^9}, {
  3.510642026727812*^9, 3.510642034652625*^9}, {3.510642839520439*^9, 
  3.510642845308049*^9}, {3.5106429258041906`*^9, 3.510642929423397*^9}, {
  3.5106434421806974`*^9, 3.510643450791913*^9}, {3.5109192830239596`*^9, 
  3.510919329933242*^9}, {3.511143879000643*^9, 3.5111438815434475`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Eigen Mode Listup", "Subsection",
 CellChangeTimes->{{3.5106417564485364`*^9, 3.510641767571356*^9}, {
  3.5106435074824123`*^9, 3.5106435076384125`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eigenmodetable", "[", 
  RowBox[{"eigenf", ",", "eigenv", ",", "0.3"}], "]"}]], "Input",
 CellChangeTimes->{{3.5106432816876154`*^9, 3.510643291593633*^9}, {
   3.510921715107025*^9, 3.510921737338297*^9}, {3.510922100897091*^9, 
   3.5109221122847424`*^9}, 3.5109585862628655`*^9, {3.511407145469161*^9, 
   3.5114071462179623`*^9}, {3.5114900165592413`*^9, 3.511490016886842*^9}, {
   3.512727357924039*^9, 3.51272735881324*^9}, {3.5127749540546885`*^9, 
   3.5127749543510895`*^9}}],

Cell[BoxData[
 TagBox[GridBox[{
    {"\<\"N\"\>", "\<\"Freq\"\>", "\<\"Type\"\>"},
    {"1", "0.7058058325609774`", "\<\"zTM(-0.999807)\"\>"},
    {"2", "0.7058202695834438`", "\<\"xTM(-1.)\"\>"},
    {"3", "1.2871485599600814`", "\<\"yawTM(1.)\"\>"},
    {"4", "3.421300247439843`", "\<\"pitchTM(-1.)\"\>"},
    {"5", "15.67236638932915`", "\<\"yTM(1.)\"\>"},
    {"6", "22.11192799725625`", "\<\"rollTM(-1.)\"\>"}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.27999999999999997`], {
        Offset[0.7]}, 
       Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}],
  Function[BoxForm`e$, 
   TableForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{
  3.510643292919636*^9, 3.5106433400473185`*^9, 3.5106434238662653`*^9, 
   3.5106434762355576`*^9, 3.510643723168391*^9, 3.5106438344589863`*^9, 
   3.510643892756289*^9, 3.5106577700762196`*^9, 3.5108253730494413`*^9, 
   3.51082974384672*^9, 3.510829776169977*^9, 3.5108302338369813`*^9, 
   3.5108303752356296`*^9, {3.5108305553691463`*^9, 3.51083057477558*^9}, 
   3.5108307570151005`*^9, 3.510878378699349*^9, 3.5109217415145354`*^9, 
   3.510921781893845*^9, {3.5109221017821417`*^9, 3.5109221126697645`*^9}, 
   3.510956284636422*^9, 3.510958555156411*^9, 3.5109585879476686`*^9, 
   3.510976004120376*^9, 3.5109762241431627`*^9, 3.510981586648802*^9, 
   3.510997623852915*^9, 3.510997670574997*^9, 3.5109978286968746`*^9, 
   3.5110005994633493`*^9, 3.5110012262880497`*^9, 3.511002249466259*^9, 
   3.5110027958260183`*^9, 3.5110032275255766`*^9, 3.511003483288026*^9, 
   3.5110045135606356`*^9, 3.5110674087517023`*^9, 3.511144001897659*^9, 
   3.5111469200231843`*^9, 3.511147217780907*^9, 3.5111479339157653`*^9, 
   3.511149448413225*^9, {3.5111495020461197`*^9, 3.5111495104389343`*^9}, 
   3.511150737209489*^9, 3.51115124445998*^9, 3.5111525863899364`*^9, 
   3.511152661114068*^9, 3.5111551241458*^9, 3.5112282013762183`*^9, 
   3.511228596821313*^9, 3.5112287471743765`*^9, 3.5112289126906676`*^9, 
   3.5112289448423243`*^9, 3.5112292057371826`*^9, 3.5112293438286247`*^9, 
   3.511229669666397*^9, {3.511229940545273*^9, 3.511229945724482*^9}, 
   3.511230367908024*^9, 3.5112305163578844`*^9, 3.5112318289909897`*^9, 
   3.5112323159368496`*^9, 3.511232424778241*^9, 3.5112324581154995`*^9, 
   3.51123249266956*^9, 3.511232537082838*^9, 3.5112325809033155`*^9, 
   3.5112326565478477`*^9, 3.5112328450429792`*^9, 3.511232907536689*^9, 
   3.511233228663253*^9, 3.511329491833839*^9, 3.5113304379287033`*^9, 
   3.5113306313690434`*^9, 3.511331015972119*^9, 3.511331177635203*^9, 
   3.5113329869263806`*^9, 3.511406697171174*^9, 3.5114067818637223`*^9, 
   3.5114069491428165`*^9, 3.5114071480587654`*^9, 3.5114315657357717`*^9, 
   3.511490058913316*^9, 3.511494334787651*^9, 3.511496090224537*^9, 
   3.512355787464673*^9, 3.5123560041613693`*^9, 3.5127108132167463`*^9, 
   3.5127123258698044`*^9, 3.5127125177033415`*^9, 3.5127126119275064`*^9, 
   3.512730655448633*^9, 3.5127329310730324`*^9, 3.512774841625291*^9, 
   3.5127749548034897`*^9, 3.5127767917635202`*^9, 3.5181496136622777`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Eigen Mode 3D-Graphics", "Subsection",
 CellChangeTimes->{{3.5108295124827137`*^9, 3.510829513075515*^9}, {
  3.510956048202407*^9, 3.510956066017638*^9}}],

Cell[CellGroupData[{

Cell["Plot Range Definition", "Subsubsection",
 CellChangeTimes->{{3.51082764299103*^9, 3.5108276467818365`*^9}, {
   3.5109210878847294`*^9, 3.51092116774127*^9}, 3.5109212261165724`*^9, 
   3.511003172847481*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"maxz", "=", 
    RowBox[{"0.8", "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"z", "-", 
    RowBox[{"width", " ", "of", " ", "the", " ", "system"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"maxx", "=", 
    RowBox[{"0.8", "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"x", "-", 
    RowBox[{"width", " ", "of", " ", "the", " ", "system"}]}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"ytop", "=", 
    RowBox[{"0.0", "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Top", " ", "height", " ", "of", " ", "the", " ", "system"}], 
   "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"ybottom", "=", 
    RowBox[{
     RowBox[{"-", "1.0"}], "meter"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Bottom", " ", "height", " ", "of", " ", "the", " ", "system"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.5109211968977213`*^9, 3.5109212408741984`*^9}, {
   3.5109220308510847`*^9, 3.5109220309910927`*^9}, {3.510958542473589*^9, 
   3.51095854336279*^9}, {3.510968368593365*^9, 3.510968418575853*^9}, {
   3.5109954409480605`*^9, 3.510995447197418*^9}, {3.5109987157704387`*^9, 
   3.510998749638098*^9}, {3.511001189939986*^9, 3.511001205306013*^9}, 
   3.511003172847481*^9, {3.511129432507734*^9, 3.5111294518908424`*^9}, {
   3.511152966983805*^9, 3.5111529708682117`*^9}, {3.511228562750853*^9, 
   3.5112285692248645`*^9}, {3.511229309087364*^9, 3.5112293130029707`*^9}, {
   3.511329984950708*^9, 3.5113299977115307`*^9}, {3.5127101502779818`*^9, 
   3.512710157438394*^9}, {3.512774924648637*^9, 3.5127749351006556`*^9}, {
   3.5127768067395463`*^9, 3.512776816349163*^9}, {3.512776867517253*^9, 
   3.5127768929608974`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Eigen Mode 3D-Graphics", "Subsubsection",
 CellChangeTimes->{{3.512709012365183*^9, 3.5127090238936033`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eigenplot", "[", 
  RowBox[{"eigenv", ",", "0.2", ",", "1"}], "]"}]], "Input",
 CellChangeTimes->{{3.5108281708803577`*^9, 3.5108281797879734`*^9}, {
   3.510829089456771*^9, 3.510829089612771*^9}, {3.5108291378012557`*^9, 
   3.510829143760466*^9}, {3.510829207346178*^9, 3.5108292251926093`*^9}, {
   3.510829318496373*^9, 3.5108293323959975`*^9}, {3.510829380802882*^9, 
   3.510829395279708*^9}, {3.5108294539514112`*^9, 3.510829454965413*^9}, {
   3.5108783919749727`*^9, 3.5108784826891317`*^9}, {3.5108858833120837`*^9, 
   3.510885917039343*^9}, {3.510922053102357*^9, 3.5109220655000668`*^9}, {
   3.510922145501642*^9, 3.510922154304146*^9}, {3.5109586034072957`*^9, 
   3.5109586510965796`*^9}, 3.510965620835739*^9, {3.510980713272068*^9, 
   3.510980713412468*^9}, {3.510980751289335*^9, 3.5109807514297347`*^9}, {
   3.5109808130498433`*^9, 3.5109808621743293`*^9}, {3.5109855664036036`*^9, 
   3.5109855667624035`*^9}, {3.5109976790926123`*^9, 
   3.5109976792174125`*^9}, {3.511001058291355*^9, 3.511001058431755*^9}, {
   3.5110023185587797`*^9, 3.5110024376649895`*^9}, {3.5110673866620636`*^9, 
   3.511067386771264*^9}, {3.5110674313873425`*^9, 3.5110674526969795`*^9}, {
   3.511144014627281*^9, 3.511144014752082*^9}, {3.5111443615094905`*^9, 
   3.5111444172171884`*^9}, {3.5111444872457113`*^9, 
   3.5111444874173117`*^9}, {3.5111456221165047`*^9, 3.511145622912106*^9}, {
   3.5111469295080013`*^9, 3.5111469378540154`*^9}, {3.511151653352298*^9, 
   3.5111516692799263`*^9}, {3.511151703802787*^9, 3.511151709512397*^9}, {
   3.511151764393293*^9, 3.5111517990409536`*^9}, {3.5111527228745766`*^9, 
   3.5111527274921846`*^9}, {3.5111617369272847`*^9, 
   3.5111617373016853`*^9}, {3.5112282465694976`*^9, 
   3.5112282467566977`*^9}, {3.51122895378114*^9, 3.5112289539215403`*^9}, {
   3.511229358477051*^9, 3.511229397243119*^9}, {3.511229844729905*^9, 
   3.5112298448859053`*^9}, {3.5112299955977697`*^9, 3.511230027671426*^9}, {
   3.511230389014861*^9, 3.5112304126021023`*^9}, {3.511230529056307*^9, 
   3.5112305886172113`*^9}, {3.5112323789921603`*^9, 3.511232379179361*^9}, {
   3.511329531691909*^9, 3.5113295318791094`*^9}, {3.5113300860076857`*^9, 
   3.5113301632590213`*^9}, {3.5113302681380053`*^9, 3.511330268933607*^9}, {
   3.511330485617987*^9, 3.5113304862263885`*^9}, {3.5114070321661625`*^9, 
   3.5114070887786617`*^9}, {3.511407177605218*^9, 3.511407177886018*^9}, {
   3.511431574518587*^9, 3.511431579869396*^9}, {3.512710907830912*^9, 
   3.5127109337113576`*^9}, {3.512730678006273*^9, 3.5127307128567343`*^9}, 
   3.5127748485049033`*^9, {3.512774963820306*^9, 3.5127750423196435`*^9}, 
   3.512776799828734*^9, {3.5127768767056694`*^9, 3.512776912601332*^9}, {
   3.512777102578466*^9, 3.5127771027968664`*^9}}],

Cell[BoxData[
 Graphics3DBox[{Point3DBox[{0.4, 0., 0.4}], Point3DBox[{-0.4, 0., 0.4}], 
   Point3DBox[{0.4, 0., -0.4}], Point3DBox[{-0.4, 0., -0.4}], 
   Point3DBox[{0.4, -1., 0.4}], Point3DBox[{-0.4, -1., 0.4}], 
   Point3DBox[{0.4, -1., -0.4}], Point3DBox[{-0.4, -1., -0.4}], 
   Line3DBox[{{-0.125, -0.001109096806378509, -0.015}, {-0.125, \
-0.49984359390720473`, -0.21495290864367794`}}], 
   Line3DBox[{{-0.125, -0.001109096806378509, 
    0.015}, {-0.125, -0.4999615909177685, -0.18495314069948382`}}], 
   Line3DBox[{{0.125, -0.001109096806378509, 0.015}, {
    0.125, -0.4999615909177685, -0.18495314069948382`}}], 
   Line3DBox[{{0.125, -0.001109096806378509, -0.015}, {
    0.125, -0.49984359390720473`, -0.21495290864367794`}}], 
   GeometricTransformation3DBox[
    GeometricTransformation3DBox[GeometricTransformation3DBox[{
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{
         0.11888206453689419`, -0.4633845486077247, -0.14996132024218561`}, {
         0.11888206453689419`, -0.4633845486077247, -0.24996132024218565`}, {
         0.10112712429686843`, -0.428538516368034, -0.24996132024218565`}, {
         0.10112712429686843`, -0.428538516368034, -0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{
         0.10112712429686843`, -0.428538516368034, -0.14996132024218561`}, {
         0.10112712429686843`, -0.428538516368034, -0.24996132024218565`}, {
         0.07347315653655914, -0.4008845486077247, -0.24996132024218565`}, {
         0.07347315653655914, -0.4008845486077247, -0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{
         0.07347315653655914, -0.4008845486077247, -0.14996132024218561`}, {
         0.07347315653655914, -0.4008845486077247, -0.24996132024218565`}, {
         0.03862712429686843, -0.38312960836769894`, -0.24996132024218565`}, {
         0.03862712429686843, -0.38312960836769894`, \
-0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{
         0.03862712429686843, -0.38312960836769894`, -0.14996132024218561`}, {
         0.03862712429686843, -0.38312960836769894`, -0.24996132024218565`}, {
         0., -0.37701167290459314`, -0.24996132024218565`}, {
         0., -0.37701167290459314`, -0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{0., -0.37701167290459314`, -0.14996132024218561`}, {
         0., -0.37701167290459314`, -0.24996132024218565`}, \
{-0.03862712429686843, -0.38312960836769894`, -0.24996132024218565`}, \
{-0.03862712429686843, -0.38312960836769894`, -0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{-0.03862712429686843, -0.38312960836769894`, \
-0.14996132024218561`}, {-0.03862712429686843, -0.38312960836769894`, \
-0.24996132024218565`}, {-0.07347315653655914, -0.4008845486077247, \
-0.24996132024218565`}, {-0.07347315653655914, -0.4008845486077247, \
-0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{-0.07347315653655914, -0.4008845486077247, \
-0.14996132024218561`}, {-0.07347315653655914, -0.4008845486077247, \
-0.24996132024218565`}, {-0.10112712429686843`, -0.428538516368034, \
-0.24996132024218565`}, {-0.10112712429686843`, -0.428538516368034, \
-0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{-0.10112712429686843`, -0.428538516368034, \
-0.14996132024218561`}, {-0.10112712429686843`, -0.428538516368034, \
-0.24996132024218565`}, {-0.11888206453689419`, -0.4633845486077247, \
-0.24996132024218565`}, {-0.11888206453689419`, -0.4633845486077247, \
-0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{-0.11888206453689419`, -0.4633845486077247, \
-0.14996132024218561`}, {-0.11888206453689419`, -0.4633845486077247, \
-0.24996132024218565`}, {-0.125, -0.5020116729045931, -0.24996132024218565`}, \
{-0.125, -0.5020116729045931, -0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{-0.125, -0.5020116729045931, -0.14996132024218561`}, \
{-0.125, -0.5020116729045931, -0.24996132024218565`}, {-0.11888206453689419`, \
-0.5406387972014616, -0.24996132024218565`}, {-0.11888206453689419`, \
-0.5406387972014616, -0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{-0.11888206453689419`, -0.5406387972014616, \
-0.14996132024218561`}, {-0.11888206453689419`, -0.5406387972014616, \
-0.24996132024218565`}, {-0.10112712429686843`, -0.5754848294411523, \
-0.24996132024218565`}, {-0.10112712429686843`, -0.5754848294411523, \
-0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{-0.10112712429686843`, -0.5754848294411523, \
-0.14996132024218561`}, {-0.10112712429686843`, -0.5754848294411523, \
-0.24996132024218565`}, {-0.07347315653655914, -0.6031387972014616, \
-0.24996132024218565`}, {-0.07347315653655914, -0.6031387972014616, \
-0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{-0.07347315653655914, -0.6031387972014616, \
-0.14996132024218561`}, {-0.07347315653655914, -0.6031387972014616, \
-0.24996132024218565`}, {-0.03862712429686843, -0.6208937374414873, \
-0.24996132024218565`}, {-0.03862712429686843, -0.6208937374414873, \
-0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{-0.03862712429686843, -0.6208937374414873, \
-0.14996132024218561`}, {-0.03862712429686843, -0.6208937374414873, \
-0.24996132024218565`}, {0., -0.6270116729045931, -0.24996132024218565`}, {
         0., -0.6270116729045931, -0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{0., -0.6270116729045931, -0.14996132024218561`}, {
         0., -0.6270116729045931, -0.24996132024218565`}, {
         0.03862712429686843, -0.6208937374414873, -0.24996132024218565`}, {
         0.03862712429686843, -0.6208937374414873, -0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{
         0.03862712429686843, -0.6208937374414873, -0.14996132024218561`}, {
         0.03862712429686843, -0.6208937374414873, -0.24996132024218565`}, {
         0.07347315653655914, -0.6031387972014616, -0.24996132024218565`}, {
         0.07347315653655914, -0.6031387972014616, -0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{
         0.07347315653655914, -0.6031387972014616, -0.14996132024218561`}, {
         0.07347315653655914, -0.6031387972014616, -0.24996132024218565`}, {
         0.10112712429686843`, -0.5754848294411523, -0.24996132024218565`}, {
         0.10112712429686843`, -0.5754848294411523, -0.14996132024218561`}}]}, 
       {RGBColor[0, 0, 1], 
        Polygon3DBox[{{
         0.10112712429686843`, -0.5754848294411523, -0.14996132024218561`}, {
         0.10112712429686843`, -0.5754848294411523, -0.24996132024218565`}, {
         0.11888206453689419`, -0.5406387972014616, -0.24996132024218565`}, {
         0.11888206453689419`, -0.5406387972014616, -0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{
         0.11888206453689419`, -0.5406387972014616, -0.14996132024218561`}, {
         0.11888206453689419`, -0.5406387972014616, -0.24996132024218565`}, {
         0.125, -0.5020116729045931, -0.24996132024218565`}, {
         0.125, -0.5020116729045931, -0.14996132024218561`}}]}, 
       {RGBColor[1, 0, 0], 
        Polygon3DBox[{{0.125, -0.5020116729045931, -0.14996132024218561`}, {
         0.125, -0.5020116729045931, -0.24996132024218565`}, {
         0.11888206453689419`, -0.4633845486077247, -0.24996132024218565`}, {
         0.11888206453689419`, -0.4633845486077247, \
-0.14996132024218561`}}]}}, {{{1., 0., 0.}, {0., 
       0.9999922648064712, -0.003933233685457812}, {0., 0.003933233685457812, 
       0.9999922648064712}}, {0., -0.0007903777580088311, 
       0.001972982482851021}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 1.}}, {
      0., 0., 0.}}], {{{1., 0., 0.}, {0., 1., 0.}, {0., 0., 1.}}, {0., 0., 
     0.}}], GeometricTransformation3DBox[
    GeometricTransformation3DBox[
     GeometricTransformation3DBox[{
       Polygon3DBox[{{0.19972186842198225`, 0, 0.06489356881873896}, {
         0.009510565162951531, 0, 0.003090169943749473}, {
         0.00809016994374947, 0, 0.005877852522924728}, {
         0.169893568818739, 0, 0.12343490298141938`}}], 
       Polygon3DBox[{{0.169893568818739, 0, 0.12343490298141938`}, {
         0.00809016994374947, 0, 0.005877852522924728}, {
         0.005877852522924728, 0, 0.00809016994374947}, {
         0.12343490298141938`, 0, 0.169893568818739}}], 
       Polygon3DBox[{{0.12343490298141938`, 0, 0.169893568818739}, {
         0.005877852522924728, 0, 0.00809016994374947}, {
         0.003090169943749473, 0, 0.009510565162951531}, {
         0.06489356881873896, 0, 0.19972186842198225`}}], 
       Polygon3DBox[{{0.06489356881873896, 0, 0.19972186842198225`}, {
         0.003090169943749473, 0, 0.009510565162951531}, {
         0., 0, 0.009999999999999995}, {0., 0, 0.21000000000000002`}}], 
       Polygon3DBox[{{0., 0, 0.21000000000000002`}, {
         0., 0, 0.009999999999999995}, {-0.003090169943749473, 0, 
          0.009510565162951531}, {-0.06489356881873896, 0, 
          0.19972186842198225`}}], 
       Polygon3DBox[{{-0.06489356881873896, 0, 
          0.19972186842198225`}, {-0.003090169943749473, 0, 
          0.009510565162951531}, {-0.005877852522924728, 0, 
          0.00809016994374947}, {-0.12343490298141938`, 0, 
          0.169893568818739}}], 
       Polygon3DBox[{{-0.12343490298141938`, 0, 
          0.169893568818739}, {-0.005877852522924728, 0, 
          0.00809016994374947}, {-0.00809016994374947, 0, 
          0.005877852522924728}, {-0.169893568818739, 0, 
          0.12343490298141938`}}], 
       Polygon3DBox[{{-0.169893568818739, 0, 
          0.12343490298141938`}, {-0.00809016994374947, 0, 
          0.005877852522924728}, {-0.009510565162951531, 0, 
          0.003090169943749473}, {-0.19972186842198225`, 0, 
          0.06489356881873896}}], 
       Polygon3DBox[{{-0.19972186842198225`, 0, 
          0.06489356881873896}, {-0.009510565162951531, 0, 
          0.003090169943749473}, {-0.009999999999999995, 0, 
          0.}, {-0.21000000000000002`, 0, 0.}}], 
       Polygon3DBox[{{-0.21000000000000002`, 0, 0.}, {-0.009999999999999995, 
          0, 0.}, {-0.009510565162951531, 
          0, -0.003090169943749473}, {-0.19972186842198225`, 
          0, -0.06489356881873896}}], 
       Polygon3DBox[{{-0.19972186842198225`, 
          0, -0.06489356881873896}, {-0.009510565162951531, 
          0, -0.003090169943749473}, {-0.00809016994374947, 
          0, -0.005877852522924728}, {-0.169893568818739, 
          0, -0.12343490298141938`}}], 
       Polygon3DBox[{{-0.169893568818739, 
          0, -0.12343490298141938`}, {-0.00809016994374947, 
          0, -0.005877852522924728}, {-0.005877852522924728, 
          0, -0.00809016994374947}, {-0.12343490298141938`, 
          0, -0.169893568818739}}], 
       Polygon3DBox[{{-0.12343490298141938`, 
          0, -0.169893568818739}, {-0.005877852522924728, 
          0, -0.00809016994374947}, {-0.003090169943749473, 
          0, -0.009510565162951531}, {-0.06489356881873896, 
          0, -0.19972186842198225`}}], 
       Polygon3DBox[{{-0.06489356881873896, 
          0, -0.19972186842198225`}, {-0.003090169943749473, 
          0, -0.009510565162951531}, {0., 0, -0.009999999999999995}, {
         0., 0, -0.21000000000000002`}}], 
       Polygon3DBox[{{0., 0, -0.21000000000000002`}, {
         0., 0, -0.009999999999999995}, {
         0.003090169943749473, 0, -0.009510565162951531}, {
         0.06489356881873896, 0, -0.19972186842198225`}}], 
       Polygon3DBox[{{0.06489356881873896, 0, -0.19972186842198225`}, {
         0.003090169943749473, 0, -0.009510565162951531}, {
         0.005877852522924728, 0, -0.00809016994374947}, {
         0.12343490298141938`, 0, -0.169893568818739}}], 
       Polygon3DBox[{{0.12343490298141938`, 0, -0.169893568818739}, {
         0.005877852522924728, 0, -0.00809016994374947}, {
         0.00809016994374947, 0, -0.005877852522924728}, {
         0.169893568818739, 0, -0.12343490298141938`}}], 
       Polygon3DBox[{{0.169893568818739, 0, -0.12343490298141938`}, {
         0.00809016994374947, 0, -0.005877852522924728}, {
         0.009510565162951531, 0, -0.003090169943749473}, {
         0.19972186842198225`, 0, -0.06489356881873896}}], 
       Polygon3DBox[{{0.19972186842198225`, 0, -0.06489356881873896}, {
         0.009510565162951531, 0, -0.003090169943749473}, {
         0.009999999999999995, 0, 0.}, {0.21000000000000002`, 0, 0.}}], 
       Polygon3DBox[{{0.21000000000000002`, 0, 0.}, {
         0.009999999999999995, 0, 0.}, {
         0.009510565162951531, 0, 0.003090169943749473}, {
         0.19972186842198225`, 0, 0.06489356881873896}}]}, {{{1, 0, 0}, {0, 1,
        0}, {0, 0, 1}}, {0, 0, 0}}], {{{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}, {0, 
      0, 0}}], {{{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}, {0, 0, 0}}]},
  AutomaticImageSize->True,
  ImageSize->{347.7801404434861, 423.2131909884581},
  ViewPoint->{-3.3107164403470715`, 0.6674171654154907, -0.20907170761335944`},
  ViewVertical->{-1.1216019795806162`, 
   0.43536262920406704`, -0.09138808652066317}]], "Output",
 CellChangeTimes->{
  3.5111617395012894`*^9, {3.511228226351862*^9, 3.511228247177899*^9}, 
   3.511228383444138*^9, 3.511228597258114*^9, 3.511228747564378*^9, 
   3.5112287993252687`*^9, 3.5112289130494685`*^9, {3.511228945216725*^9, 
   3.511228955434743*^9}, 3.5112292060803833`*^9, {3.5112293442498255`*^9, 
   3.51122939789832*^9}, 3.5112296699939976`*^9, 3.511229845541106*^9, {
   3.5112299408728733`*^9, 3.5112299460676827`*^9}, {3.5112300007925787`*^9, 
   3.5112300283266273`*^9}, {3.5112303683760242`*^9, 3.511230413304103*^9}, {
   3.5112305166542845`*^9, 3.5112305897248135`*^9}, 3.511231829458991*^9, 
   3.5112323163736506`*^9, 3.511232379912562*^9, 3.511232425199442*^9, 
   3.5112324586459007`*^9, 3.511232492997161*^9, 3.511232537472839*^9, 
   3.511232581215316*^9, 3.5112326569066486`*^9, 3.5112328454329796`*^9, 
   3.5112329079110894`*^9, 3.5112332290844536`*^9, 3.511329534437514*^9, 
   3.511329918135791*^9, {3.511330019067968*^9, 3.511330036477598*^9}, {
   3.5113300876612883`*^9, 3.511330164990624*^9}, 3.511330270368809*^9, 
   3.511330439254706*^9, 3.511330487630391*^9, 3.5113306319774446`*^9, 
   3.5113329875503817`*^9, 3.5114066981539755`*^9, 3.5114067825033236`*^9, 
   3.5114069498136177`*^9, {3.511407033226964*^9, 3.5114070948626723`*^9}, 
   3.51140717883762*^9, {3.511431565876172*^9, 3.511431582537001*^9}, 
   3.511490059584117*^9, 3.5114943360824537`*^9, 3.5114960912229385`*^9, 
   3.5123557887907486`*^9, {3.512355920904023*^9, 3.5123559369096513`*^9}, 
   3.51235600480097*^9, 3.512710814199548*^9, {3.512710909468915*^9, 
   3.512710934569359*^9}, 3.5127123264782057`*^9, 3.512712518342942*^9, 
   3.512712612567108*^9, {3.512730656244234*^9, 3.512730713636735*^9}, 
   3.5127329316658335`*^9, {3.5127748420464916`*^9, 3.5127748489105043`*^9}, {
   3.512774915460221*^9, 3.5127750431464453`*^9}, {3.5127767919351206`*^9, 
   3.5127768189231677`*^9}, {3.5127768590620384`*^9, 3.512776912866533*^9}, 
   3.512777103233667*^9, 3.5181496141926785`*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Transfer Function Plot", "Section",
 CellChangeTimes->{{3.510629600074376*^9, 3.5106296028355813`*^9}, {
  3.510922302451619*^9, 3.5109223035016794`*^9}}],

Cell[CellGroupData[{

Cell["Frequency List", "Subsection",
 CellChangeTimes->{{3.505457309098371*^9, 3.5054573134988823`*^9}, {
  3.505457392369238*^9, 3.505457393435096*^9}, {3.507903438769166*^9, 
  3.507903442019352*^9}, {3.5079450761978855`*^9, 3.5079450876015053`*^9}, {
  3.507945118302359*^9, 3.507945140813199*^9}, {3.5109227629829597`*^9, 
  3.510922763595995*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"fmin", "=", 
    RowBox[{"1", " ", 
     SuperscriptBox["10", 
      RowBox[{"-", "1"}]], " ", "hertz"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"minimum", " ", "frequency"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"fmax", "=", 
    RowBox[{"5", " ", 
     SuperscriptBox["10", "1"], "hertz"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"maximum", " ", "freqnency"}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"ngrid", "=", "2001"}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"number", " ", "of", " ", "grids"}], 
   "*)"}]}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5079437907558236`*^9, 3.5079439830729613`*^9}, {
   3.5079440622431*^9, 3.5079440624147005`*^9}, {3.5079440956739593`*^9, 
   3.507944096875161*^9}, 3.5079441778237033`*^9, {3.5079445464225545`*^9, 
   3.5079445466097546`*^9}, {3.50794501640298*^9, 3.5079450202405868`*^9}, 
   3.5079450559646497`*^9, 3.507945701790184*^9, {3.507945743301857*^9, 
   3.5079457477166643`*^9}, {3.508056511433575*^9, 3.508056511776776*^9}, {
   3.5080566504610195`*^9, 3.50805665075742*^9}, {3.508117933375839*^9, 
   3.5081179356066427`*^9}, {3.5082143648046265`*^9, 3.508214371684239*^9}, {
   3.510405201938513*^9, 3.510405203850622*^9}, {3.5104081570565357`*^9, 
   3.510408158312608*^9}, {3.5104088030074825`*^9, 3.510408811078944*^9}, {
   3.5104129367289176`*^9, 3.5104129458664403`*^9}, {3.5104455061152945`*^9, 
   3.5104455111697035`*^9}, {3.5105372720362353`*^9, 
   3.5105372721766357`*^9}, {3.5105373756672173`*^9, 3.510537375854418*^9}, 
   3.5106333737914104`*^9, {3.5106358171925025`*^9, 3.510635817473303*^9}, {
   3.5109228206962614`*^9, 3.5109228592864685`*^9}, 3.5109676607265215`*^9, {
   3.5110038763463163`*^9, 3.5110038811043243`*^9}, {3.5113312825453873`*^9, 
   3.511331283746589*^9}, {3.5114943820713344`*^9, 3.5114943893565474`*^9}, {
   3.5114955948018646`*^9, 3.5114955949422646`*^9}, {3.511497703881379*^9, 
   3.5114977039749794`*^9}, {3.512727423163353*^9, 3.512727491881474*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"freq", "=", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"imin", ",", "imax", ",", "dgrid"}], "}"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"imin", "=", 
        RowBox[{"Log", "[", 
         RowBox[{"10", ",", "fmin"}], "]"}]}], ";", 
       RowBox[{"imax", "=", 
        RowBox[{"Log", "[", 
         RowBox[{"10", ",", "fmax"}], "]"}]}], ";", 
       RowBox[{"dgrid", "=", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"imax", "-", "imin"}], ")"}], "/", 
         RowBox[{"(", 
          RowBox[{"ngrid", "-", "1"}], ")"}]}]}], ";", "\n", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"10", "^", "i"}], ",", 
         RowBox[{"{", 
          RowBox[{"i", ",", "imin", ",", "imax", ",", "dgrid"}], "}"}]}], 
        "]"}]}]}], "]"}]}], ";"}], 
  RowBox[{"(*", 
   RowBox[{"Frequency", " ", "list"}], "*)"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.5079437907558236`*^9, 3.5079439830729613`*^9}, {
   3.5079440622431*^9, 3.5079440624147005`*^9}, {3.5079440956739593`*^9, 
   3.507944096875161*^9}, 3.5079441778237033`*^9, {3.5079445464225545`*^9, 
   3.5079445466097546`*^9}, {3.50794501640298*^9, 3.5079450202405868`*^9}, 
   3.5079450559646497`*^9, 3.507945701790184*^9, {3.507945743301857*^9, 
   3.5079457477166643`*^9}, {3.508056511433575*^9, 3.508056511776776*^9}, {
   3.5080566504610195`*^9, 3.50805665075742*^9}, {3.508117933375839*^9, 
   3.5081179356066427`*^9}, {3.5082143648046265`*^9, 3.508214371684239*^9}, {
   3.510405201938513*^9, 3.510405203850622*^9}, {3.5104081570565357`*^9, 
   3.510408158312608*^9}, {3.5104088030074825`*^9, 3.510408811078944*^9}, {
   3.5104129367289176`*^9, 3.5104129458664403`*^9}, {3.5104455061152945`*^9, 
   3.5104455111697035`*^9}, {3.5105372720362353`*^9, 
   3.5105372721766357`*^9}, {3.5105373756672173`*^9, 3.510537375854418*^9}, 
   3.5106333737914104`*^9, {3.5106358171925025`*^9, 3.510635817473303*^9}, {
   3.5109228206962614`*^9, 3.5109228592864685`*^9}, 3.5109676607265215`*^9, {
   3.5110038763463163`*^9, 3.5110038811043243`*^9}, {3.5113312825453873`*^9, 
   3.511331283746589*^9}, {3.5114943820713344`*^9, 3.5114943893565474`*^9}, {
   3.5114955948018646`*^9, 3.5114955949422646`*^9}, {3.511497703881379*^9, 
   3.5114977039749794`*^9}, {3.512727423163353*^9, 3.512727540428759*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Actuator Definition", "Subsection",
 CellChangeTimes->{{3.5109654612318583`*^9, 3.510965462635861*^9}, {
  3.5109690024068785`*^9, 3.510969004138481*^9}}],

Cell[CellGroupData[{

Cell["Actuator Name List", "Subsubsection",
 CellChangeTimes->{{3.5113307936249285`*^9, 3.5113308073529525`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"allacts", "=", 
    RowBox[{"{", 
     RowBox[{"aTM1", ",", "aTM2", ",", "aTM3", ",", "aTM4"}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5109727782843103`*^9, 3.5109728075967617`*^9}, 
   3.51133080025494*^9, {3.5113912910255685`*^9, 3.5113912956275764`*^9}, {
   3.5114124581364727`*^9, 3.511412463674483*^9}, {3.5114130712487497`*^9, 
   3.5114130854447746`*^9}, 3.5127114692567*^9, 3.5127751682898645`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Making Actuator List", "Subsubsection",
 CellChangeTimes->{{3.5113308136397634`*^9, 3.511330818600572*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"actuatorlist", "=", 
    RowBox[{"Join", "[", "\[IndentingNewLine]", 
     RowBox[{"Table", "[", 
      RowBox[{"(*", "TM", "*)"}], "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{"posTM", ",", 
         RowBox[{"(*", 
          RowBox[{
          "coordinate", " ", "list", " ", "of", " ", "actuated", " ", 
           "body"}], "*)"}], "\[IndentingNewLine]", 
         RowBox[{"posTMa", "[", 
          RowBox[{"[", "n", "]"}], "]"}], ",", 
         RowBox[{"(*", 
          RowBox[{
          "position", " ", "of", " ", "actuator", " ", "at", " ", "actuated", 
           " ", "body"}], "*)"}], "\[IndentingNewLine]", "posg", ",", 
         RowBox[{"(*", 
          RowBox[{
          "coordinate", " ", "list", " ", "of", " ", "recoil", " ", "body"}], 
          "*)"}], "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
         RowBox[{"(*", 
          RowBox[{
          "position", " ", "of", " ", "actuator", " ", "at", " ", "recoil", 
           " ", "body"}], "*)"}], "\[IndentingNewLine]", 
         RowBox[{"dirTMa", "[", 
          RowBox[{"[", "n", "]"}], "]"}]}], 
        RowBox[{"(*", 
         RowBox[{
         "working", " ", "direction", " ", "defined", " ", "at", " ", 
          "recoil", " ", "body"}], "*)"}], "\[IndentingNewLine]", "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"n", ",", "4"}], "}"}]}], "\[IndentingNewLine]", "]"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.5109654818394947`*^9, 3.510965501292729*^9}, {
   3.510967724749034*^9, 3.510967731956247*^9}, {3.510969081296217*^9, 
   3.5109691312787046`*^9}, {3.5109691764251842`*^9, 3.510969496490946*^9}, {
   3.510969534071412*^9, 3.5109695354442143`*^9}, {3.5109728140863733`*^9, 
   3.510972844880827*^9}, 3.510974608651125*^9, {3.511003646932313*^9, 
   3.5110036534843245`*^9}, {3.5111471459271812`*^9, 
   3.5111471546631966`*^9}, {3.5111473903484106`*^9, 3.511147422125666*^9}, {
   3.5111552347923946`*^9, 3.5111552422648077`*^9}, {3.5113307607556705`*^9, 
   3.511330775575697*^9}, {3.5113321606397295`*^9, 3.511332172963751*^9}, {
   3.5113913047691927`*^9, 3.5113913416788573`*^9}, {3.5114131119804215`*^9, 
   3.5114131610581074`*^9}, 3.5127114769319134`*^9, {3.5127751755126777`*^9, 
   3.5127751759026785`*^9}, {3.512776947139793*^9, 3.512776954144205*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Transfer Function Plot", "Subsection",
 CellChangeTimes->{{3.5106327360154905`*^9, 3.510632742271102*^9}, {
  3.5109229122064953`*^9, 3.5109229133585615`*^9}, {3.510975022395052*^9, 
  3.510975046309894*^9}, {3.511431088725732*^9, 3.511431089770934*^9}}],

Cell[CellGroupData[{

Cell["Usage Notes", "Subsubsection",
 CellChangeTimes->{{3.512731063640951*^9, 3.5127310662461557`*^9}}],

Cell[BoxData[
 RowBox[{"?", "tfplot"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127276236861057`*^9, 3.5127276254801083`*^9}, 
   3.5127278088272305`*^9, 3.5127285744453754`*^9}],

Cell[BoxData[
 StyleBox["\<\"tfplot[{param},{var},freqlist] calculates transfer functions \
from input parameters' displacement to output variables' displacement.\"\>", 
  "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.5181496144266787`*^9},
 CellTags->"Info3518182014-3702807"],

Cell[BoxData[
 RowBox[{"?", "tfplotf"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5127277012026415`*^9, 3.512727705445849*^9}, 
   3.5127278088272305`*^9, 3.5127285744609756`*^9}],

Cell[BoxData[
 StyleBox["\<\"tfplot[{varin},{varout},freq] calculates transfer functions \
from force exerted to the input variables, to output variables' displacement.\
\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.518149614598279*^9},
 CellTags->"Info3518182014-8479246"],

Cell[BoxData[
 RowBox[{"?", "tfplota"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.512727708347454*^9, 3.512727710375458*^9}, 
   3.5127278088272305`*^9, 3.5127285744609756`*^9}],

Cell[BoxData[
 StyleBox["\<\"tfplota[{act},{var},freqlist] shows transfer functions from \
input actuator force to output variables' displacement. Input can be a linear \
combination of actuators.\"\>", "MSG"]], "Print", "PrintUsage",
 CellChangeTimes->{3.5181496147074795`*^9},
 CellTags->"Info3518182014-8742825"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Plots", "Subsubsection",
 CellChangeTimes->{{3.5127310719245653`*^9, 3.512731072782567*^9}}],

Cell[CellGroupData[{

Cell["Transfer function from disp. to disp.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511418345274813*^9, 3.511418367785653*^9}}],

Cell[BoxData[
 RowBox[{"tfplot", "[", 
  RowBox[{
   RowBox[{"{", "zg", "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM"}], "}"}], ",", "freq"}], "]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.510957748603794*^9, 3.5109577777446456`*^9}, {
   3.510957947894144*^9, 3.510957950312149*^9}, 3.5109588754873734`*^9, {
   3.510960251768591*^9, 3.5109602568386*^9}, {3.5109644688037157`*^9, 
   3.5109645325454273`*^9}, {3.5109646318084016`*^9, 3.510964636956411*^9}, {
   3.5109652306946535`*^9, 3.510965232036256*^9}, {3.5109653621872845`*^9, 
   3.510965374776507*^9}, {3.510965421685789*^9, 3.510965427161399*^9}, {
   3.5109750765583467`*^9, 3.5109750793195515`*^9}, {3.5109818646568904`*^9, 
   3.5109818950301437`*^9}, {3.5109829019735146`*^9, 
   3.5109829028627157`*^9}, {3.511147957815007*^9, 3.511147970092229*^9}, {
   3.5111480189515142`*^9, 3.5111480302771344`*^9}, {3.511152738053403*^9, 
   3.5111527568982363`*^9}, {3.5111614206837378`*^9, 
   3.5111614428905745`*^9}, {3.511232181917014*^9, 3.5112321830402164`*^9}, {
   3.5112326734114776`*^9, 3.5112326900723066`*^9}, {3.5112327680568438`*^9, 
   3.5112327702252474`*^9}, 3.51123286263981*^9, {3.5112329654595904`*^9, 
   3.5112329723704023`*^9}, {3.51123301048127*^9, 3.51123301626888*^9}, {
   3.511233061602559*^9, 3.511233087561005*^9}, {3.511233247133685*^9, 
   3.511233263388914*^9}, {3.511330838459407*^9, 3.511330842000613*^9}, {
   3.5113312692229633`*^9, 3.5113312709701667`*^9}, {3.5113904840985513`*^9, 
   3.511390509822996*^9}, {3.511414980348903*^9, 3.5114149810821047`*^9}, 
   3.511418367785653*^9, {3.512728004420374*^9, 3.5127280098959837`*^9}, {
   3.512733378388218*^9, 3.512733383676627*^9}, {3.5127354019145722`*^9, 
   3.512735408201383*^9}}],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[Large], LineBox[CompressedData["
1:eJwUVnc81/8T/3wkJISQ9c2qRKWiYZ9IpYwQ2giVVFZUKJKkgZaMzOyZmRX3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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
       LineBox[CompressedData["
1:eJwUl3k8lN8Xx41sM2PfjbFmSaksRdmeohQlRQgVUiKVIuqLItklStmFiEiy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        "]]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{-2.302585092994045, -9.331263923106688},
     BaseStyle->{FontSize -> Scaled[0.04]},
     CoordinatesToolOptions:>{"DisplayFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& ), "CopiedValueFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& )},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     Frame->True,
     FrameLabel->{
       FormBox["\"Frequency [Hz]\"", TraditionalForm]},
     FrameTicks->{{{{-9.210340371976182, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "4"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-4.605170185988091, 
          FormBox["0.01`", TraditionalForm]}, {0, 
          FormBox["1", TraditionalForm]}, {4.605170185988092, 
          FormBox["100", TraditionalForm]}, {-6.725433722188183, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.074846156047033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.683979847360021, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.403677882205863, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.184988681241033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.005647752585217, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.853631545286591, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.721704002244043, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.120263536200091, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.4696759700589417`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.07880966137193, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.7985076962177716, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.579818495252942, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.40047756659712525`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2484613592984996, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.11653381625595151`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.4849066497880004`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.1354942159291497`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.5263605246161616`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.8066624897703196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.02535169073515, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.204692619390966, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.356708826689592, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.48863636973214, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-9.210340371976182, 
          FormBox["\"\"", TraditionalForm]}, {-4.605170185988091, 
          FormBox["\"\"", TraditionalForm]}, {0, 
          FormBox["\"\"", TraditionalForm]}, {4.605170185988092, 
          FormBox["\"\"", TraditionalForm]}, {-6.725433722188183, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.074846156047033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.683979847360021, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.403677882205863, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.184988681241033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.005647752585217, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.853631545286591, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.721704002244043, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.120263536200091, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.4696759700589417`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.07880966137193, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.7985076962177716, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.579818495252942, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.40047756659712525`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2484613592984996, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.11653381625595151`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.4849066497880004`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.1354942159291497`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.5263605246161616`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.8066624897703196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.02535169073515, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.204692619390966, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.356708826689592, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.48863636973214, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}, {{{-2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-0.6931471805599453, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {0., 
          FormBox[
           TagBox[
            InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {1.6094379124341003`, 
          FormBox[
           TagBox[
            InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {3.912023005428146, 
          FormBox[
           TagBox[
            InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {4.605170185988092, 
          FormBox[
           TagBox[
            InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {6.214608098422191, 
          FormBox[
           TagBox[
            InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {-0.6931471805599453, 
          FormBox["\"\"", TraditionalForm]}, {0., 
          FormBox["\"\"", TraditionalForm]}, {1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm]}, {2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {3.912023005428146, 
          FormBox["\"\"", TraditionalForm]}, {4.605170185988092, 
          FormBox["\"\"", TraditionalForm]}, {6.214608098422191, 
          FormBox["\"\"", TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}},
     GridLines->{{-2.302585092994046, -0.6931471805599453, 0., 
       1.6094379124341003`, 2.302585092994046, 3.912023005428146, 
       4.605170185988092, 
       6.214608098422191, -1.6094379124341003`, -1.203972804325936, \
-0.916290731874155, -0.5108256237659907, -0.35667494393873245`, \
-0.2231435513142097, -0.10536051565782628`, 0.6931471805599453, 
       1.0986122886681098`, 1.3862943611198906`, 1.791759469228055, 
       1.9459101490553132`, 2.0794415416798357`, 2.1972245773362196`, 
       2.995732273553991, 3.4011973816621555`, 3.6888794541139363`, 
       4.0943445622221, 4.248495242049359, 4.382026634673881, 
       4.499809670330265, 5.298317366548036, 5.703782474656201, 
       5.991464547107982}, {-9.210340371976182, -4.605170185988091, 0, 
        4.605170185988092, -6.725433722188183, -6.074846156047033, \
-5.683979847360021, -5.403677882205863, -5.184988681241033, \
-5.005647752585217, -4.853631545286591, -4.721704002244043, \
-2.120263536200091, -1.4696759700589417`, -1.07880966137193, \
-0.7985076962177716, -0.579818495252942, -0.40047756659712525`, \
-0.2484613592984996, -0.11653381625595151`, 2.4849066497880004`, 
        3.1354942159291497`, 3.5263605246161616`, 3.8066624897703196`, 
        4.02535169073515, 4.204692619390966, 4.356708826689592, 
        4.48863636973214}},
     ImageSize->600,
     Method->{},
     PlotRange->{{-2.302585092994045, 
      3.9120230054281455`}, {-9.331263923106688, 6.7321277725201245`}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], 
       Scaled[0.02]},
     Ticks->{{{-2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-0.6931471805599453, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {0., 
         FormBox[
          TagBox[
           InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {1.6094379124341003`, 
         FormBox[
          TagBox[
           InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {3.912023005428146, 
         FormBox[
          TagBox[
           InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {4.605170185988092, 
         FormBox[
          TagBox[
           InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {6.214608098422191, 
         FormBox[
          TagBox[
           InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-1.6094379124341003`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.203972804325936, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.916290731874155, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.5108256237659907, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.35667494393873245`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.2231435513142097, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.10536051565782628`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {0.6931471805599453, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.0986122886681098`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.3862943611198906`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.791759469228055, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.9459101490553132`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.0794415416798357`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.1972245773362196`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.995732273553991, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.4011973816621555`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.6888794541139363`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.0943445622221, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.248495242049359, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.382026634673881, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.499809670330265, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.298317366548036, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.703782474656201, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.991464547107982, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}, {{-9.210340371976182, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "4"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-4.605170185988091, 
         FormBox["0.01`", TraditionalForm]}, {0, 
         FormBox["1", TraditionalForm]}, {4.605170185988092, 
         FormBox["100", TraditionalForm]}, {-6.725433722188183, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-6.074846156047033, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.683979847360021, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.403677882205863, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.184988681241033, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.005647752585217, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-4.853631545286591, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-4.721704002244043, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-2.120263536200091, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.4696759700589417`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.07880966137193, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.7985076962177716, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.579818495252942, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.40047756659712525`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.2484613592984996, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.11653381625595151`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.4849066497880004`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.1354942159291497`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.5263605246161616`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.8066624897703196`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.02535169073515, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.204692619390966, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.356708826689592, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.48863636973214, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], GraphicsGroupBox[{
     {GrayLevel[0], RectangleBox[{0.35, 0.3}, {0.78, 0.45999999999999996`}]}, 
     {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
      GraphicsGroupBox[{
        RectangleBox[{0.35, 0.3}, {0.78, 0.45999999999999996`}], InsetBox[
         GraphicsBox[{{{InsetBox[
              GraphicsBox[{
                {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 0}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"pitchTM/zg\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[1, 2]}, {1.05, 0.5}], {-1, 0},
               Automatic, {1, 0}]}, {InsetBox[
              GraphicsBox[{
                {Hue[0.67, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 1}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"zTM/zg\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[3, 2]}, {1.05, 1.5}], {-1, 0},
               Automatic, {1, 0}]}}, {}},
          AspectRatio->0.372093023255814,
          FormatType->TraditionalForm,
          PlotRange->{{-0.1, 3.6}, {-0.1, 2.1}}], {0.35, 0.3}, {
         Left, Bottom}, {0.43, 0.16}]}]}}]},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->600.,
  PlotRange->All]], "Output",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511161422990941*^9, 3.5111614478109827`*^9}, 
   3.511230657179332*^9, 3.511230687973786*^9, 3.511231850987028*^9, 
   3.5112321855830207`*^9, 3.5112326978099203`*^9, 3.5112327732360525`*^9, {
   3.5112328563529987`*^9, 3.5112328659626155`*^9}, 3.5112329164599047`*^9, 
   3.5112329747416067`*^9, {3.5112330127276735`*^9, 3.5112330177040825`*^9}, {
   3.5112330645665646`*^9, 3.511233088918207*^9}, {3.5112332403788733`*^9, 
   3.511233264777316*^9}, 3.511330866367856*^9, 3.5113313123882394`*^9, 
   3.511390364321541*^9, 3.5113904650821176`*^9, {3.511390495814172*^9, 
   3.5113905162346077`*^9}, 3.511418367785653*^9, 3.5114185710696096`*^9, 
   3.511431634843893*^9, 3.512730750000399*^9, 3.5127333926310434`*^9, {
   3.512735394317359*^9, 3.5127354208530054`*^9}, 3.512775202353125*^9, 
   3.5127769673106284`*^9, 3.5181496162362823`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Transfer function from force to disp.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5109750603967185`*^9, 3.5109750715975385`*^9}, {
  3.5114183861780853`*^9, 3.5114184005613103`*^9}}],

Cell[BoxData[
 RowBox[{"tfplotf", "[", 
  RowBox[{
   RowBox[{"{", "zTM", "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM"}], "}"}], ",", "freq"}], "]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.510957748603794*^9, 3.5109577777446456`*^9}, {
   3.510957947894144*^9, 3.510957950312149*^9}, 3.5109588754873734`*^9, {
   3.510960251768591*^9, 3.5109602568386*^9}, {3.5109644688037157`*^9, 
   3.5109645325454273`*^9}, {3.5109646318084016`*^9, 3.510964636956411*^9}, {
   3.5109652306946535`*^9, 3.510965232036256*^9}, {3.5109653621872845`*^9, 
   3.510965374776507*^9}, {3.510965421685789*^9, 3.510965427161399*^9}, {
   3.5109750765583467`*^9, 3.5109750893035693`*^9}, {3.5109828951875024`*^9, 
   3.510982897106306*^9}, {3.511002574321229*^9, 3.511002578689237*^9}, {
   3.511147126083946*^9, 3.511147126583147*^9}, {3.51133135800272*^9, 
   3.5113313625891275`*^9}, 3.5114184005613103`*^9, {3.511431802278987*^9, 
   3.511431817223813*^9}, {3.5114319322428155`*^9, 3.511431934083619*^9}, {
   3.511494423146207*^9, 3.511494426297412*^9}, {3.5114944629106765`*^9, 
   3.511494463940278*^9}, {3.511494528649192*^9, 3.511494529819194*^9}, {
   3.511494611407337*^9, 3.5114946157285447`*^9}, {3.5114947070511055`*^9, 
   3.51149470978111*^9}, {3.511495153274289*^9, 3.511495157252296*^9}, {
   3.511495906471814*^9, 3.5114959082502174`*^9}, 3.511495988091158*^9, {
   3.5114960625188885`*^9, 3.5114960693985004`*^9}, {3.5114976579860983`*^9, 
   3.5114976911673565`*^9}, {3.511497813627572*^9, 3.511497817184378*^9}, {
   3.5123560718342876`*^9, 3.512356077871499*^9}, {3.5123561129091606`*^9, 
   3.512356116699967*^9}, {3.5123595749764414`*^9, 3.51235961417931*^9}, 
   3.512359690713044*^9, {3.512423733837125*^9, 3.5124237421987395`*^9}, {
   3.512728016650796*^9, 3.5127280238736086`*^9}, {3.512776980352251*^9, 
   3.5127769804926515`*^9}}]
}, Open  ]],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[Large], LineBox[CompressedData["
1:eJwU2nc8Vf8fB3BE2ePec+2RiEiIviXrCBkJUSGVSIMoISkUyarQUER2KisZ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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
       LineBox[CompressedData["
1:eJwU2nc81d8fB3CrSFmFkLj3ujeU+mYkWUcoScgIUUYapGGEjEoyixLKKiNK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        "]]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{-2.302585092994045, -23.201307808249528`},
     BaseStyle->{FontSize -> Scaled[0.04]},
     CoordinatesToolOptions:>{"DisplayFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& ), "CopiedValueFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& )},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     Frame->True,
     FrameLabel->{
       FormBox["\"Frequency [Hz]\"", TraditionalForm]},
     FrameTicks->{{{{-23.025850929940457`, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "10"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-18.420680743952367`, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "8"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-13.815510557964274`, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "6"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-9.210340371976182, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "4"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-4.605170185988091, 
          FormBox["0.01`", TraditionalForm]}, {0, 
          FormBox["1", TraditionalForm]}, {-20.540944280152456`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.890356714011308`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.499490405324295`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.219188440170136`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.000499239205308`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.821158310549492`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.669142103250866`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.537214560208316`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-15.935774094164366`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-15.285186528023216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.894320219336205`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.614018254182046`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.395329053217216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.215988124561399`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.063971917262773`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-13.932044374220226`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.330603908176274`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.680016342035124`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.289150033348113`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.008848068193954`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.790158867229126, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.610817938573309, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.458801731274683, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.326874188232134, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.725433722188183, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.074846156047033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.683979847360021, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.403677882205863, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.184988681241033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.005647752585217, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.853631545286591, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.721704002244043, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.120263536200091, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.4696759700589417`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.07880966137193, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.7985076962177716, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.579818495252942, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.40047756659712525`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2484613592984996, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.11653381625595151`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-23.025850929940457`, 
          FormBox["\"\"", TraditionalForm]}, {-18.420680743952367`, 
          FormBox["\"\"", TraditionalForm]}, {-13.815510557964274`, 
          FormBox["\"\"", TraditionalForm]}, {-9.210340371976182, 
          FormBox["\"\"", TraditionalForm]}, {-4.605170185988091, 
          FormBox["\"\"", TraditionalForm]}, {0, 
          FormBox["\"\"", TraditionalForm]}, {-20.540944280152456`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.890356714011308`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.499490405324295`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.219188440170136`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-19.000499239205308`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.821158310549492`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.669142103250866`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.537214560208316`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-15.935774094164366`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-15.285186528023216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.894320219336205`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.614018254182046`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.395329053217216`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.215988124561399`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-14.063971917262773`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-13.932044374220226`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.330603908176274`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.680016342035124`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.289150033348113`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-10.008848068193954`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.790158867229126, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.610817938573309, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.458801731274683, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.326874188232134, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.725433722188183, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-6.074846156047033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.683979847360021, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.403677882205863, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.184988681241033, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-5.005647752585217, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.853631545286591, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.721704002244043, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.120263536200091, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.4696759700589417`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.07880966137193, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.7985076962177716, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.579818495252942, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.40047756659712525`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2484613592984996, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.11653381625595151`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}, {{{-2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-0.6931471805599453, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {0., 
          FormBox[
           TagBox[
            InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {1.6094379124341003`, 
          FormBox[
           TagBox[
            InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {3.912023005428146, 
          FormBox[
           TagBox[
            InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {4.605170185988092, 
          FormBox[
           TagBox[
            InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {6.214608098422191, 
          FormBox[
           TagBox[
            InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {-0.6931471805599453, 
          FormBox["\"\"", TraditionalForm]}, {0., 
          FormBox["\"\"", TraditionalForm]}, {1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm]}, {2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {3.912023005428146, 
          FormBox["\"\"", TraditionalForm]}, {4.605170185988092, 
          FormBox["\"\"", TraditionalForm]}, {6.214608098422191, 
          FormBox["\"\"", TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}},
     GridLines->{{-2.302585092994046, -0.6931471805599453, 0., 
       1.6094379124341003`, 2.302585092994046, 3.912023005428146, 
       4.605170185988092, 
       6.214608098422191, -1.6094379124341003`, -1.203972804325936, \
-0.916290731874155, -0.5108256237659907, -0.35667494393873245`, \
-0.2231435513142097, -0.10536051565782628`, 0.6931471805599453, 
       1.0986122886681098`, 1.3862943611198906`, 1.791759469228055, 
       1.9459101490553132`, 2.0794415416798357`, 2.1972245773362196`, 
       2.995732273553991, 3.4011973816621555`, 3.6888794541139363`, 
       4.0943445622221, 4.248495242049359, 4.382026634673881, 
       4.499809670330265, 5.298317366548036, 5.703782474656201, 
       5.991464547107982}, {-23.025850929940457`, -18.420680743952367`, \
-13.815510557964274`, -9.210340371976182, -4.605170185988091, 
        0, -20.540944280152456`, -19.890356714011308`, -19.499490405324295`, \
-19.219188440170136`, -19.000499239205308`, -18.821158310549492`, \
-18.669142103250866`, -18.537214560208316`, -15.935774094164366`, \
-15.285186528023216`, -14.894320219336205`, -14.614018254182046`, \
-14.395329053217216`, -14.215988124561399`, -14.063971917262773`, \
-13.932044374220226`, -11.330603908176274`, -10.680016342035124`, \
-10.289150033348113`, -10.008848068193954`, -9.790158867229126, \
-9.610817938573309, -9.458801731274683, -9.326874188232134, \
-6.725433722188183, -6.074846156047033, -5.683979847360021, \
-5.403677882205863, -5.184988681241033, -5.005647752585217, \
-4.853631545286591, -4.721704002244043, -2.120263536200091, \
-1.4696759700589417`, -1.07880966137193, -0.7985076962177716, \
-0.579818495252942, -0.40047756659712525`, -0.2484613592984996, \
-0.11653381625595151`}},
     ImageSize->600,
     Method->{},
     PlotRange->{{-2.302585092994045, 
      3.9120230054281455`}, {-23.201307808249528`, 1.382959607460855}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], 
       Scaled[0.02]},
     Ticks->{{{-2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-0.6931471805599453, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {0., 
         FormBox[
          TagBox[
           InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {1.6094379124341003`, 
         FormBox[
          TagBox[
           InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {3.912023005428146, 
         FormBox[
          TagBox[
           InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {4.605170185988092, 
         FormBox[
          TagBox[
           InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {6.214608098422191, 
         FormBox[
          TagBox[
           InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-1.6094379124341003`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.203972804325936, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.916290731874155, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.5108256237659907, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.35667494393873245`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.2231435513142097, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.10536051565782628`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {0.6931471805599453, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.0986122886681098`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.3862943611198906`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.791759469228055, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.9459101490553132`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.0794415416798357`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.1972245773362196`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.995732273553991, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.4011973816621555`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.6888794541139363`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.0943445622221, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.248495242049359, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.382026634673881, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.499809670330265, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.298317366548036, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.703782474656201, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.991464547107982, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}, {{-23.025850929940457`, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "10"}]}, "Superscript", SyntaxForm -> 
           SuperscriptBox], TraditionalForm]}, {-18.420680743952367`, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "8"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-13.815510557964274`, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "6"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-9.210340371976182, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "4"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-4.605170185988091, 
         FormBox["0.01`", TraditionalForm]}, {0, 
         FormBox["1", TraditionalForm]}, {-20.540944280152456`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-19.890356714011308`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-19.499490405324295`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-19.219188440170136`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-19.000499239205308`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-18.821158310549492`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-18.669142103250866`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-18.537214560208316`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-15.935774094164366`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-15.285186528023216`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.894320219336205`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.614018254182046`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.395329053217216`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.215988124561399`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-14.063971917262773`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-13.932044374220226`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-11.330603908176274`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-10.680016342035124`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-10.289150033348113`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-10.008848068193954`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.790158867229126, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.610817938573309, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.458801731274683, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.326874188232134, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-6.725433722188183, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-6.074846156047033, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.683979847360021, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.403677882205863, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.184988681241033, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-5.005647752585217, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-4.853631545286591, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-4.721704002244043, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-2.120263536200091, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.4696759700589417`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.07880966137193, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.7985076962177716, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.579818495252942, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.40047756659712525`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.2484613592984996, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.11653381625595151`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], GraphicsGroupBox[{
     {GrayLevel[0], RectangleBox[{0.35, 0.3}, {0.85, 0.45999999999999996`}]}, 
     {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
      GraphicsGroupBox[{
        RectangleBox[{0.35, 0.3}, {0.85, 0.45999999999999996`}], InsetBox[
         GraphicsBox[{{{InsetBox[
              GraphicsBox[{
                {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 0}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"pitchTM/FzTM\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[1, 2]}, {1.05, 0.5}], {-1, 0},
               Automatic, {1, 0}]}, {InsetBox[
              GraphicsBox[{
                {Hue[0.67, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 1}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"zTM/FzTM\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[3, 2]}, {1.05, 1.5}], {-1, 0},
               Automatic, {1, 0}]}}, {}},
          AspectRatio->0.32,
          FormatType->TraditionalForm,
          PlotRange->{{-0.1, 3.6}, {-0.1, 2.1}}], {0.35, 0.3}, {
         Left, Bottom}, {0.5, 0.16}]}]}}]},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->600.,
  PlotRange->All]], "Output",
 CellChangeTimes->{3.5127769819122543`*^9, 3.5181496174998846`*^9}],

Cell[CellGroupData[{

Cell["Transfer function from actuator to disp.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.510969554959849*^9, 3.510969560841059*^9}, {
   3.5109751070876007`*^9, 3.5109751100048056`*^9}, {3.5114184059745197`*^9, 
   3.511418444678188*^9}, 3.5127279886643467`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"ca1", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"+", "0.25"}]}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"ca2", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"+", "0.25"}], ",", 
     RowBox[{"-", "0.25"}], ",", 
     RowBox[{"-", "0.25"}], ",", 
     RowBox[{"+", "0.25"}]}], "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"aTMlist", "=", 
   RowBox[{"{", 
    RowBox[{"aTM1", ",", "aTM2", ",", "aTM3", ",", "aTM4"}], "}"}]}], 
  ";"}]}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.512727879666955*^9, 3.5127279886643467`*^9}}],

Cell[BoxData[
 RowBox[{"tfplota", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"ca1", ".", "aTMlist"}], ",", 
     RowBox[{"ca2", ".", "aTMlist"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM"}], "}"}], ",", "freq"}], "]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.510974630101163*^9, 3.510974667307228*^9}, {
   3.5109751707513123`*^9, 3.510975230827018*^9}, {3.510975278219901*^9, 
   3.510975296534333*^9}, 3.510975617333297*^9, 3.5109762079503345`*^9, {
   3.510976711909239*^9, 3.5109767681005373`*^9}, {3.510981275918256*^9, 
   3.510981277462659*^9}, {3.5109815367911143`*^9, 3.510981543436726*^9}, {
   3.5110039773252935`*^9, 3.5110039774500937`*^9}, {3.5111476837225256`*^9, 
   3.511147683909726*^9}, {3.511149615115118*^9, 3.5111496153023186`*^9}, {
   3.5111518249681997`*^9, 3.5111518251085997`*^9}, {3.511155202125937*^9, 
   3.511155202281937*^9}, {3.511155261842842*^9, 3.511155266881651*^9}, {
   3.51133147809173*^9, 3.511331483863741*^9}, {3.5113317639466324`*^9, 
   3.511331775989854*^9}, {3.5113321818401666`*^9, 3.5113321940861883`*^9}, {
   3.511332324533617*^9, 3.511332347590458*^9}, {3.511332426900997*^9, 
   3.511332473513879*^9}, {3.511332686797454*^9, 3.51133270737389*^9}, {
   3.511332765078391*^9, 3.511332767293595*^9}, {3.511391414234585*^9, 
   3.511391414983386*^9}, {3.511418452665402*^9, 3.511418453414203*^9}, {
   3.51141849147827*^9, 3.5114185140983095`*^9}, {3.511418638882929*^9, 
   3.51141863970973*^9}, {3.5114187268670835`*^9, 3.511418727069884*^9}, {
   3.511418809126028*^9, 3.5114188156312394`*^9}, {3.5114318640550957`*^9, 
   3.5114318802947245`*^9}, {3.5127278701353383`*^9, 3.512727873208544*^9}, {
   3.5127279545158863`*^9, 3.5127279886643467`*^9}}],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[Large], LineBox[CompressedData["
1:eJwU2nc8Vf8fB3BE2ePec+2RiEiIviXrCBkJUSGVSIMoISkUyarQUER2KisZ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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
       LineBox[CompressedData["
1:eJwU2nc81d8fB3CrSFmFkLj3ujeU+mYkWUcoScgIUUYapGGEjEoyixLKKiNK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        "]]}, 
      {Hue[0.14213595499957954`, 0.6, 0.6], Thickness[Large], 
       LineBox[CompressedData["
1:eJwU2nc8V98fB3CUjKxUKnxtn/sZWjStK7OF7HZGw6gUklBGGSW0ZKQQZYSM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        "]]}, 
      {Hue[0.37820393249936934`, 0.6, 0.6], Thickness[Large], 
       LineBox[CompressedData["
1:eJwU2ndcjt8bB/CKSEvzeRTfFIooGoTWiVJIJVEUaaGEVCoaSloaRKhol7Y0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        "]]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{-2.302585092994045, -26.014718525009567`},
     BaseStyle->{FontSize -> Scaled[0.04]},
     CoordinatesToolOptions:>{"DisplayFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& ), "CopiedValueFunction" -> ({
         Exp[
          Part[#, 1]], 
         Exp[
          Part[#, 2]]}& )},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     Frame->True,
     FrameLabel->{
       FormBox["\"Frequency [Hz]\"", TraditionalForm]},
     FrameTicks->{{{{-25.328436022934504`, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "11"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-20.72326583694641, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "9"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-16.11809565095832, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "7"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-11.512925464970229`, 
          FormBox[
           TemplateBox[{"10", 
             RowBox[{"-", "5"}]}, "Superscript", SyntaxForm -> 
            SuperscriptBox], TraditionalForm]}, {-6.907755278982137, 
          FormBox["0.001`", TraditionalForm]}, {-2.3025850929940455`, 
          FormBox["0.1`", TraditionalForm]}, {2.302585092994046, 
          FormBox["10", TraditionalForm]}, {-22.843529373146502`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-22.192941807005354`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.80207549831834, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.521773533164183`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.303084332199354`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.123743403543536`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-20.97172719624491, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-20.839799653202363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.238359187158412`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-17.58777162101726, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-17.196905312330248`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.916603347176093`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.69791414621126, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.518573217555446`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.36655701025682, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.234629467214273`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-13.633189001170319`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.98260143502917, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.591735126342158`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.311433161188, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.09274396022317, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.913403031567354`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.761386824268728`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.62945928122618, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.028018815182229, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-8.377431249041079, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.986564940354067, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.706262975199909, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.487573774235079, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.308232845579262, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.156216638280637, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.024289095238088, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.422848629194137, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-3.7722610630529876`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-3.3813947543659757`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-3.101092789211817, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.882403588246988, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.703062659591171, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.551046452292545, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.419118909249997, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.1823215567939546, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.8329091229351039, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.2237754316221159`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.5040773967762742`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.7227665977411037`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9021075263969205`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0541237336955462`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.186051276738094, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-25.328436022934504`, 
          FormBox["\"\"", TraditionalForm]}, {-20.72326583694641, 
          FormBox["\"\"", TraditionalForm]}, {-16.11809565095832, 
          FormBox["\"\"", TraditionalForm]}, {-11.512925464970229`, 
          FormBox["\"\"", TraditionalForm]}, {-6.907755278982137, 
          FormBox["\"\"", TraditionalForm]}, {-2.3025850929940455`, 
          FormBox["\"\"", TraditionalForm]}, {2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {-22.843529373146502`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-22.192941807005354`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.80207549831834, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.521773533164183`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.303084332199354`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-21.123743403543536`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-20.97172719624491, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-20.839799653202363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-18.238359187158412`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-17.58777162101726, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-17.196905312330248`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.916603347176093`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.69791414621126, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.518573217555446`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.36655701025682, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-16.234629467214273`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-13.633189001170319`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.98260143502917, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.591735126342158`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.311433161188, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-12.09274396022317, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.913403031567354`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.761386824268728`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-11.62945928122618, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-9.028018815182229, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-8.377431249041079, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.986564940354067, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.706262975199909, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.487573774235079, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.308232845579262, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.156216638280637, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-7.024289095238088, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-4.422848629194137, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-3.7722610630529876`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-3.3813947543659757`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-3.101092789211817, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.882403588246988, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.703062659591171, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.551046452292545, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-2.419118909249997, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.1823215567939546, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.8329091229351039, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.2237754316221159`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.5040773967762742`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.7227665977411037`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9021075263969205`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0541237336955462`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.186051276738094, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}, {{{-2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-0.6931471805599453, 
          FormBox[
           TagBox[
            InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {0., 
          FormBox[
           TagBox[
            InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {1.6094379124341003`, 
          FormBox[
           TagBox[
            InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {2.302585092994046, 
          FormBox[
           TagBox[
            InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {3.912023005428146, 
          FormBox[
           TagBox[
            InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {4.605170185988092, 
          FormBox[
           TagBox[
            InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {6.214608098422191, 
          FormBox[
           TagBox[
            InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
            NumberForm[#, {
              DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
           TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}, {{-2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {-0.6931471805599453, 
          FormBox["\"\"", TraditionalForm]}, {0., 
          FormBox["\"\"", TraditionalForm]}, {1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm]}, {2.302585092994046, 
          FormBox["\"\"", TraditionalForm]}, {3.912023005428146, 
          FormBox["\"\"", TraditionalForm]}, {4.605170185988092, 
          FormBox["\"\"", TraditionalForm]}, {6.214608098422191, 
          FormBox["\"\"", TraditionalForm]}, {-1.6094379124341003`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-1.203972804325936, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.916290731874155, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.5108256237659907, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.35667494393873245`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.2231435513142097, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {-0.10536051565782628`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {0.6931471805599453, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.0986122886681098`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.3862943611198906`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.791759469228055, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {1.9459101490553132`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.0794415416798357`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.1972245773362196`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {2.995732273553991, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.4011973816621555`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {3.6888794541139363`, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.0943445622221, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.248495242049359, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.382026634673881, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {4.499809670330265, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.298317366548036, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.703782474656201, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}, {5.991464547107982, 
          FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
           Thickness[0.001]}}}}},
     GridLines->{{-2.302585092994046, -0.6931471805599453, 0., 
       1.6094379124341003`, 2.302585092994046, 3.912023005428146, 
       4.605170185988092, 
       6.214608098422191, -1.6094379124341003`, -1.203972804325936, \
-0.916290731874155, -0.5108256237659907, -0.35667494393873245`, \
-0.2231435513142097, -0.10536051565782628`, 0.6931471805599453, 
       1.0986122886681098`, 1.3862943611198906`, 1.791759469228055, 
       1.9459101490553132`, 2.0794415416798357`, 2.1972245773362196`, 
       2.995732273553991, 3.4011973816621555`, 3.6888794541139363`, 
       4.0943445622221, 4.248495242049359, 4.382026634673881, 
       4.499809670330265, 5.298317366548036, 5.703782474656201, 
       5.991464547107982}, CompressedData["
1:eJwBxQE6/iFib1JlAQAAADcAAAD+VBliFFQ5wLj/K/MnuTTAc6o+hDseMMBb
qqIqngYnwKD/j5mKoRvAFVW1u7FrAsAWVbW7sWsCQC7hforx1zbAgB5fomQx
NsA/M+LRVM01wMnbRPOShTXA6rlO75ZNNcCyUM+lrR81wHOBEB3D+DTAQLIt
HP3WNMDpi5EbBT0ywDrJcTN4ljHA+d30YmgyMcCEhleEpuowwKRkYYCqsjDA
bfvhNsGEMMAuLCOu1l0wwPtcQK0QPDDARm1IWTFEK8Dq5wiJF/cpwGgRD+j3
LinAfWLUKnSfKMC+HugifC8owE9M6Y+p0yfA0a1rftSFJ8BrD6Z8SEInwLzC
bXtYDiLAXz0uqz7BIMC7zWgUPvIfwORv85k20x7AZugaikbzHcCIQx1koTsd
wI0GIkH3nxzAv8mWPd8YHMBiMCY7/7ARwFFLTjWXLQ7AS/FmsRgNC8CcNXy8
Cc8IwKImy5wpDwfA5tzPUN+fBcDuYtkKi2gEwFTpwgNbWgPAdZbkEVBWxz9p
e3EIMafqP8FxB4yVlPM/HuncdbMQ+D8UBz+1c5D7P4uaNU0Ib/4/PUeRbNhu
AEDXwKdzCH0BQMGQ05U=
       "]},
     ImageSize->600,
     Method->{},
     PlotRange->{{-2.302585092994045, 
      3.9120230054281455`}, {-26.014718525009567`, 1.382959607460855}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], 
       Scaled[0.02]},
     Ticks->{{{-2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-0.6931471805599453, 
         FormBox[
          TagBox[
           InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {0., 
         FormBox[
          TagBox[
           InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {1.6094379124341003`, 
         FormBox[
          TagBox[
           InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {2.302585092994046, 
         FormBox[
          TagBox[
           InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {3.912023005428146, 
         FormBox[
          TagBox[
           InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {4.605170185988092, 
         FormBox[
          TagBox[
           InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {6.214608098422191, 
         FormBox[
          TagBox[
           InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
           NumberForm[#, {
             DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
          TraditionalForm]}, {-1.6094379124341003`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-1.203972804325936, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.916290731874155, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.5108256237659907, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.35667494393873245`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.2231435513142097, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-0.10536051565782628`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {0.6931471805599453, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.0986122886681098`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.3862943611198906`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.791759469228055, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.9459101490553132`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.0794415416798357`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.1972245773362196`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.995732273553991, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.4011973816621555`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {3.6888794541139363`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.0943445622221, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.248495242049359, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.382026634673881, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {4.499809670330265, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.298317366548036, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.703782474656201, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {5.991464547107982, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}, {{-25.328436022934504`, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "11"}]}, "Superscript", SyntaxForm -> 
           SuperscriptBox], TraditionalForm]}, {-20.72326583694641, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "9"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-16.11809565095832, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "7"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-11.512925464970229`, 
         FormBox[
          TemplateBox[{"10", 
            RowBox[{"-", "5"}]}, "Superscript", SyntaxForm -> SuperscriptBox],
           TraditionalForm]}, {-6.907755278982137, 
         FormBox["0.001`", TraditionalForm]}, {-2.3025850929940455`, 
         FormBox["0.1`", TraditionalForm]}, {2.302585092994046, 
         FormBox["10", TraditionalForm]}, {-22.843529373146502`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-22.192941807005354`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-21.80207549831834, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-21.521773533164183`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-21.303084332199354`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-21.123743403543536`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-20.97172719624491, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-20.839799653202363`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-18.238359187158412`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-17.58777162101726, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-17.196905312330248`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-16.916603347176093`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-16.69791414621126, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-16.518573217555446`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-16.36655701025682, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-16.234629467214273`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-13.633189001170319`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-12.98260143502917, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-12.591735126342158`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-12.311433161188, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-12.09274396022317, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-11.913403031567354`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-11.761386824268728`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-11.62945928122618, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-9.028018815182229, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-8.377431249041079, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-7.986564940354067, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-7.706262975199909, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-7.487573774235079, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-7.308232845579262, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-7.156216638280637, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-7.024289095238088, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-4.422848629194137, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-3.7722610630529876`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-3.3813947543659757`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-3.101092789211817, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-2.882403588246988, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-2.703062659591171, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-2.551046452292545, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {-2.419118909249997, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {0.1823215567939546, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {0.8329091229351039, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.2237754316221159`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.5040773967762742`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.7227665977411037`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {1.9021075263969205`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.0541237336955462`, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}, {2.186051276738094, 
         FormBox["\"\"", TraditionalForm], {0.00375, 0.}, {
          Thickness[0.001]}}}}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], GraphicsGroupBox[{
     {GrayLevel[0], RectangleBox[{0.35, 0.3}, {0.85, 0.62}]}, 
     {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
      GraphicsGroupBox[{RectangleBox[{0.35, 0.3}, {0.85, 0.62}], InsetBox[
         GraphicsBox[{{{InsetBox[
              GraphicsBox[{
                {Hue[0.37820393249936846`, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 0}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"pitchTM/Act2\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[1, 2]}, {1.05, 0.5}], {-1, 0},
               Automatic, {1, 0}]}, {InsetBox[
              GraphicsBox[{
                {Hue[0.1421359549995791, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 1}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"zTM/Act2\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[3, 2]}, {1.05, 1.5}], {-1, 0},
               Automatic, {1, 0}]}, {InsetBox[
              GraphicsBox[{
                {Hue[0.9060679774997897, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 2}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"pitchTM/Act1\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[5, 2]}, {1.05, 2.5}], {-1, 0},
               Automatic, {1, 0}]}, {InsetBox[
              GraphicsBox[{
                {Hue[0.67, 0.6, 0.6], Thickness[Large], 
                 LineBox[{{0, 0.5}, {1, 0.5}}]}, {}},
               PlotRange->{{0, 1}, {0, 1}}], {0, 3}, {Left, Bottom}, {1, 1}], 
             InsetBox[
              StyleBox["\<\"zTM/Act1\"\>",
               StripOnInput->False,
               FontSize->14], NCache[{1.05, Rational[7, 2]}, {1.05, 3.5}], {-1, 0},
               Automatic, {1, 0}]}}, {}},
          AspectRatio->0.64,
          FormatType->TraditionalForm,
          PlotRange->{{-0.1, 3.6}, {-0.1, 4.1}}], {0.35, 0.3}, {
         Left, Bottom}, {0.5, 0.32}]}]}}]},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->600.,
  PlotRange->All]], "Output",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{3.512730783259658*^9, 3.5127752121499424`*^9, 
  3.512775611606246*^9, 3.5127769685586305`*^9, 3.5181496192938876`*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["State-Space Model", "Section",
 CellChangeTimes->{{3.511335807091339*^9, 3.5113358106481447`*^9}, 
   3.511337659532192*^9, {3.512773070004176*^9, 3.5127730729993815`*^9}}],

Cell[CellGroupData[{

Cell["State-Space Matrices", "Subsection",
 CellChangeTimes->{{3.511338449601981*^9, 3.511338476356028*^9}}],

Cell[CellGroupData[{

Cell["Set Input", "Subsubsection",
 CellChangeTimes->{{3.511345101119687*^9, 3.5113451151285114`*^9}, 
   3.5113452765575953`*^9, {3.5113491519476023`*^9, 3.511349152649603*^9}}],

Cell[CellGroupData[{

Cell["Create parameter input vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, 
   3.5113919865527897`*^9, {3.5113932945782876`*^9, 3.511393300724698*^9}, 
   3.511393991963113*^9, {3.511394540569276*^9, 3.5113945443288827`*^9}, 
   3.512728164258255*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"inputp", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"inputvecp", "=", 
   RowBox[{"makeinputvecp", "[", "inputp", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, 3.511347728554302*^9, {
   3.51135156339744*^9, 3.5113515638186407`*^9}, 3.5113919865527897`*^9, {
   3.5113923432630167`*^9, 3.511392343497017*^9}, {3.511393282316666*^9, 
   3.5113932878546762`*^9}, 3.5113933213323345`*^9, 3.511393991963113*^9, {
   3.5127281565986414`*^9, 3.5127281605922484`*^9}, {3.5127308942850523`*^9, 
   3.51273090969788*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create force input vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, 
   3.5113919865527897`*^9, {3.5113932945782876`*^9, 3.511393313891122*^9}, 
   3.5113939948803177`*^9, {3.5113945468248873`*^9, 3.5113945475892887`*^9}, 
   3.5127281856926928`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"inputf", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"inputvecf", "=", 
   RowBox[{"makeinputvecf", "[", "inputf", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, 3.511347728554302*^9, {
   3.51135156339744*^9, 3.5113515638186407`*^9}, 3.5113919865527897`*^9, {
   3.5113923432630167`*^9, 3.511392343497017*^9}, {3.511393282316666*^9, 
   3.5113932878546762`*^9}, {3.5113933227831373`*^9, 3.511393333874757*^9}, 
   3.5113939948803177`*^9, {3.511419195257906*^9, 3.5114191953359065`*^9}, 
   3.51143273690463*^9, {3.5127281700770655`*^9, 3.5127281744762726`*^9}, {
   3.5127309033018684`*^9, 3.512730912833485*^9}, {3.512775346247778*^9, 
   3.512775358103799*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create actuator input vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, 
   3.5113919865527897`*^9, {3.5113932945782876`*^9, 3.511393346042778*^9}, {
   3.511394549476892*^9, 3.5113945509276943`*^9}, {3.5127281815742855`*^9, 
   3.5127281824634867`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"inputa", "=", 
   RowBox[{"{", 
    RowBox[{"aTM1", ",", "aTM2", ",", "aTM3", ",", "aTM4"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"inputveca", "=", 
   RowBox[{"makeinputveca", "[", "inputa", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, 3.511347728554302*^9, {
   3.51135156339744*^9, 3.5113515638186407`*^9}, 3.5113919865527897`*^9, {
   3.5113923432630167`*^9, 3.511392343497017*^9}, {3.511393282316666*^9, 
   3.5113932878546762`*^9}, {3.5113933227831373`*^9, 3.511393333874757*^9}, {
   3.5113934356805353`*^9, 3.511393441889346*^9}, {3.5113939599674563`*^9, 
   3.5113939625102606`*^9}, {3.5114327391510344`*^9, 
   3.5114327467638474`*^9}, {3.512728188204297*^9, 3.512728199093116*^9}, {
   3.5127309146586885`*^9, 3.51273092685791*^9}, {3.512775320070932*^9, 
   3.512775328604147*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Set Output", "Subsubsection",
 CellChangeTimes->{{3.5113471295288496`*^9, 3.511347130230851*^9}, {
  3.511349154038006*^9, 3.511349154693207*^9}}],

Cell[CellGroupData[{

Cell["Create output vector", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345666090279*^9, 3.5113456727358913`*^9}, 
   3.511346162498751*^9, {3.511346966477363*^9, 3.5113469772257824`*^9}, {
   3.511347126221644*^9, 3.5113471378280644`*^9}, {3.511394532363662*^9, 
   3.5113945368876696`*^9}, {3.512728214271943*^9, 3.5127282158943453`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"outputv", "=", 
   RowBox[{"{", 
    RowBox[{"zTM", ",", "pitchTM", ",", "yawTM"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"outputvecv", "=", 
   RowBox[{"makeoutputvecv", "[", "outputv", "]"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511345708522354*^9, 3.5113457533100324`*^9}, 
   3.5113458022629185`*^9, 3.511346162498751*^9, {3.5113470309210763`*^9, 
   3.511347050530311*^9}, 3.511347126221644*^9, {3.5113926520031586`*^9, 
   3.51139265832117*^9}, {3.5113944608687363`*^9, 3.511394475579562*^9}, 
   3.5114190053119726`*^9, {3.512728211682338*^9, 3.5127282259251633`*^9}, {
   3.5127309369199276`*^9, 3.512730975311595*^9}, {3.512775368119016*^9, 
   3.512775390692256*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["State-Space Matrices", "Subsubsection",
 CellChangeTimes->{{3.5113428440609226`*^9, 3.5113428707837696`*^9}, {
  3.5114035961001205`*^9, 3.511403604336935*^9}, {3.511404827987484*^9, 
  3.511404832542692*^9}}],

Cell[CellGroupData[{

Cell["\<\
Damping frequency (for use in converting structure damping to viscous damping)\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511404853665129*^9, 3.5114048575495358`*^9}, {
  3.5114049021188145`*^9, 3.5114049368912754`*^9}, {3.5114050434862623`*^9, 
  3.5114050447498646`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"fdamp", "=", 
    RowBox[{"0.03", "hertz"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511404881448778*^9, 3.5114049021188145`*^9}, {
  3.5114059906771326`*^9, 3.511405991878335*^9}, {3.511406097755721*^9, 
  3.5114061003297253`*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Create the \"A\" matrix. Structure damping is considered to be zero.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113491808076525`*^9}, {
  3.511349227935335*^9, 3.5113492291365376`*^9}, {3.5114048144466605`*^9, 
  3.51140481996907*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssAr", "=", 
    RowBox[{"ArrayFlatten", "[", "\[IndentingNewLine]", 
     RowBox[{"{", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"IdentityMatrix", "[", 
          RowBox[{"Length", "[", "allvars", "]"}], "]"}]}], 
        "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matKvv"}]}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matGvv"}]}]}], 
        "\[IndentingNewLine]", "}"}]}], "\[IndentingNewLine]", "}"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.},
 CellChangeTimes->{{3.5113426757990274`*^9, 3.5113426772342296`*^9}, {
   3.5113427437527466`*^9, 3.5113427464047513`*^9}, {3.511342836120509*^9, 
   3.511342836338909*^9}, {3.5113428792545843`*^9, 3.511342914588647*^9}, {
   3.5113433061181345`*^9, 3.5113433157745514`*^9}, {3.511343413555523*^9, 
   3.51134342885915*^9}, {3.511343479060038*^9, 3.5113435935642395`*^9}, {
   3.511343633001108*^9, 3.5113436752147827`*^9}, {3.5113438056778116`*^9, 
   3.5113438390618706`*^9}, 3.5113438807139435`*^9, 3.51134391846601*^9, {
   3.511347221849812*^9, 3.511347406320136*^9}, {3.511347441045797*^9, 
   3.5113474641026373`*^9}, 3.511347506503512*^9, 3.511349063230246*^9, 
   3.5113491808076525`*^9, {3.511352291731519*^9, 3.5113522926051207`*^9}, {
   3.5113527927143993`*^9, 3.5113528109664316`*^9}, {3.5113905926279416`*^9, 
   3.5113905982283516`*^9}, {3.5114050655291014`*^9, 3.511405068290306*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Create the \"A\" matrix. Structure damping is approximated by viscous damping \
such that the amount of phase lag is the same at damping frequency.\
\>", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113491808076525`*^9}, {
  3.511349227935335*^9, 3.5113492291365376`*^9}, {3.51140477454179*^9, 
  3.5114048023878393`*^9}, {3.5114048635711465`*^9, 3.5114048754583673`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssAa", "=", 
    RowBox[{"ArrayFlatten", "[", "\[IndentingNewLine]", 
     RowBox[{"{", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"IdentityMatrix", "[", 
          RowBox[{"Length", "[", "allvars", "]"}], "]"}]}], 
        "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matKvv"}]}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{
            RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matGvv"}]}], "-", 
          RowBox[{
           RowBox[{
            RowBox[{"Inverse", "[", "matMvv", "]"}], ".", 
            RowBox[{"Im", "[", "matKvvc", "]"}]}], "/", 
           RowBox[{"(", 
            RowBox[{"2", 
             RowBox[{"N", "[", "Pi", "]"}], "fdamp"}], ")"}]}]}]}], 
        "\[IndentingNewLine]", "}"}]}], "\[IndentingNewLine]", "}"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.5113426757990274`*^9, 3.5113426772342296`*^9}, {
   3.5113427437527466`*^9, 3.5113427464047513`*^9}, {3.511342836120509*^9, 
   3.511342836338909*^9}, {3.5113428792545843`*^9, 3.511342914588647*^9}, {
   3.5113433061181345`*^9, 3.5113433157745514`*^9}, {3.511343413555523*^9, 
   3.51134342885915*^9}, {3.511343479060038*^9, 3.5113435935642395`*^9}, {
   3.511343633001108*^9, 3.5113436752147827`*^9}, {3.5113438056778116`*^9, 
   3.5113438390618706`*^9}, 3.5113438807139435`*^9, 3.51134391846601*^9, {
   3.511347221849812*^9, 3.511347406320136*^9}, {3.511347441045797*^9, 
   3.5113474641026373`*^9}, 3.511347506503512*^9, 3.511349063230246*^9, 
   3.5113491808076525`*^9, {3.511352291731519*^9, 3.5113522926051207`*^9}, {
   3.5113527927143993`*^9, 3.5113528109664316`*^9}, {3.5113905926279416`*^9, 
   3.5113905982283516`*^9}, 3.5114048023878393`*^9, {3.511405008043*^9, 
   3.5114051058707724`*^9}, {3.5114054963638587`*^9, 3.511405506831477*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create the \"B\" matrix.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113492256733313`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssB", "=", 
    RowBox[{"ArrayFlatten", "[", "\[IndentingNewLine]", 
     RowBox[{"{", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "inputp", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "inputf", "]"}], "}"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Table", "[", 
          RowBox[{"0", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "allvars", "]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "inputa", "]"}], "}"}]}], "]"}]}], 
        "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "matKpv", ".", 
           "inputvecp"}]}], ",", "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "inputvecf"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"Inverse", "[", "matMvv", "]"}], ".", "inputveca"}]}], 
        "\[IndentingNewLine]", "}"}]}], "\[IndentingNewLine]", "}"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.},
 CellChangeTimes->{{3.511343921024414*^9, 3.5113439340036373`*^9}, {
   3.5113475427891755`*^9, 3.5113477094598684`*^9}, {3.5113478660841436`*^9, 
   3.5113479747539344`*^9}, {3.511348254119225*^9, 3.5113482681436496`*^9}, {
   3.5113485186332893`*^9, 3.51134851882049*^9}, 3.511349064587448*^9, 
   3.511349196797681*^9, {3.5113528177836437`*^9, 3.5113528253184566`*^9}, {
   3.5113529173430185`*^9, 3.5113529199950233`*^9}, {3.511392437830383*^9, 
   3.511392440373187*^9}, {3.5113925180769234`*^9, 3.511392522725732*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create the \"C\" matrix.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113492162977147`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssC", "=", 
    RowBox[{"Join", "[", "\[IndentingNewLine]", 
     RowBox[{"outputvecv", ",", "\[IndentingNewLine]", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "allvars", "]"}], "}"}]}], "]"}], ",", "2"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.},
 CellChangeTimes->{{3.511348273494459*^9, 3.5113482899836884`*^9}, {
   3.5113483981230783`*^9, 3.5113484770436163`*^9}, {3.5113485408165283`*^9, 
   3.511348569692179*^9}, 3.5113486013602347`*^9, {3.511348992686922*^9, 
   3.5113490063369465`*^9}, {3.511349044089012*^9, 3.5113490659446507`*^9}, 
   3.5113492071716986`*^9, {3.5113529435354643`*^9, 3.511352946218669*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Create the \"D\" matrix.", "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511349164942425*^9, 3.5113491808076525`*^9}, {
  3.5113492109469056`*^9, 3.5113492202133217`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ssD", "=", 
    RowBox[{"Join", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "inputp", "]"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "inputf", "]"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "outputv", "]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Length", "[", "inputa", "]"}], "}"}]}], "]"}], ",", "2"}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.},
 CellChangeTimes->{{3.511349068035054*^9, 3.511349108813526*^9}, {
  3.5113492202133217`*^9, 3.511349221913725*^9}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Export", "Subsection",
 CellChangeTimes->{{3.5113492434105625`*^9, 3.5113492506957755`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"matlabexport", "[", 
    RowBox[{"\"\<matlab/ss110426.m\>\"", ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssA\>\"", ",", "ssAa"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssB\>\"", ",", "ssB"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssC\>\"", ",", "ssC"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"\"\<ssD\>\"", ",", "ssD"}], "}"}]}], "]"}], ";"}], 
  "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.511349255204183*^9, 3.511349323360703*^9}, 
   3.5114051453544416`*^9, {3.5114052158381658`*^9, 3.5114052352194*^9}, 
   3.5114055594347696`*^9, 3.511418896018181*^9, 3.511431975798092*^9, {
   3.5114323267943096`*^9, 3.5114323285415125`*^9}, {3.51277540460748*^9, 
   3.512775405574682*^9}}]
}, Open  ]]
}, Closed]]
}, Open  ]]
},
WindowSize->{1008, 632},
WindowMargins->{{-6, Automatic}, {Automatic, -12}},
ShowSelection->True,
FrontEndVersion->"7.0 for Microsoft Windows (32-bit) (2008\:5e7412\:670810\
\:65e5)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "Info3518182010-7910842"->{
  Cell[66618, 1743, 355, 6, 70, "Print",
   CellTags->"Info3518182010-7910842"]},
 "Info3518182010-5560469"->{
  Cell[71721, 1871, 597, 9, 70, "Print",
   CellTags->"Info3518182010-5560469"]},
 "Info3518182010-4499236"->{
  Cell[79132, 2051, 575, 9, 70, "Print",
   CellTags->"Info3518182010-4499236"]},
 "Info3518182011-4076727"->{
  Cell[82464, 2127, 374, 6, 70, "Print",
   CellTags->"Info3518182011-4076727"]},
 "Info3518182014-3702807"->{
  Cell[157734, 3742, 284, 5, 44, "Print",
   CellTags->"Info3518182014-3702807"]},
 "Info3518182014-8479246"->{
  Cell[158247, 3755, 287, 5, 44, "Print",
   CellTags->"Info3518182014-8479246"]},
 "Info3518182014-8742825"->{
  Cell[158761, 3768, 315, 5, 67, "Print",
   CellTags->"Info3518182014-8742825"]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"Info3518182010-7910842", 612143, 11643},
 {"Info3518182010-5560469", 612254, 11646},
 {"Info3518182010-4499236", 612365, 11649},
 {"Info3518182011-4076727", 612476, 11652},
 {"Info3518182014-3702807", 612587, 11655},
 {"Info3518182014-8479246", 612699, 11658},
 {"Info3518182014-8742825", 612811, 11661}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[567, 22, 611, 11, 77, "Subtitle"],
Cell[CellGroupData[{
Cell[1203, 37, 172, 2, 71, "Section"],
Cell[CellGroupData[{
Cell[1400, 43, 157, 2, 36, "Subsection"],
Cell[1560, 47, 1275, 20, 65, "Text"]
}, Open  ]],
Cell[CellGroupData[{
Cell[2872, 72, 105, 1, 36, "Subsection"],
Cell[2980, 75, 346, 7, 47, "Text"]
}, Open  ]],
Cell[CellGroupData[{
Cell[3363, 87, 160, 2, 36, "Subsection"],
Cell[3526, 91, 552, 11, 83, "Text"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[4127, 108, 204, 3, 71, "Section"],
Cell[CellGroupData[{
Cell[4356, 115, 204, 3, 36, "Subsection"],
Cell[4563, 120, 337, 4, 29, "Text"],
Cell[4903, 126, 514, 9, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[5454, 140, 203, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[5682, 147, 265, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[5950, 155, 465, 10, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[6452, 170, 219, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[6674, 175, 306, 7, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[7029, 188, 110, 1, 36, "Subsection"],
Cell[7142, 191, 518, 9, 47, "Text"],
Cell[7663, 202, 279, 7, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[7979, 214, 205, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[8209, 221, 297, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[8509, 227, 302, 6, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[8848, 238, 242, 5, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[9093, 245, 447, 7, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9589, 258, 203, 3, 36, "Subsection"],
Cell[9795, 263, 417, 7, 29, "Text"],
Cell[10215, 272, 2734, 79, 155, "Input"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12998, 357, 300, 4, 41, "Section"],
Cell[CellGroupData[{
Cell[13323, 365, 107, 1, 36, "Subsection"],
Cell[13433, 368, 603, 14, 54, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[14073, 387, 109, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[14207, 392, 210, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[14420, 397, 785, 22, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[15242, 424, 184, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[15429, 429, 1448, 32, 74, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[16914, 466, 177, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[17094, 471, 1658, 35, 74, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[18789, 511, 203, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[18995, 516, 1507, 32, 74, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[20551, 554, 151, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[20727, 560, 136, 2, 27, "Subsubsection"],
Cell[20866, 564, 2160, 38, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[23075, 608, 95, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[23195, 613, 177, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[23397, 619, 236, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[23636, 624, 1885, 52, 232, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[25558, 681, 145, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[25706, 685, 2442, 61, 232, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28197, 752, 229, 3, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[28451, 759, 222, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[28676, 764, 748, 14, 83, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[29427, 780, 244, 6, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[29708, 791, 193, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[29904, 796, 879, 17, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[30820, 818, 306, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[31129, 824, 1256, 29, 98, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[32422, 858, 169, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[32594, 863, 1576, 38, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[34207, 906, 223, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}],
Cell[34433, 911, 1087, 28, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10004.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[35569, 945, 138, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[35732, 951, 171, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[35906, 956, 412, 9, 65, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[36355, 970, 258, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[36616, 975, 322, 9, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[36975, 989, 245, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[37223, 994, 3564, 81, 172, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[40824, 1080, 170, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[40997, 1084, 3623, 83, 172, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[44681, 1174, 99, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[44805, 1179, 102, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[44932, 1184, 169, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[45104, 1189, 1098, 33, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[46239, 1227, 204, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[46446, 1232, 1334, 37, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[47829, 1275, 163, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[48017, 1281, 221, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[48241, 1286, 758, 17, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[49060, 1310, 100, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[49185, 1315, 106, 1, 27, "Subsubsection"],
Cell[49294, 1318, 549, 11, 83, "Text"],
Cell[49846, 1331, 337, 7, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[50220, 1343, 213, 3, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[50458, 1350, 173, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[50634, 1355, 1363, 29, 192, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[52034, 1389, 198, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10006.}],
Cell[52235, 1394, 718, 20, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10006.}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[53026, 1422, 224, 3, 41, "Section"],
Cell[CellGroupData[{
Cell[53275, 1429, 203, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[53503, 1436, 125, 1, 27, "Subsubsection"],
Cell[53631, 1439, 955, 26, 72, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[54623, 1470, 174, 2, 27, "Subsubsection"],
Cell[54800, 1474, 2040, 37, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[56877, 1516, 105, 1, 27, "Subsubsection"],
Cell[56985, 1519, 803, 19, 92, "Input"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[57837, 1544, 204, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[58066, 1551, 186, 2, 27, "Subsubsection"],
Cell[58255, 1555, 317, 9, 31, "Input"],
Cell[58575, 1566, 1758, 33, 72, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[60370, 1604, 272, 5, 27, "Subsubsection"],
Cell[60645, 1611, 1227, 25, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[61909, 1641, 103, 1, 27, "Subsubsection"],
Cell[62015, 1644, 863, 21, 92, "Input"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[62927, 1671, 234, 3, 36, "Subsection"],
Cell[63164, 1676, 2733, 39, 52, "Input"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[65946, 1721, 160, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[66131, 1727, 207, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[66363, 1734, 106, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[66494, 1739, 121, 2, 31, "Input"],
Cell[66618, 1743, 355, 6, 70, "Print",
 CellTags->"Info3518182010-7910842"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[67022, 1755, 108, 1, 27, "Subsubsection"],
Cell[67133, 1758, 2457, 45, 172, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[69627, 1808, 101, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[69753, 1813, 169, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[69925, 1818, 1302, 31, 152, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[71288, 1856, 157, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[71470, 1862, 104, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[71599, 1867, 119, 2, 31, "Input"],
Cell[71721, 1871, 597, 9, 70, "Print",
 CellTags->"Info3518182010-5560469"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[72367, 1886, 107, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[72499, 1891, 278, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[72780, 1897, 569, 11, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[73386, 1913, 381, 5, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[73770, 1920, 4757, 108, 347, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[78588, 2035, 259, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[78872, 2042, 109, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[79006, 2047, 123, 2, 31, "Input"],
Cell[79132, 2051, 575, 9, 70, "Print",
 CellTags->"Info3518182010-4499236"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[79756, 2066, 107, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[79888, 2071, 182, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[80073, 2076, 1941, 30, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[82075, 2113, 104, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[82204, 2118, 109, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[82338, 2123, 123, 2, 31, "Input"],
Cell[82464, 2127, 374, 6, 70, "Print",
 CellTags->"Info3518182011-4076727"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[82887, 2139, 112, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[83024, 2144, 232, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[83259, 2149, 447, 9, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[83767, 2165, 264, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[84056, 2172, 167, 2, 27, "Subsubsection"],
Cell[84226, 2176, 1848, 32, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[86111, 2213, 115, 1, 27, "Subsubsection"],
Cell[86229, 2216, 795, 18, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[87061, 2239, 109, 1, 27, "Subsubsection"],
Cell[87173, 2242, 2007, 34, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[89217, 2281, 108, 1, 27, "Subsubsection"],
Cell[89328, 2284, 436, 9, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[89801, 2298, 96, 1, 27, "Subsubsection"],
Cell[89900, 2301, 875, 17, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[90824, 2324, 113, 1, 36, "Subsection"],
Cell[90940, 2327, 574, 11, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[91551, 2343, 118, 1, 36, "Subsection"],
Cell[91672, 2346, 281, 7, 52, "Input"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[92002, 2359, 139, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[92166, 2365, 235, 3, 36, "Subsection"],
Cell[92404, 2370, 1414, 22, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[93855, 2397, 138, 2, 36, "Subsection"],
Cell[93996, 2401, 269, 3, 29, "Text"],
Cell[CellGroupData[{
Cell[94290, 2408, 1917, 33, 132, "Input"],
Cell[96210, 2443, 3133, 48, 70, "Output"]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[99404, 2498, 132, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[99561, 2504, 99, 1, 36, "Subsection"],
Cell[99663, 2507, 1611, 34, 72, "Input"],
Cell[101277, 2543, 1594, 34, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[102908, 2582, 155, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[103088, 2588, 197, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[103288, 2593, 1874, 38, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[105165, 2633, 1648, 35, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[106816, 2670, 4305, 99, 292, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[111158, 2774, 204, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[111365, 2779, 1662, 35, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[113030, 2816, 1546, 34, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[114579, 2852, 4251, 98, 292, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[118879, 2956, 153, 2, 36, "Subsection"],
Cell[119035, 2960, 1711, 36, 72, "Input"],
Cell[120749, 2998, 1694, 35, 92, "Input"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[122492, 3039, 94, 1, 41, "Section"],
Cell[CellGroupData[{
Cell[122611, 3044, 168, 2, 36, "Subsection"],
Cell[122782, 3048, 1095, 29, 92, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[123914, 3082, 160, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[124099, 3088, 509, 8, 31, "Input"],
Cell[124611, 3098, 3314, 55, 70, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[127974, 3159, 161, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[128160, 3165, 214, 3, 27, "Subsubsection"],
Cell[128377, 3170, 1782, 43, 112, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[130196, 3218, 115, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[130336, 3223, 2793, 39, 31, "Input"],
Cell[133132, 3264, 15213, 256, 70, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[148418, 3528, 160, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[148603, 3534, 352, 5, 36, "Subsection"],
Cell[CellGroupData[{
Cell[148980, 3543, 2119, 41, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[151102, 3586, 2457, 50, 92, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[153608, 3642, 160, 2, 36, "Subsection"],
Cell[CellGroupData[{
Cell[153793, 3648, 113, 1, 27, "Subsubsection"],
Cell[153909, 3651, 491, 10, 52, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[154437, 3666, 113, 1, 27, "Subsubsection"],
Cell[154553, 3669, 2513, 51, 272, "Input"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[157115, 3726, 260, 3, 36, "Subsection"],
Cell[CellGroupData[{
Cell[157400, 3733, 104, 1, 27, "Subsubsection"],
Cell[157507, 3736, 224, 4, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[157734, 3742, 284, 5, 44, "Print",
 CellTags->"Info3518182014-3702807"],
Cell[158021, 3749, 223, 4, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[158247, 3755, 287, 5, 44, "Print",
 CellTags->"Info3518182014-8479246"],
Cell[158537, 3762, 221, 4, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[158761, 3768, 315, 5, 67, "Print",
 CellTags->"Info3518182014-8742825"]
}, Open  ]],
Cell[CellGroupData[{
Cell[159113, 3778, 98, 1, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[159236, 3783, 172, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[159411, 3787, 1791, 28, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[161205, 3817, 110383, 1882, 390, "Output",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[271625, 5704, 229, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[271857, 5709, 1903, 29, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[273775, 5741, 115638, 1980, 390, "Output"],
Cell[CellGroupData[{
Cell[389438, 7725, 305, 4, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[389746, 7731, 750, 23, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[390499, 7756, 1789, 29, 31, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[392291, 7787, 201003, 3390, 402, "Output",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[593367, 11185, 178, 2, 41, "Section"],
Cell[CellGroupData[{
Cell[593570, 11191, 108, 1, 36, "Subsection"],
Cell[CellGroupData[{
Cell[593703, 11196, 178, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[593906, 11202, 416, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[594325, 11210, 764, 15, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[595126, 11230, 418, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[595547, 11238, 914, 17, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[596498, 11260, 421, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[596922, 11268, 1033, 20, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[598004, 11294, 152, 2, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[598181, 11300, 383, 5, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[598567, 11307, 822, 17, 72, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[599438, 11330, 215, 3, 27, "Subsubsection"],
Cell[CellGroupData[{
Cell[599678, 11337, 329, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[600010, 11345, 358, 8, 52, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[600405, 11358, 314, 6, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}],
Cell[600722, 11366, 2122, 42, 272, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[602881, 11413, 443, 7, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[603327, 11422, 2494, 51, 272, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[605858, 11478, 161, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}],
Cell[606022, 11482, 2297, 50, 312, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[608356, 11537, 161, 2, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}],
Cell[608520, 11541, 910, 18, 112, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10002.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[609467, 11564, 214, 3, 29, "Text",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}],
Cell[609684, 11569, 1136, 29, 132, "Input",
 CellGroupingRules->{GroupTogetherGrouping, 10003.}]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[610881, 11605, 98, 1, 36, "Subsection"],
Cell[610982, 11608, 778, 18, 52, "Input"]
}, Open  ]]
}, Closed]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
