#{{{ ==== Import modules ====

import gtrace.beam as beam
import gtrace.optcomp as opt
from gtrace.draw import drawAllBeams, drawAllOptics, transAll, rotateAll
import sdxf
from unit import *
import numpy as np
pi = np.pi
import scipy.optimize as sopt
#}}}

#{{{ === Length Parameters ===

L_PRM_PR2 = 17.1
L_PR2_PR3 = 14.1
L_PR3_BS = 17.1
L_ASYM = 3.33103
L_BS_ITMX = 25.0 + L_ASYM/2
L_BS_ITMY = 25.0 - L_ASYM/2

L_PRC_X = L_PRM_PR2 + L_PR2_PR3 + L_PR3_BS + L_BS_ITMX
L_PRC_Y = L_PRM_PR2 + L_PR2_PR3 + L_PR3_BS + L_BS_ITMY

L_SRM_SR2 = 17.1
L_SR2_SR3 = 14.1
L_SR3_BS = 17.1

#}}}

#{{{ === Define Mirrors ===

PRM = opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=10*cm, thickness=6*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=1./(337.1*m),
                 inv_ROC_AR=0,
                 Refl_HR=0.9, Trans_HR=1-0.9,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.45, name='PRM')

PR2= opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=10*cm, thickness=6*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=-1./(3.9*m),
                 Refl_HR=1-100*ppm, Trans_HR=100*ppm,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.45, name='PR2')

PR3= opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=38*cm, thickness=12*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=1./(32*m),
                 Refl_HR=1-100*ppm, Trans_HR=100*ppm,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.45, name='PR3')

BS= opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=38*cm, thickness=12*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=0.,
                 Refl_HR=0.5, Trans_HR=0.5,
                 Refl_AR=100*ppm, Trans_AR=1-100*ppm,
                 n=1.45, name='BS')

ITMX= opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=38*cm, thickness=12*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=1./(7098.*m),
                 Refl_HR=1-100*ppm, Trans_HR=100*ppm,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.754, name='ITMX')

ITMY= opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=38*cm, thickness=12*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=1./(7098.*m),
                 Refl_HR=1-100*ppm, Trans_HR=100*ppm,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.754, name='ITMY')

SRM = opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=10*cm, thickness=6*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=1./(337.1*m),
                 inv_ROC_AR=0,
                 Refl_HR=0.9, Trans_HR=1-0.9,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.45, name='SRM')

SR2= opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=10*cm, thickness=6*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=-1./(3.9*m),
                 Refl_HR=1-100*ppm, Trans_HR=100*ppm,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.45, name='SR2')

SR3= opt.Mirror(HRcenter=[0,0], normAngleHR=0.0,
                 diameter=38*cm, thickness=12*cm,
                 wedgeAngle=deg2rad(0.25), inv_ROC_HR=1./(32*m),
                 Refl_HR=1-100*ppm, Trans_HR=100*ppm,
                 Refl_AR=500*ppm, Trans_AR=1-500*ppm,
                 n=1.45, name='SR3')

opticsList = [PRM, PR2, PR3, BS, ITMX, ITMY, SR3, SR2, SRM]

#}}}

#{{{ === Source beam ===

#Get the q-parameter of the beam matching the PRM ROC.
q = beam.Rw2q(ROC=1/PRM.inv_ROC_HR, w=4.29*mm)
b = beam.GaussianBeam(q0=q, pos=[0,0], dirAngle=pi)
beams = PRM.hitFromHR(b)
b1=beams['t1']
b1.propagate(1)
b1.flip()
b1.qx = b1.qy

srcBeam = b1

#}}}

#{{{ Test

#{{{ init 
for x in initValues.keys():
    globals()[x]= initValues[x] 

beamList=[]

#Tweak Mirror Properties
BS.wedgeAngle = BSwedge
ITMX.wedgeAngle = ITMXWedge
ITMY.wedgeAngle = ITMYWedge    
#}}}

#{{{ PRC

#First, prepare a beam which matches the ROC of the PRM and a given waist size
q = beam.Rw2q(ROC=-1/PRM.inv_ROC_HR, w=4.29*mm)
beam_on_PRM = beam.GaussianBeam(q0=q, pos=[0,0], dirAngle=initAngle)
#Set the starting point for measuring the optical distance    
beam_on_PRM.optDist = 0.0

#Position the PRM at [0,0]
PRM.HRcenter = [0,0]
#Slant the PRM by the initAngle
PRM.normAngleHR = initAngle

#Compute the location of PR2
#PR2 is L_PRM_PR2 away from the HRcenter of PRM
PR2.HRcenter = beam_on_PRM.dirVect * L_PRM_PR2 + PRM.HRcenter
PR2.normVectHR = - beam_on_PRM.dirVect
PR2.rotate(foldingAngle)

beams = PR2.hitFromHR(beam_on_PRM, order=0)
beamList.append(beams['input'])
beam_from_PR2 = beams['r1']

#Compute the location of PR3
PR3.HRcenter = beam_from_PR2.dirVect * L_PR2_PR3 + PR2.HRcenter
PR3.normVectHR = - beam_from_PR2.dirVect
PR3.rotate(-foldingAngle)

beams = PR3.hitFromHR(beam_from_PR2, order=0)
beamList.append(beams['input'])
beam_from_PR3 = beams['r1']

#Compute the location of BS
BS.HRcenter = beam_from_PR3.dirVect * L_PR3_BS + PR3.HRcenter
#BS orientation is always at 45 degrees.
BS.normAngleHR = deg2rad(90+45)

beams = BS.hitFromHR(beam_from_PR3, order=0)
beamList.append(beams['input'])
beamList.append(beams['s1'])
beam_s1_BS = beams['s1']
beam_refl_BS = beams['r1']
beam_trans_BS = beams['t1']

#Compute ITMX location
BS_s1_optDist = beam_trans_BS.optDist - beam_s1_BS.optDist
ITMX.HRcenter = beam_trans_BS.dirVect * (L_BS_ITMX - BS_s1_optDist + PRCXTweak) \
                + beam_trans_BS.pos + ITMXOffset
ITMX.normVectHR = [1,0] #ITMX must be looking toward +x


#Hit ITMX
beams = ITMX.hitFromAR(beam_trans_BS, order=0)
beamList.append(beams['input'])
beamList.append(beams['s1'])
beam_ITMX_s1 = beams['s1']
beam_from_ITMX = beams['t1']

#Compute ITMY location
ITMY.HRcenter = beam_refl_BS.dirVect * (L_BS_ITMY + PRCYTweak) \
                + beam_refl_BS.pos + ITMYOffset
ITMY.normVectHR = [0,1] #ITMY must be looking toward +y

#Hit ITMY
beams = ITMY.hitFromAR(beam_refl_BS, order=0)
beamList.append(beams['input'])
beamList.append(beams['s1'])
beam_ITMY_s1 = beams['s1']
beam_from_ITMY = beams['t1']

#}}}

#{{{ SRCX

#Construct SRCX
beam_to_ITMX = beam_from_ITMX
beam_to_ITMX.propagate(1.0)
beam_to_ITMX.flip()
beams = ITMX.hitFromHR(beam_to_ITMX)
#The surface of the ITMX is the start point 
beam_ITMX_AR = beams['t1']
beam_ITMX_AR.optDist = beam_ITMX_AR.optDist - beams['s1'].optDist
beam_ITMX_AR.Gouyx = beam_ITMX_AR.Gouyx - beams['s1'].Gouyx    
beam_ITMX_AR.Gouyy = beam_ITMX_AR.Gouyy - beams['s1'].Gouyy

#Hit the BS
beams = BS.hitFromAR(beam_ITMX_AR, order=1)
beam_BS_s2 = beams['s2']
beamList.append(beam_BS_s2)
beam_BS_to_SR3 = beams['r2']

#SR3
SR3.HRcenter = beam_BS_to_SR3.dirVect * L_SR3_BS + beam_BS_to_SR3.pos
SR3.normVectHR = - beam_BS_to_SR3.dirVect
SR3.rotate(-foldingAngle)
#Hit SR3
beams = SR3.hitFromHR(beam_BS_to_SR3)
beamList.append(beams['input'])
beam_from_SR3 = beams['r1']

#SR2
SR2.HRcenter = beam_from_SR3.dirVect * L_SR2_SR3 + beam_from_SR3.pos
SR2.normVectHR = - beam_from_SR3.dirVect
SR2.rotate(foldingAngle)
#Hit SR2
beams = SR2.hitFromHR(beam_from_SR3)
beamList.append(beams['input'])
beam_from_SR2 = beams['r1']

#SRM
SRM.HRcenter = beam_from_SR2.dirVect * L_SRM_SR2 + beam_from_SR2.pos
SRM.normVectHR = - beam_from_SR2.dirVect

#Hit SRM
beams = SRM.hitFromHR(beam_from_SR2)
beamList.append(beams['input'])
beamList.append(beams['s1'])
beamList.append(beams['t1'])
beam_on_SRMX = beams['s1']

#}}}

#{{{ SRCY

#Construct SRCY
beam_to_ITMY = beam_from_ITMY
beam_to_ITMY.propagate(1.0)
beam_to_ITMY.flip()
beams = ITMY.hitFromHR(beam_to_ITMY)
#The surface of the ITMY is the start point 
beam_ITMY_AR = beams['t1']
beam_ITMY_AR.optDist = beam_ITMY_AR.optDist - beams['s1'].optDist
beam_ITMY_AR.Gouyx = beam_ITMY_AR.Gouyx - beams['s1'].Gouyx    
beam_ITMY_AR.Gouyy = beam_ITMY_AR.Gouyy - beams['s1'].Gouyy

#Hit the BS
beams = BS.hitFromHR(beam_ITMY_AR, order=1)
beamList.append(beams['s1'])
beam_BS_to_SR3 = beams['t1']

#SR3
SR3.HRcenter = beam_BS_to_SR3.dirVect * L_SR3_BS + beam_BS_to_SR3.pos
SR3.normVectHR = - beam_BS_to_SR3.dirVect
SR3.rotate(-foldingAngle)
#Hit SR3
beams = SR3.hitFromHR(beam_BS_to_SR3)
beam_from_SR3 = beams['r1']

#SR2
SR2.HRcenter = beam_from_SR3.dirVect * L_SR2_SR3 + beam_from_SR3.pos
SR2.normVectHR = - beam_from_SR3.dirVect
SR2.rotate(foldingAngle)

#Hit SR2
beams = SR2.hitFromHR(beam_from_SR3)
beam_from_SR2 = beams['r1']

#SRM
SRM.HRcenter = beam_from_SR2.dirVect * L_SRM_SR2 + beam_from_SR2.pos
SRM.normVectHR = - beam_from_SR2.dirVect

#Hit SRM
beams = SRM.hitFromHR(beam_from_SR2)
beam_on_SRMY = beams['s1']
    
#}}}

#}}}

#{{{ === Function to construct beam path ===

def constructLCGTOpticalPath():

#{{{ init

    #Tweak Mirror Properties
    BS.wedgeAngle = BSwedge
    ITMX.wedgeAngle = ITMXWedge
    ITMY.wedgeAngle = ITMYWedge    

#}}}

#{{{ PRC

    #First, prepare a beam which matches the ROC of the PRM and a given waist size
    q = beam.Rw2q(ROC=-1/PRM.inv_ROC_HR, w=4.29*mm)
    beam_on_PRM = beam.GaussianBeam(q0=q, pos=[0,0], dirAngle=initAngle)
    #Set the starting point for measuring the optical distance    
    beam_on_PRM.optDist = 0.0

    #Position the PRM at [0,0]
    PRM.HRcenter = [0,0]
    #Slant the PRM by the initAngle
    PRM.normAngleHR = initAngle

    #Compute the location of PR2
    #PR2 is L_PRM_PR2 away from the HRcenter of PRM
    PR2.HRcenter = beam_on_PRM.dirVect * L_PRM_PR2 + PRM.HRcenter
    PR2.normVectHR = - beam_on_PRM.dirVect
    PR2.rotate(foldingAngle)

    beams = PR2.hitFromHR(beam_on_PRM, order=0)
    beamList.append(beams['input'])
    beam_from_PR2 = beams['r1']

    #Compute the location of PR3
    PR3.HRcenter = beam_from_PR2.dirVect * L_PR2_PR3 + PR2.HRcenter
    PR3.normVectHR = - beam_from_PR2.dirVect
    PR3.rotate(-foldingAngle)

    beams = PR3.hitFromHR(beam_from_PR2, order=0)
    beamList.append(beams['input'])
    beam_from_PR3 = beams['r1']

    #Compute the location of BS
    BS.HRcenter = beam_from_PR3.dirVect * L_PR3_BS + PR3.HRcenter
    #BS orientation is always at 45 degrees.
    BS.normAngleHR = deg2rad(90+45)

    beams = BS.hitFromHR(beam_from_PR3, order=0)
    beamList.append(beams['input'])
    beamList.append(beams['s1'])
    beam_s1_BS = beams['s1']
    beam_refl_BS = beams['r1']
    beam_trans_BS = beams['t1']

    #Compute ITMX location
    BS_s1_optDist = beam_trans_BS.optDist - beam_s1_BS.optDist
    ITMX.HRcenter = beam_trans_BS.dirVect * (L_BS_ITMX - BS_s1_optDist + PRCXTweak) \
                    + beam_trans_BS.pos + ITMXOffset
    ITMX.normVectHR = [1,0] #ITMX must be looking toward +x


    #Hit ITMX
    beams = ITMX.hitFromAR(beam_trans_BS, order=0)
    beamList.append(beams['input'])
    beamList.append(beams['s1'])
    beam_ITMX_s1 = beams['s1']
    beam_from_ITMX = beams['t1']

    #Compute ITMY location
    ITMY.HRcenter = beam_refl_BS.dirVect * (L_BS_ITMY + PRCYTweak) \
                    + beam_refl_BS.pos + ITMYOffset
    ITMY.normVectHR = [0,1] #ITMY must be looking toward +y

    #Hit ITMY
    beams = ITMY.hitFromAR(beam_refl_BS, order=0)
    beamList.append(beams['input'])
    beamList.append(beams['s1'])
    beam_ITMY_s1 = beams['s1']
    beam_from_ITMY = beams['t1']

#}}}

#{{{ SRCX

    #Construct SRCX
    beam_to_ITMX = beam_from_ITMX
    beam_to_ITMX.propagate(1.0)
    beam_to_ITMX.flip()
    beams = ITMX.hitFromHR(beam_to_ITMX)
    #The surface of the ITMX is the start point 
    beam_ITMX_AR = beams['t1']
    beam_ITMX_AR.optDist = beam_ITMX_AR.optDist - beams['s1'].optDist
    beam_ITMX_AR.Gouyx = beam_ITMX_AR.Gouyx - beams['s1'].Gouyx    
    beam_ITMX_AR.Gouyy = beam_ITMX_AR.Gouyy - beams['s1'].Gouyy
    
    #Hit the BS
    beams = BS.hitFromAR(beam_ITMX_AR, order=1)
    beam_BS_s2 = beams['s2']
    beamList.append(beam_BS_s2)
    beam_BS_to_SR3 = beams['r2']
    
    #SR3
    SR3.HRcenter = beam_BS_to_SR3.dirVect * L_SR3_BS + beam_BS_to_SR3.pos
    SR3.normVectHR = - beam_BS_to_SR3.dirVect
    SR3.rotate(-foldingAngle)
    #Hit SR3
    beams = SR3.hitFromHR(beam_BS_to_SR3)
    beamList.append(beams['input'])
    beam_from_SR3 = beams['r1']
    
    #SR2
    SR2.HRcenter = beam_from_SR3.dirVect * L_SR2_SR3 + beam_from_SR3.pos
    SR2.normVectHR = - beam_from_SR3.dirVect
    SR2.rotate(foldingAngle)
    #Hit SR2
    beams = SR2.hitFromHR(beam_from_SR3)
    beamList.append(beams['input'])
    beam_from_SR2 = beams['r1']
    
    #SRM
    SRM.HRcenter = beam_from_SR2.dirVect * L_SRM_SR2 + beam_from_SR2.pos
    SRM.normVectHR = - beam_from_SR2.dirVect
    
    #Hit SRM
    beams = SRM.hitFromHR(beam_from_SR2)
    beamList.append(beams['input'])
    beamList.append(beams['s1'])
    beamList.append(beams['t1'])
    beam_on_SRMX = beams['s1']

#}}}

#{{{ SRCY

    #Construct SRCY
    beam_to_ITMY = beam_from_ITMY
    beam_to_ITMY.propagate(1.0)
    beam_to_ITMY.flip()
    beams = ITMY.hitFromHR(beam_to_ITMY)
    #The surface of the ITMY is the start point 
    beam_ITMY_AR = beams['t1']
    beam_ITMY_AR.optDist = beam_ITMY_AR.optDist - beams['s1'].optDist
    beam_ITMY_AR.Gouyx = beam_ITMY_AR.Gouyx - beams['s1'].Gouyx    
    beam_ITMY_AR.Gouyy = beam_ITMY_AR.Gouyy - beams['s1'].Gouyy
    
    #Hit the BS
    beams = BS.hitFromHR(beam_ITMY_AR, order=1)
    beamList.append(beams['s1'])
    beam_BS_to_SR3 = beams['t1']
    
    #SR3
    SR3.HRcenter = beam_BS_to_SR3.dirVect * L_SR3_BS + beam_BS_to_SR3.pos
    SR3.normVectHR = - beam_BS_to_SR3.dirVect
    SR3.rotate(-foldingAngle)
    #Hit SR3
    beams = SR3.hitFromHR(beam_BS_to_SR3)
    beam_from_SR3 = beams['r1']
    
    #SR2
    SR2.HRcenter = beam_from_SR3.dirVect * L_SR2_SR3 + beam_from_SR3.pos
    SR2.normVectHR = - beam_from_SR3.dirVect
    SR2.rotate(foldingAngle)
    
    #Hit SR2
    beams = SR2.hitFromHR(beam_from_SR3)
    beam_from_SR2 = beams['r1']

    #SRM
    SRM.HRcenter = beam_from_SR2.dirVect * L_SRM_SR2 + beam_from_SR2.pos
    SRM.normVectHR = - beam_from_SR2.dirVect

    #Hit SRM
    beams = SRM.hitFromHR(beam_from_SR2)
    beam_on_SRMY = beams['s1']

    
#}}}

#{{{ Extract parameters

    #Critical parameters
    results = {'ITMXangle': rad2deg(beam_ITMX_s1.dirAngle),
               'ITMYangle': rad2deg(beam_ITMY_s1.dirAngle),
               'ITMXbeamCentering': (ITMX.HRcenter - beam_from_ITMX.pos)[1],
               'ITMYbeamCentering': (ITMY.HRcenter - beam_from_ITMY.pos)[0],
               'LPRCX': beam_from_ITMX.optDist,
               'LPRCY': beam_from_ITMY.optDist,
               'LSRCX': beam_on_SRMX.optDist,
               'LSRCY': beam_on_SRMY.optDist}

    return results

#}}}

#}}}

#{{{ === Optimize ===

#{{{ Initialize

beamList = []

initValues={'initAngle': deg2rad(0.18), 'foldingAngle': deg2rad(.7),
            'BSwedge': deg2rad(0.25), 'ITMXWedge': deg2rad(0.25),
            'ITMYWedge': deg2rad(0.25), 'ITMXOffset': [0.0,0.0],
            'ITMYOffset': [0.0,0.0], 'PRCXTweak': 0.0, 'PRCYTweak': 0.0}

#}}}

#{{{ First make the ITMY incident angle right
for x in initValues.keys():
    globals()[x]= initValues[x] 

def testfun(x):
    beamList = []
    results = constructLCGTOpticalPath()
    return np.abs(results['ITMYangle'] -90.0)

ans = sopt.fmin(testfun, initValues['initAngle'], full_output=True)
if ans[1] < 0.001:
    initValues['initAngle'] = float(ans[0])

#}}}

#{{{ Now tweak the BS wedge angle to get the ITMX straight
for x in initValues.keys():
    globals()[x]= initValues[x] 

def testfun(x):
    beamList = []
    results = constructLCGTOpticalPath()
    return np.abs(results['ITMXangle'])

ans = sopt.fmin(testfun, initValues['BSwedge'], full_output=True)
if ans[1] < 0.001:
    initValues['BSwedge'] = float(ans[0])

#}}}

#{{{ Move ITMX and ITMY to center the beam
for x in initValues.keys():
    globals()[x]= initValues[x] 

results = constructLCGTOpticalPath()

initValues['ITMXOffset'] = [0.0, -results['ITMXbeamCentering']]
initValues['ITMYOffset'] = [-results['ITMYbeamCentering'], 0.0]

#}}}

#{{{ Move ITMX and ITMY to make the PRC length right
for x in initValues.keys():
    globals()[x]= initValues[x] 

results = constructLCGTOpticalPath()

initValues['PRCXTweak'] = L_PRC_X  - results['LPRCX']
initValues['PRCYTweak'] = L_PRC_Y  - results['LPRCY']

#}}}

#{{{ Check the results

for x in initValues.keys():
    globals()[x]= initValues[x] 

results = constructLCGTOpticalPath()

#}}}

#}}}

#{{{ === Draw ===

#{{{ === DXF object ===

d = sdxf.Drawing()
d.layers.append(sdxf.Layer("main_beam", color=1))
d.layers.append(sdxf.Layer("main_beam_width", color=2))
d.layers.append(sdxf.Layer("aux_beam", color=3))
d.layers.append(sdxf.Layer("aux_beam_width", color=4))

#}}}

#{{{ === Draw Beams ===

drawAllBeams(d, beamList, drawWidth=True, drawPower=False,
                 drawROC=False, drawGouy=False, drawOptDist=False,
                    fontSize=0.01)

#}}}

#{{{ === Draw Optics ===

drawAllOptics(d, opticsList, drawName=True)

#}}}

#{{{ Save DXF file 

d.saveas('/home/aso/LCGT_OptLayout.dxf')

#}}}

#}}}

