#{{{ Import modules
import numpy as np
pi = np.pi
array = np.array
sqrt = np.lib.scimath.sqrt
from numpy.linalg import norm

from enthought.traits.api import HasTraits, Int, Float, CFloat, CArray, List, Str
from enthought.traits.ui.api import View, Group, HGroup, VGroup, Item, Readonly, spring,\
     Label, Handler
from enthought.traits.ui.menu import Action, MenuBar, Menu, CloseAction
from enthought.enable.component_editor import ComponentEditor
from enthought.chaco.api import marker_trait, Plot, ArrayPlotData, Legend
from enthought.chaco.tools.api import LegendTool, PanTool, ZoomTool, RectZoomTool
from enthought.kiva.fonttools.font import Font

import optics.geometric
from unit import *
import copy
import sdxf
#}}}

#{{{ Generic Optics Class

class Optics(HasTraits):
    '''
    A general optics class from which other specific
    optics classes are derived.
    '''
    name = Str()
    center = CArray(dtype=np.float64, shape=(2,))
    rotationAngle = CFloat(0.0) #in rad

#}}}

#{{{ Mirror Class

class Mirror(Optics):
    '''
    Representing a partial reflective mirror.

    Traits:

    HRcenter: CArray(dtype=np.float64, shape=(2,))
            The coordinates of the HRcenter of the HR surface.
            
    '''

#{{{ Traits definitions

    HRcenter = CArray(dtype=np.float64, shape=(2,))    
    normVectHR = CArray(dtype=np.float64, shape=(2,))
    normAngleHR = CFloat()
    
    ARcenter = CArray(dtype=np.float64, shape=(2,))
    normVectAR = CArray(dtype=np.float64, shape=(2,))
    normAngleAR = CFloat()
    
    diameter = CFloat(25.0*cm) #
    ARdiameter = CFloat()
    thickness = CFloat(15.0*cm) # 
    wedgeAngle = CFloat(0.25*pi/180) # in rad
    n = CFloat(1.45) #Index of refraction

    inv_ROC_HR = CFloat(1.0/7000.0) #Inverse of the ROC of the HR surface.
    inv_ROC_AR = CFloat(0.0) #Inverse of the ROC of the AR surface.

    Refl_HR = CFloat(99.0) #Power reflectivity of the HR side.
    Trans_HR = CFloat(1.0) #Power transmittance of the HR side.

    Refl_AR = CFloat(0.01) #Power reflectivity of the AR side.
    Trans_AR = CFloat(99.99) #Power transmittance of the HR side.


#}}}

#{{{ __init__

    def __init__(self, HRcenter=[0.0,0.0], normAngleHR=0.0,
                 normVectHR=False, diameter=25.0*cm, thickness=15.0*cm,
                 wedgeAngle=0.25*pi/180., inv_ROC_HR=1.0/7000.0, inv_ROC_AR=0.0,
                 Refl_HR=99.0, Trans_HR=1.0, Refl_AR=0.01, Trans_AR=99.99, n=1.45,
                 name="Mirror"):

        self.HRcenter = HRcenter
        self._HRcenter_changed(0,0)
        
        #Convert rotationAngle to normVectHR or vice versa.
        if normVectHR:
            self.normVectHR = normVectHR
        else:
            self.normAngleHR = normAngleHR

        self.diameter = diameter
        self.thickness = thickness
        self.wedgeAngle = wedgeAngle
        self.ARdiameter = self.diameter/np.cos(self.wedgeAngle)
        self.inv_ROC_HR = inv_ROC_HR
        self.inv_ROC_AR = inv_ROC_AR
        self.Refl_HR = Refl_HR
        self.Trans_HR = Trans_HR
        self.Refl_AR = Refl_AR
        self.Trans_AR = Trans_AR
        self.n = n
        self._normAngleHR_changed(0,0)
        self.name = name
        

#}}}

#{{{ rotate

    def rotate(self, angle, center=False):
        '''
        Rotate the mirror 
        '''

        if center:
            center = np.array(center)
            pointer = self.HRcenter - center
            pointer = optics.geometric.vector_rotation_2D(pointer, angle)
            self.HRcenter = center + pointer
            
        self.normAngleHR = self.normAngleHR + angle

    
#}}}

#{{{ Translate

    def translate(self, trVect):
        trVect = np.array(trVect)
        self.center = self.center + trVect
        
#}}}

#{{{ Draw

    def draw(self, dxf, drawName=False):
        '''
        Draw itself
        '''
        
        plVect = optics.geometric.vector_rotation_2D(self.normVectHR, pi/2)
        p1 = self.HRcenter + plVect * self.diameter/2
        p2 = p1 - plVect * self.diameter
        p3 = p2 - self.normVectHR * (self.thickness - np.tan(self.wedgeAngle)*self.diameter/2)
        p4 = p1 - self.normVectHR * (self.thickness + np.tan(self.wedgeAngle)*self.diameter/2) 

        dxf.append(sdxf.Line(points=[p2,p3], layer="Mirrors"))
        dxf.append(sdxf.Line(points=[p4,p1], layer="Mirrors"))
        
        d = self.thickness/10
        l1 = p1 - self.normVectHR * d
        l2 = p2 - self.normVectHR * d
        dxf.append(sdxf.Line(points=[tuple(l1), tuple(l2)],
                       layer="Mirrors"))

        #Draw Curved surface

        #HR
        if np.abs(self.inv_ROC_HR) > 1.0/1e5:
            R = 1/self.inv_ROC_HR
            theta = np.arcsin(self.diameter/2/R)
            sag = R*(1-np.cos(theta))
            x = np.linspace(0, self.diameter/2, 30)
            y = R*(1.0 - np.sqrt(1.0 - x**2/(R**2))) -sag
            x2 = -np.flipud(x)
            y2 = np.flipud(y)
            x = np.hstack((x2,x))
            y = np.hstack((y2,y))
            v = np.vstack((x,y))
            v = optics.geometric.vector_rotation_2D(v, self.normAngleHR - pi/2)
            v = v.T + self.HRcenter
            dxf.append(sdxf.LwPolyLine(points=list(v), layer="Mirrors"))
        else:
            dxf.append(sdxf.Line(points=[p1,p2], layer="Mirrors"))

        #AR
        if np.abs(self.inv_ROC_AR) > 1.0/1e5:
            diameter = self.diameter/np.cos(self.wedgeAngle)
            
            R = 1/self.inv_ROC_AR
            theta = np.arcsin(diameter/2/R)
            sag = R*(1-np.cos(theta))
            x = np.linspace(0, diameter/2, 30)
            y = R*(1.0 - np.sqrt(1.0 - x**2/(R**2))) -sag
            x2 = -np.flipud(x)
            y2 = np.flipud(y)
            x = np.hstack((x2,x))
            y = np.hstack((y2,y))
            v = np.vstack((x,y))
            v = optics.geometric.vector_rotation_2D(v, self.normAngleAR - pi/2)
            v = v.T + self.ARcenter
            dxf.append(sdxf.LwPolyLine(points=list(v), layer="Mirrors"))
        else:
            dxf.append(sdxf.Line(points=[p3,p4], layer="Mirrors"))

            
        if drawName:
            center = (p1+p2+p3+p4)/4.
            height = self.thickness/4.
            width = height*len(self.name)
            center = center - np.array([width/2, height/2])
            
            dxf.append(sdxf.Text(text=self.name, point=center,
                                 height=height, layer='text'))
            

#}}}

#{{{ hitFromHR

    def hitFromHR(self, beam, order=0):
        '''
        Compute the reflected and deflected beams when
        an input beam hit the HR surface.
        '''

        #A dictionary to hold beams
        beams={}
        
        #Get the intersection point
        ans = optics.geometric.line_arc_intersection(pos=beam.pos, dirVect=beam.dirVect,
                                                     chord_center=self.HRcenter,
                                                     chordNormVect=self.normVectHR,
                                                     invROC=self.inv_ROC_HR,
                                                     diameter=self.diameter)
        if not ans['isHit']:
            #The input beam does not hit the mirror.
            print('The beam does not hit the mirror')
            return beams

        beam_in = beam.copy() #Make a copy
        beam_in.length = ans['distance']
        beams['input']= beam_in

        #Local normal angle
        localNormAngle = ans['localNormAngle']
        
        #Propagate the input beam to the intersection point
        beam_on_HR = beam_in.copy()
        beam_on_HR.propagate(ans['distance'])

        #Calculate reflection and deflection angles along with the ABCD matrices
        #for reflection and deflection.
        (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                    optics.geometric.refl_defl_angle(beam_on_HR.dirAngle,
                                               localNormAngle,
                                               1.0, self.n, invROC=self.inv_ROC_HR)        
        #Reflected beam
        beam_r1 = beam_on_HR.copy()
        beam_r1.dirAngle = reflAngle
        beam_r1.P = beam_r1.P * self.Refl_HR
        beam_r1.ABCDTrans(Mrx, Mry)
        beams['r1'] = beam_r1
        
        #Transmitted beam
        beam_s1 = beam_on_HR.copy()
        beam_s1.dirAngle = deflAngle
        beam_s1.n = self.n
        beam_s1.ABCDTrans(Mtx, Mty)
        beam_s1.P = beam_s1.P * self.Trans_HR

        #Hit AR from back
        ans = optics.geometric.line_arc_intersection(pos=beam_s1.pos, dirVect=beam_s1.dirVect,
                                                     chord_center=self.ARcenter,
                                                     chordNormVect=-self.normVectAR,
                                                     invROC=-self.inv_ROC_AR,
                                                     diameter=self.ARdiameter)

        beam_s1.length = ans['distance']
        beams['s1'] = beam_s1

        #Local normal angle
        localNormAngle = ans['localNormAngle']
        
        #Propagate the beam to the AR surface
        beam_on_AR = beam_s1.copy()
        beam_on_AR.propagate(ans['distance'])

        (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                    optics.geometric.refl_defl_angle(beam_on_AR.dirAngle,
                                               localNormAngle,
                                               self.n, 1.0, invROC=-self.inv_ROC_AR)
                                               
        #Reflected beam
        beam_sr = beam_on_AR.copy()
        beam_sr.dirAngle = reflAngle
        beam_sr.P = beam_sr.P * self.Refl_AR
        beam_sr.ABCDTrans(Mrx, Mry)
        
        #Transmitted beam
        beam_t1 = beam_on_AR.copy()
        beam_t1.dirAngle = deflAngle
        beam_t1.n = 1.0
        beam_t1.P = beam_on_AR.P * self.Trans_AR
        beam_t1.ABCDTrans(Mtx, Mty)        
        beams['t1'] = beam_t1

        #Calculate higher order reflections

        ii = 1
        while ii <= order:

            #Hit the HR from the back
            
            #Get the intersection point
            ans = optics.geometric.line_arc_intersection(pos=beam_sr.pos, dirVect=beam_sr.dirVect,
                                                         chord_center=self.HRcenter,
                                                         chordNormVect=-self.normVectHR,
                                                         invROC=-self.inv_ROC_HR,
                                                         diameter=self.diameter)

            if not ans['isHit']:
                break

            beam_sr.length = ans['distance']
            beam_sr.layer = 'aux_beam'
            beams['s'+str(2*ii)]= beam_sr

            #Local normal angle
            localNormAngle = ans['localNormAngle']            

            #Propagate the input beam to the intersection point
            beam_on_HR = beam_sr.copy()
            beam_on_HR.propagate(ans['distance'])

            #Calculate reflection and deflection angles along with the ABCD matrices
            #for reflection and deflection.
            (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                        optics.geometric.refl_defl_angle(beam_on_HR.dirAngle,
                                                         localNormAngle,
                                                         self.n, 1.0, invROC=-self.inv_ROC_HR)

            #Reflected by HR
            beam_s1 = beam_on_HR.copy()
            beam_s1.dirAngle = reflAngle
            beam_s1.P = beam_s1.P * self.Refl_HR
            beam_s1.ABCDTrans(Mrx, Mry)

            #Transmitted through HR
            beam_r1 = beam_on_HR.copy()
            beam_r1.dirAngle = deflAngle
            beam_r1.n = 1.0
            beam_r1.ABCDTrans(Mtx, Mty)
            beam_r1.P = beam_r1.P * self.Trans_HR
            beams['r'+str(ii+1)] = beam_r1


            #Hit AR from back
            ans = optics.geometric.line_arc_intersection(pos=beam_s1.pos, dirVect=beam_s1.dirVect,
                                                         chord_center=self.ARcenter,
                                                         chordNormVect=-self.normVectAR,
                                                         invROC=-self.inv_ROC_AR,
                                                         diameter=self.ARdiameter)
            
            if not ans['isHit']:
                break
            
            beam_s1.length = ans['distance']
            beams['s'+str(2*ii+1)] = beam_s1

            #Local normal angle
            localNormAngle = ans['localNormAngle']

            #Propagate the beam to the AR surface
            beam_on_AR = beam_s1.copy()
            beam_on_AR.propagate(ans['distance'])

            (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                        optics.geometric.refl_defl_angle(beam_on_AR.dirAngle,
                                                         localNormAngle,
                                                         self.n, 1.0, invROC=-self.inv_ROC_AR)
            #Reflected beam
            beam_sr = beam_on_AR.copy()
            beam_sr.dirAngle = reflAngle
            beam_sr.P = beam_sr.P * self.Refl_AR
            beam_sr.ABCDTrans(Mrx, Mry)

            #Transmitted beam
            beam_t1 = beam_on_AR.copy()
            beam_t1.dirAngle = deflAngle
            beam_t1.n = 1.0
            beam_t1.P = beam_on_AR.P * self.Trans_AR
            beam_t1.ABCDTrans(Mtx, Mty)        
            beams['t'+str(ii+1)] = beam_t1

            ii=ii+1


        return beams
#}}}

#{{{ hitFromAR

    def hitFromAR(self, beam, order=0):
        '''
        Compute the reflected and deflected beams when
        an input beam hit the AR surface.
        '''

        #A dictionary to hold beams
        beams={}
        
        #Get the intersection point
        ans = optics.geometric.line_arc_intersection(pos=beam.pos, dirVect=beam.dirVect,
                                                     chord_center=self.ARcenter,
                                                     chordNormVect=self.normVectAR,
                                                     invROC=self.inv_ROC_AR,
                                                     diameter=self.ARdiameter)

        if not ans['isHit']:
            #The input beam does not hit the mirror.
            print('The beam does not hit the mirror')
            return beams

        beam_in = beam.copy() #Make a copy
        beam_in.length = ans['distance']
        beams['input']= beam_in

        #Local normal angle
        localNormAngle = ans['localNormAngle']        
        
        #Propagate the input beam to the intersection point
        beam_on_AR = beam_in.copy()
        beam_on_AR.propagate(ans['distance'])

        #Calculate reflection and deflection angles along with the ABCD matrices
        #for reflection and deflection.
        (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                    optics.geometric.refl_defl_angle(beam_on_AR.dirAngle,
                                               localNormAngle,
                                               1.0, self.n, invROC=self.inv_ROC_AR)        
        #Reflected beam
        beam_r1 = beam_on_AR.copy()
        beam_r1.dirAngle = reflAngle
        beam_r1.P = beam_r1.P * self.Refl_AR
        beam_r1.ABCDTrans(Mrx, Mry)
        beams['r1'] = beam_r1
        
        #Transmitted beam
        beam_s1 = beam_on_AR.copy()
        beam_s1.dirAngle = deflAngle
        beam_s1.n = self.n
        beam_s1.ABCDTrans(Mtx, Mty)
        beam_s1.P = beam_s1.P * self.Trans_AR

        #Hit HR from back
        ans = optics.geometric.line_arc_intersection(pos=beam_s1.pos, dirVect=beam_s1.dirVect,
                                                     chord_center=self.HRcenter,
                                                     chordNormVect=-self.normVectHR,
                                                     invROC=-self.inv_ROC_HR,
                                                     diameter=self.diameter)
        
        beam_s1.length = ans['distance']
        beams['s1'] = beam_s1

        #Local normal angle
        localNormAngle = ans['localNormAngle']                
        
        #Propagate the beam to the HR surface
        beam_on_HR = beam_s1.copy()
        beam_on_HR.propagate(ans['distance'])

        (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                    optics.geometric.refl_defl_angle(beam_on_HR.dirAngle,
                                               localNormAngle,
                                               self.n, 1.0, invROC=-self.inv_ROC_HR)
                                               
        #Reflected beam
        beam_sr = beam_on_HR.copy()
        beam_sr.dirAngle = reflAngle
        beam_sr.P = beam_sr.P * self.Refl_HR
        beam_sr.ABCDTrans(Mrx, Mry)
        
        #Transmitted beam
        beam_t1 = beam_on_HR.copy()
        beam_t1.dirAngle = deflAngle
        beam_t1.n = 1.0
        beam_t1.P = beam_on_HR.P * self.Trans_HR
        beam_t1.ABCDTrans(Mtx, Mty)        
        beams['t1'] = beam_t1

        #Calculate higher order reflections

        ii = 1
        while ii <= order:

            #Hit AR from back
            
            #Get the intersection point
            ans = optics.geometric.line_arc_intersection(pos=beam_sr.pos, dirVect=beam_sr.dirVect,
                                                         chord_center=self.ARcenter,
                                                         chordNormVect=-self.normVectAR,
                                                         invROC=-self.inv_ROC_AR,
                                                         diameter=self.ARdiameter)
            
            if not ans['isHit']:
                break

            beam_sr.length = ans['distance']
            beam_sr.layer = 'aux_beam'
            beams['s'+str(2*ii)]= beam_sr

            #Local normal angle
            localNormAngle = ans['localNormAngle']            

            #Propagate the input beam to the intersection point
            beam_on_AR = beam_sr.copy()
            beam_on_AR.propagate(ans['distance'])

            #Calculate reflection and deflection angles along with the ABCD matrices
            #for reflection and deflection.
            (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                        optics.geometric.refl_defl_angle(beam_on_AR.dirAngle,
                                                         localNormAngle,
                                                         self.n, 1.0, invROC=-self.inv_ROC_AR)

            #Reflected by AR
            beam_s1 = beam_on_AR.copy()
            beam_s1.dirAngle = reflAngle
            beam_s1.P = beam_s1.P * self.Refl_AR
            beam_s1.ABCDTrans(Mrx, Mry)

            #Transmitted through AR
            beam_r1 = beam_on_AR.copy()
            beam_r1.dirAngle = deflAngle
            beam_r1.n = 1.0
            beam_r1.ABCDTrans(Mtx, Mty)
            beam_r1.P = beam_r1.P * self.Trans_AR
            beams['r'+str(ii+1)] = beam_r1


            #Hit HR from back
            ans = optics.geometric.line_arc_intersection(pos=beam_s1.pos, dirVect=beam_s1.dirVect,
                                                         chord_center=self.HRcenter,
                                                         chordNormVect=-self.normVectHR,
                                                         invROC=-self.inv_ROC_HR,
                                                         diameter=self.diameter)
            
            if not ans['isHit']:
                break
            
            beam_s1.length = ans['distance']
            beams['s'+str(2*ii+1)] = beam_s1

            #Local normal angle
            localNormAngle = ans['localNormAngle']            

            #Propagate the beam to the HR surface
            beam_on_HR = beam_s1.copy()
            beam_on_HR.propagate(ans['distance'])

            (reflAngle, deflAngle, Mrx, Mry, Mtx, Mty) = \
                        optics.geometric.refl_defl_angle(beam_on_HR.dirAngle,
                                                         localNormAngle,
                                                         self.n, 1.0, invROC=-self.inv_ROC_HR)
            #Reflected beam
            beam_sr = beam_on_HR.copy()
            beam_sr.dirAngle = reflAngle
            beam_sr.P = beam_sr.P * self.Refl_HR
            beam_sr.ABCDTrans(Mrx, Mry)

            #Transmitted beam
            beam_t1 = beam_on_HR.copy()
            beam_t1.dirAngle = deflAngle
            beam_t1.n = 1.0
            beam_t1.P = beam_t1.P * self.Trans_HR
            beam_t1.ABCDTrans(Mtx, Mty)        
            beams['t'+str(ii+1)] = beam_t1

            ii=ii+1


        return beams

#}}}

#{{{ Notification handlers

    def _normAngleHR_changed(self, old, new):
        self.set(trait_change_notify=False,
                 normVectHR = array([np.cos(self.normAngleHR), np.sin(self.normAngleHR)]))
        self.set(trait_change_notify=False,        
                 normAngleHR = np.mod(self.normAngleHR, 2*pi))
        
        self.normVectAR = optics.geometric.vector_rotation_2D(self.normVectHR, pi+self.wedgeAngle)
        self.ARcenter = self.HRcenter - self.normVectHR * self.thickness
        self.normAngleAR = np.mod(self.normAngleHR + pi + self.wedgeAngle, 2*pi)
            
    def _normVectHR_changed(self, old, new):
        #Normalize
        self.set(trait_change_notify=False,        
                 normVectHR = self.normVectHR/np.linalg.norm(array(self.normVectHR)))
        #Update dirAngle accordingly
        self.set(trait_change_notify=False,        
                 normAngleHR = np.mod(np.arctan2(self.normVectHR[1],
                                                   self.normVectHR[0]), 2*pi))

        self.normVectAR = optics.geometric.vector_rotation_2D(self.normVectHR, pi+self.wedgeAngle)
        self.ARcenter = self.HRcenter - self.normVectHR * self.thickness
        self.normAngleAR = np.mod(self.normAngleHR + pi + self.wedgeAngle, 2*pi)

    def _HRcenter_changed(self, old, new):
        self.ARcenter = self.HRcenter - self.normVectHR * self.thickness
        self.set(trait_change_notify=False,
                 center = (self.HRcenter + self.ARcenter)/2.0)

    def _center_changed(self, old, new):
        self.set(trait_change_notify=False,
                 HRcenter = self.center + self.normVectHR * self.thickness/2.0)
        self.ARcenter = self.HRcenter - self.normVectHR * self.thickness

    def _wedgeAngle_changed(self, old, new):
        self.normAngleAR = np.mod(self.normAngleHR + pi + self.wedgeAngle, 2*pi)
        self.normVectAR = optics.geometric.vector_rotation_2D(self.normVectHR, pi+self.wedgeAngle)
    
#}}}        

#}}}




















