"""
Cavity
"""

#{{{ Import modules
import numpy as np
pi = np.pi
c = 299792458.0
import optics.gaussian as gauss
from enthought.traits.api import CFloat, HasTraits
#}}}


#{{{ Class Cavity

class Cavity(HasTraits):
    """
    A class to represent a Fabry-Perot cavity.

    Attributes:
    r1: Input mirror reflectivity (amplitude)
    r2: End mirror reflectivity (amplitude)
    rp1: Input mirror reflectivity (power)
    rp2: End mirror reflectivity (power)
    L: Length
    R1: ROC of the input mirror (positive when concave to incident light)
    R2: ROC of the end mirror (positive when concave to incident light)
    wl: Wavelength
    """
    r1 = CFloat(0.9)
    r2 = CFloat(0.99)
    rp1 = CFloat()
    rp2 = CFloat()
    tp1 = CFloat()
    tp2 = CFloat()
    t1 = CFloat()
    t2 = CFloat()

    L = CFloat(1.0)
    R1 = CFloat(-1.5)
    R2 = CFloat(1.5)
    g1 = CFloat()
    g2 = CFloat()    
    
    def __init__(self, r1=0.9, r2=0.99, L=1.0, R1=-1.5, R2=1.5, wl=1064e-9, power=False):
        """
        """
        if power:
            self.r1 = np.sqrt(r1)
            self.r2 = np.sqrt(r2)
        else:
            self.r1 = r1
            self.r2 = r2

        self.L = L
        self.R1 = R1
        self.R2 = R2
        self.wl = wl

    #{{{ Trait change handlers
    def _R1_changed(self, old, new):
        self.g1 = 1 + self.L/new

    def _R2_changed(self, old, new):
        self.g2 = 1 - self.L/new

    def _L_changed(self, old, new):
        self.g1 = 1 + new/self.R1
        self.g2 = 1 - new/self.R2

    def _r1_changed(self, old, r1):
        self.rp1 = r1**2
        self.tp1 = 1 - self.rp1
        self.t1 = np.sqrt(self.tp1)

    def _r2_changed(self, old, r2):
        self.rp2 = r2**2
        self.tp2 = 1 - self.rp2
        self.t2 = np.sqrt(self.tp2)

    #}}}

    def finesse(self):
        return finesse(self.r1, self.r2)

    def FSR(self):
        return c/(2*self.L)

    def modeSpacing(self):
        return gauss.modeSpacing(self.g1, self.g2)

    def waist(self, size=False):
        """
        Return the q parameter or the radius of the beam at the cavity waist.
        
        """
        q0 = 1j*np.sqrt(self.L)*np.sqrt(-(self.L+self.R1)*(self.L-self.R2)\
                                        *(self.L+self.R1-self.R2))\
                                        /(2*self.L+self.R1-self.R2)
        d = self.L*(self.L-self.R2)/(2*self.L+self.R1-self.R2)

        if size:
            return (gauss.qToRadius(q0, self.wl), d)
        else:
            return (q0, d)
        
    def trans(self, f=None, d=0):
        """
        """
        if not f is None:
            return self.t1 * self.t2 * np.exp(1j*self.L*2*pi*f/c)/\
                   (1-self.r1*self.r2*np.exp(2j*self.L*2*pi*f/c))
        else:
            return self.t1 * self.t2 * np.exp(2*pi*1j*d/self.wl)/\
                   (1-self.r1*self.r2*np.exp(4*pi*1j*d/self.wl))

    def refl(self, f=None, d=0):
        """
        """
        if not f is None:
            return -self.r1+self.t1**2 * self.r2 * np.exp(2j*self.L*2*pi*f/c)/\
                   (1-self.r1*self.r2*np.exp(2j*self.L*2*pi*f/c))
        else:
            return -self.r1+self.t1**2 * self.r2 * np.exp(4*pi*1j*d/self.wl)/\
                   (1-self.r1*self.r2*np.exp(4*pi*1j*d/self.wl))

    def intra(self, f=None, d=0):
        """
        """
        if not f is None:
            return self.t1/(1-self.r1*self.r2*np.exp(2j*self.L*2*pi*f/c))
        else:
            return  self.t1/(1-self.r1*self.r2*np.exp(4*pi*1j*d/self.wl))
#}}}

#{{{ Often used parameters

def finesse(r1, r2, power=False):
    if power:
        r1 = np.sqrt(r1)
        r2 = np.sqrt(r2)
        
    return pi*np.sqrt(r1*r2)/(1-r1*r2)

#}}}

